#include "series.h"

using namespace NZoom::NContainers;

namespace {
    size_t GetPoints(TInstant startTime, TInstant endTime, TDuration resolution) {
        if (endTime < startTime) {
            ythrow yexception() << "end time less than start time";
        }
        return (endTime + resolution - startTime).GetValue() / resolution.GetValue();
    }

    size_t GetOffset(TInstant seriesTime, TInstant startTime, TDuration resolution) {
        if (seriesTime < startTime) {
            ythrow yexception() << "series time less than start time";
        }
        return (seriesTime - startTime).GetValue() / resolution.GetValue();
    }

    size_t GetValueCount(size_t startOffset, size_t endOffset, size_t accumulatorCount) {
        if (!accumulatorCount) {
            return 0;
        }
        endOffset = Min(endOffset, accumulatorCount - 1);
        if (endOffset >= startOffset) {
            return endOffset + 1 - startOffset;
        } else {
            return 0;
        }
    }
}

bool TOverwriteAnythingPolicy::SkipOperation(TInstant, TInstant, TInstant, TInstant, TDuration) const {
    return false;
}

bool TOverwriteAnythingPolicy::SkipPoint(TInstant) const {
    return false;
}

bool TOverwriteContinuousPolicy::SkipOperation(TInstant leftStartTime, TInstant leftEndTime,
                                               TInstant rightStartTime, TInstant, TDuration resolution) const {
    if (!leftStartTime && !leftEndTime) {
        return false;
    }
    return leftEndTime + resolution < rightStartTime;
}

bool TOverwriteContinuousPolicy::SkipPoint(TInstant) const {
    return false;
}

TOverwriteWithSkipPolicy::TOverwriteWithSkipPolicy(TInstant since, TInstant until)
    : Since(since)
    , Until(until)
{
}

bool TOverwriteWithSkipPolicy::SkipOperation(TInstant, TInstant, TInstant, TInstant, TDuration) const {
    return false;
}

bool TOverwriteWithSkipPolicy::SkipPoint(TInstant pointTime) const {
    return pointTime < Since || pointTime > Until;
}

TTimestampedSeriesContainer::TOverwritingVisitor::TOverwritingVisitor(TTimestampedSeriesContainer* container,
                                                                      const IOverwritePolicy& policy)
    : Container(container)
    , Policy(policy)
{
}

void TTimestampedSeriesContainer::TOverwritingVisitor::OnStartTime(TInstant startTime) {
    CurrentTime = startTime;
}

void TTimestampedSeriesContainer::TOverwritingVisitor::OnValueCount(size_t) {
}

void TTimestampedSeriesContainer::TOverwritingVisitor::OnValue(NZoom::NValue::TValueRef value) {
    if (!Policy.SkipPoint(CurrentTime)) {
        size_t offset = GetOffset(CurrentTime, Container->StartTime, Container->Resolution);
        Container->Accumulators->Set(value, offset);
    }
    CurrentTime += Container->Resolution;
}

TTimestampedSeriesContainer::TTimestampedSeriesContainer(TMaybe<NZoom::NAccumulators::EAccumulatorType> accumulator,
                                                         TInstant startTime, TInstant endTime, TDuration resolution, bool merge)
    : StartTime(startTime)
    , Resolution(resolution)
    , Merge(merge)
{
    if (accumulator.Defined()) {
        Accumulators.ConstructInPlace(accumulator.GetRef(), GetPoints(startTime, endTime, resolution));
    }
}

void TTimestampedSeriesContainer::Mul(const NZoom::NRecord::TTimestampedNamedSeries& series) {
    if (!Accumulators.Defined()) {
        return;
    }

    size_t currentOffset = GetOffset(series.GetStartTimestamp(), StartTime, Resolution);
    for (const auto& value : series.GetValues()) {
        if (Merge) {
            Accumulators->Merge(value, currentOffset);
        } else {
            Accumulators->Mul(value, currentOffset);
        }
        currentOffset++;
    }

    if (currentOffset) {
        if (ActualStartTime) {
            ActualStartTime = Min(ActualStartTime, series.GetStartTimestamp());
        } else {
            ActualStartTime = series.GetStartTimestamp();
        }

        TInstant endTime = StartTime + Resolution * (currentOffset - 1);
        if (ActualEndTime) {
            ActualEndTime = Max(ActualEndTime, endTime);
        } else {
            ActualEndTime = endTime;
        }
    }
}

void TTimestampedSeriesContainer::Overwrite(const TTimestampedSeriesContainer& other, const IOverwritePolicy& policy) {
    if (policy.SkipOperation(ActualStartTime, ActualEndTime, other.ActualStartTime, other.ActualEndTime, Resolution)) {
        return;
    }

    TOverwritingVisitor visitor(this, policy);
    other.Visit(visitor);

    if (ActualStartTime && other.ActualStartTime) {
        ActualStartTime = Min(ActualStartTime, other.ActualStartTime);
    } else if (other.ActualStartTime) {
        ActualStartTime = other.ActualStartTime;
    }

    if (ActualEndTime && other.ActualEndTime) {
        ActualEndTime = Max(ActualEndTime, other.ActualEndTime);
    } else if (other.ActualEndTime) {
        ActualEndTime = other.ActualEndTime;
    }
}

void TTimestampedSeriesContainer::Overwrite(const TTimestampedSeriesContainer& other) {
    TOverwriteAnythingPolicy policy;
    Overwrite(other, policy);
}

void TTimestampedSeriesContainer::Visit(ITimestampedSeriesContainerVisitor& visitor) const {
    visitor.OnStartTime(ActualStartTime ? ActualStartTime : StartTime);
    if (Accumulators.Defined() && ActualStartTime && ActualEndTime) {
        const size_t startOffset(GetOffset(ActualStartTime, StartTime, Resolution));
        const size_t endOffset(GetOffset(ActualEndTime, StartTime, Resolution));
        visitor.OnValueCount(GetValueCount(startOffset, endOffset, Accumulators->Len()));
        for (const auto offset : xrange(Accumulators->Len())) {
            if (startOffset <= offset && offset <= endOffset) {
                visitor.OnValue(Accumulators->GetValue(offset));
            }
        }
    } else {
        visitor.OnValueCount(0);
    }
}
