#include "series.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NContainers;
using namespace NZoom::NValue;
using namespace NZoom::NAccumulators;
using namespace NZoom::NRecord;

namespace {
    class TMockedVisitor final : public ITimestampedSeriesContainerVisitor {
    public:
        void OnStartTime(TInstant startTime) override {
            StartTime = startTime;
        }

        void OnValueCount(size_t count) override {
            Values.reserve(count);
        }

        void OnValue(NZoom::NValue::TValueRef value) override {
            Values.emplace_back(value);
        }

        TInstant StartTime;
        TVector<TValue> Values;
    };
}

Y_UNIT_TEST_SUITE(TTimestampedSeriesContainerTest) {
    Y_UNIT_TEST(Mul) {
        TTimestampedSeriesContainer container(
            MakeMaybe(EAccumulatorType::Summ),
            TInstant::Seconds(100),
            TInstant::Seconds(110),
            TDuration::Seconds(5),
            false
        );

        TVector<TValue> values;
        values.emplace_back(1);
        values.emplace_back(2);
        container.Mul(TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(values), TInstant::Seconds(105)));

        TMockedVisitor visitor;
        container.Visit(visitor);
        TVector<TValue> expectedValues;
        expectedValues.emplace_back(1);
        expectedValues.emplace_back(2);
        UNIT_ASSERT_VALUES_EQUAL(visitor.StartTime, TInstant::Seconds(105));
        UNIT_ASSERT_VALUES_EQUAL(visitor.Values, expectedValues);
    }

    Y_UNIT_TEST(FailedMul) {
        TTimestampedSeriesContainer container(
            MakeMaybe(EAccumulatorType::Summ),
            TInstant::Seconds(100),
            TInstant::Seconds(100),
            TDuration::Seconds(5),
            false
        );

        TVector<TValue> values;
        values.emplace_back(1);
        values.emplace_back(2);
        UNIT_ASSERT_EXCEPTION(container.Mul(
            TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(values), TInstant::Seconds(95))
        ), yexception);

        TVector<TValue> otherValues;
        otherValues.emplace_back(1);
        otherValues.emplace_back(2);
        UNIT_ASSERT_EXCEPTION(container.Mul(
            TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(otherValues), TInstant::Seconds(100))
        ), std::out_of_range);
    }

    Y_UNIT_TEST(Overwrite) {
        TTimestampedSeriesContainer container(
            MakeMaybe(EAccumulatorType::Summ),
            TInstant::Seconds(100),
            TInstant::Seconds(110),
            TDuration::Seconds(5),
            false
        );
        TVector<TValue> values;
        values.emplace_back(1);
        values.emplace_back(2);
        container.Mul(TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(values), TInstant::Seconds(100)));

        TTimestampedSeriesContainer otherContainer(
            MakeMaybe(EAccumulatorType::Summ),
            TInstant::Seconds(105),
            TInstant::Seconds(110),
            TDuration::Seconds(5),
            false
        );
        TVector<TValue> otherValues;
        otherValues.emplace_back(3);
        otherValues.emplace_back(4);
        otherContainer.Mul(TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(otherValues), TInstant::Seconds(105)));

        container.Overwrite(otherContainer);
        TMockedVisitor visitor;
        container.Visit(visitor);
        TVector<TValue> expectedValues;
        expectedValues.emplace_back(1);
        expectedValues.emplace_back(3);
        expectedValues.emplace_back(4);
        UNIT_ASSERT_VALUES_EQUAL(visitor.StartTime, TInstant::Seconds(100));
        UNIT_ASSERT_VALUES_EQUAL(visitor.Values, expectedValues);
    }

    Y_UNIT_TEST(FailedOverwrite) {
        TTimestampedSeriesContainer container(
            MakeMaybe(EAccumulatorType::Summ),
            TInstant::Seconds(100),
            TInstant::Seconds(105),
            TDuration::Seconds(5),
            false
        );
        TVector<TValue> values;
        values.emplace_back(1);
        values.emplace_back(2);
        container.Mul(TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(values), TInstant::Seconds(100)));

        TTimestampedSeriesContainer otherContainer(
            MakeMaybe(EAccumulatorType::Summ),
            TInstant::Seconds(105),
            TInstant::Seconds(110),
            TDuration::Seconds(5),
            false
        );
        TVector<TValue> otherValues;
        otherValues.emplace_back(1);
        otherValues.emplace_back(2);
        otherContainer.Mul(TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(otherValues), TInstant::Seconds(105)));

        UNIT_ASSERT_EXCEPTION(container.Overwrite(otherContainer), std::out_of_range);
    }

    Y_UNIT_TEST(ContinuousOverwrite) {
        TTimestampedSeriesContainer container(
            MakeMaybe(EAccumulatorType::Summ),
            TInstant::Seconds(100),
            TInstant::Seconds(110),
            TDuration::Seconds(5),
            false
        );
        {
            TVector<TValue> values;
            values.emplace_back(1);
            container.Mul(TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(values), TInstant::Seconds(100)));
        }

        TTimestampedSeriesContainer otherContainer(
            MakeMaybe(EAccumulatorType::Summ),
            TInstant::Seconds(105),
            TInstant::Seconds(110),
            TDuration::Seconds(5),
            false
        );
        {
            TVector<TValue> values;
            values.emplace_back(3);
            otherContainer.Mul(TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(values), TInstant::Seconds(110)));
        }

        TOverwriteContinuousPolicy policy;
        container.Overwrite(otherContainer, policy);
        {
            TMockedVisitor visitor;
            container.Visit(visitor);
            TVector<TValue> expectedValues;
            expectedValues.emplace_back(1);
            UNIT_ASSERT_VALUES_EQUAL(visitor.StartTime, TInstant::Seconds(100));
            UNIT_ASSERT_VALUES_EQUAL(visitor.Values, expectedValues);
        }

        {
            TVector<TValue> values;
            values.emplace_back(2);
            otherContainer.Mul(TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(values), TInstant::Seconds(105)));
        }

        container.Overwrite(otherContainer, policy);
        {
            TMockedVisitor visitor;
            container.Visit(visitor);
            TVector<TValue> expectedValues;
            expectedValues.emplace_back(1);
            expectedValues.emplace_back(2);
            expectedValues.emplace_back(3);
            UNIT_ASSERT_VALUES_EQUAL(visitor.StartTime, TInstant::Seconds(100));
            UNIT_ASSERT_VALUES_EQUAL(visitor.Values, expectedValues);
        }
    }

    Y_UNIT_TEST(OverwriteWithSkip) {
        TTimestampedSeriesContainer container(
            MakeMaybe(EAccumulatorType::Summ),
            TInstant::Seconds(100),
            TInstant::Seconds(110),
            TDuration::Seconds(5),
            false
        );
        TVector<TValue> values;
        values.emplace_back(1);
        values.emplace_back(2);
        container.Mul(TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(values), TInstant::Seconds(100)));

        TTimestampedSeriesContainer otherContainer(
            MakeMaybe(EAccumulatorType::Summ),
            TInstant::Seconds(105),
            TInstant::Seconds(110),
            TDuration::Seconds(5),
            false
        );
        TVector<TValue> otherValues;
        otherValues.emplace_back(4);
        otherValues.emplace_back(3);
        otherContainer.Mul(TTimestampedNamedSeries(TStringBuf("signal_summ"), std::move(otherValues), TInstant::Seconds(105)));

        TOverwriteWithSkipPolicy policy(TInstant::Seconds(110), TInstant::Seconds(110));
        container.Overwrite(otherContainer, policy);
        TMockedVisitor visitor;
        container.Visit(visitor);
        TVector<TValue> expectedValues;
        expectedValues.emplace_back(1);
        expectedValues.emplace_back(2);
        expectedValues.emplace_back(3);
        UNIT_ASSERT_VALUES_EQUAL(visitor.StartTime, TInstant::Seconds(100));
        UNIT_ASSERT_VALUES_EQUAL(visitor.Values, expectedValues);
    }
}
