#pragma once

#include <infra/yasm/common/points/accumulators/accumulators.h>
#include <infra/yasm/common/labels/signal/signal_name.h>
#include <infra/yasm/zoom/components/yasmconf/yasmconf.h>

#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/generic/hash.h>

namespace NZoom {
    namespace NContainers {

        template<typename A, typename It>
        class TStorageValueRefConstIterator : public std::iterator<std::forward_iterator_tag, A,
            typename It::difference_type, typename It::pointer, typename It::reference>
        {
        private:
            bool IsNew = false;
            It OldBegin;
            It OldEnd;
            It NewBegin;
            It NewEnd;
            It Current;

        public:
            TStorageValueRefConstIterator(bool isNew, const It& oldBegin, const It& oldEnd,
                const It newBegin, const It& newEnd, const It& current
            )
                : IsNew(isNew)
                , OldBegin(oldBegin)
                , OldEnd(oldEnd)
                , NewBegin(newBegin)
                , NewEnd(newEnd)
                , Current(current)
            {
            }

            template<typename OIt>
            TStorageValueRefConstIterator(const TStorageValueRefConstIterator<A, OIt> other)
                : IsNew(other.IsNew)
                , OldBegin(other.OldBegin)
                , OldEnd(other.OldEnd)
                , NewBegin(other.NewBegin)
                , NewEnd(other.NewEnd)
                , Current(other.Current)
            {
            }

            typename It::reference operator*() const {
                return *Current;
            }

            typename It::pointer operator->() const {
                return &(*Current);
            }

            TStorageValueRefConstIterator& operator++() {
                Next();
                return *this;
            }

            TStorageValueRefConstIterator operator++(int) {
                auto that = *this;
                Next();
                return that;
            }

            template<typename OIt>
            friend bool operator==(const TStorageValueRefConstIterator<A, OIt>& a, const TStorageValueRefConstIterator<A, OIt>& b) {
                return (a.IsNew == b.IsNew) && (a.Current == b.Current);
            }

            template<typename OIt>
            friend bool operator!=(const TStorageValueRefConstIterator<A, OIt>& a, const TStorageValueRefConstIterator<A, OIt>& b) {
                return (a.IsNew != b.IsNew) || (a.Current != b.Current);
            }

        private:
            void Next() {
                ++Current;
                if (!IsNew && Current == OldEnd) {
                    IsNew = true;
                    Current = NewBegin;
                }
            }
        };

        template<typename A>
        class TStorage {
        private:
            using TIntMap = absl::flat_hash_map<NSignal::TSignalName, A, THash<NSignal::TSignalName>>;
            TIntMap NewAccumulators;
            TIntMap OldAccumulators;
            const TAtomicSharedPtr<NZoom::NYasmConf::TTypeConf::TSignalsMap> Signals;
            const TAtomicSharedPtr<NZoom::NYasmConf::TPatterns> Patterns;
            NZoom::NAccumulators::EAggregationMethod AggregationMethod;

        public:
            TStorage(
                    TAtomicSharedPtr<NZoom::NYasmConf::TTypeConf::TSignalsMap> signals,
                    TAtomicSharedPtr<NZoom::NYasmConf::TPatterns> patterns,
                    NZoom::NAccumulators::EAggregationMethod aggregationMethod
                )
                : Signals(signals)
                , Patterns(patterns)
                , AggregationMethod(aggregationMethod)
            {
            }

            TStorageValueRefConstIterator<NZoom::NValue::TValueRef, typename TIntMap::const_iterator> begin() const {
                if (OldAccumulators.empty()) {
                    return TStorageValueRefConstIterator<NZoom::NValue::TValueRef, typename TIntMap::const_iterator>(true,
                        OldAccumulators.begin(), OldAccumulators.end(), NewAccumulators.begin(), NewAccumulators.end(),
                        NewAccumulators.begin());
                }
                return TStorageValueRefConstIterator<NZoom::NValue::TValueRef, typename TIntMap::const_iterator>(false,
                    OldAccumulators.begin(), OldAccumulators.end(), NewAccumulators.begin(), NewAccumulators.end(),
                    OldAccumulators.begin());
            }

            TStorageValueRefConstIterator<NZoom::NValue::TValueRef, typename TIntMap::const_iterator> end() const {
                return TStorageValueRefConstIterator<NZoom::NValue::TValueRef, typename TIntMap::const_iterator>(true,
                    OldAccumulators.begin(), OldAccumulators.end(), NewAccumulators.begin(), NewAccumulators.end(),
                    NewAccumulators.end());
            }


            size_t Len() const noexcept {
                return NewAccumulators.size() + OldAccumulators.size();
            }

            TIntMap& GetNewAccumulators() {
                return NewAccumulators;
            }

            TIntMap& GetOldAccumulators() {
                return OldAccumulators;
            }

            A* GetAccumulator(const NZoom::NSignal::TSignalName& name) {
                if (name.IsOld()) {
                    if (auto it = OldAccumulators.find(name); it != OldAccumulators.end()) {
                        return &it->second;
                    }

                    if (auto it = Signals->find(name.GetName()); it != Signals->end()) {
                        const NZoom::NAccumulators::TAggregationRules& aggregationRules = it->second;
                        auto [newIt, _] = OldAccumulators.emplace(name, A(aggregationRules.GetAccumulatorType(AggregationMethod)));
                        return &newIt->second;
                    }

                    const NZoom::NAccumulators::TAggregationRules* aggregationRules = Patterns->GetMatchedRule(name.GetName());
                    if (aggregationRules == nullptr) {
                        return nullptr;
                    }
                    auto [newIt, _] = OldAccumulators.emplace(name, A(aggregationRules->GetAccumulatorType(AggregationMethod)));
                    return &newIt->second;
                }

                if (auto it = NewAccumulators.find(name); it != NewAccumulators.end()) {
                    return &it->second;
                }

                auto [newIt, _] = NewAccumulators.emplace(name, A(name.GetAggregationType(AggregationMethod).GetRef()));
                return &newIt->second;
            }
        };
    }
}
