#include "record.h"

using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;

void ISignalValueCallback::SetObjectsCount(const size_t /*count*/) {
}

TRecord::TRecord() {
}

TRecord::TRecord(TVector<std::pair<NZoom::NSignal::TSignalName, NZoom::NValue::TValue>> values)
    : Values(std::move(values))
{
}

const TVector<std::pair<TSignalName, TValue>>& TRecord::GetValues() const {
    return Values;
}

TVector<std::pair<TSignalName, TValue>>& TRecord::GetValues() {
    return Values;
}

void TRecord::Process(ISignalValueCallback& callback) const {
    callback.SetObjectsCount(Values.size());
    for (const auto& p: Values) {
        auto value = p.second.GetValue();
        callback.OnSignalValue(p.first, value);
    }
}

size_t TRecord::Len() const noexcept {
    return Values.size();
}

bool TRecord::operator==(const TRecord& other) const {
    if (Values.size() != other.Values.size()) {
        return false;
    }
    THashMap<TSignalName, TValueRef> tmp(Values.size());
    for(const auto& p: Values) {
        tmp.emplace(p.first, p.second.GetValue());
    }
    for(const auto& p: other.Values) {
        const auto it = tmp.find(p.first);
        if (it.IsEnd()) {
            return false;
        }
        if (it->second != p.second.GetValue()) {
            return false;
        }
    }
    return true;
}

void ITagRecordCallback::SetObjectsCount(const size_t /*count*/) {
}

TTaggedRecord::TTaggedRecord() {
}

TTaggedRecord::TTaggedRecord(TVector<std::pair<NTags::TInstanceKey, TRecord>> values)
    : Values(std::move(values))
{
}

const TVector<std::pair<NTags::TInstanceKey, TRecord>>& TTaggedRecord::GetValues() const {
    return Values;
}

TVector<std::pair<NTags::TInstanceKey, TRecord>>& TTaggedRecord::GetValues() {
    return Values;
}

void TTaggedRecord::Process(ITagRecordCallback& callback) const {
    callback.SetObjectsCount(Values.size());
    for (const auto& p: Values) {
        callback.OnTagRecord(p.first, p.second);
    }
}

size_t TTaggedRecord::Len() const noexcept {
    return Values.size();
}

bool TTaggedRecord::operator==(const TTaggedRecord& other) const {
    if (Values.size() != other.Values.size()) {
        return false;
    }
    THashMap<NTags::TInstanceKey, const TRecord*> tmp(Values.size());
    for(const auto& p: Values) {
        tmp.emplace(p.first, &p.second);
    }
    for(const auto& p: other.Values) {
        const auto it = tmp.find(p.first);
        if (it.IsEnd()) {
            return false;
        }
        if (!(*it->second == p.second)) {
            return false;
        }
    }
    return true;
}


TNamedSeries::TNamedSeries(TNamedSeries&& other)
    : SignalName(std::move(other.SignalName))
    , Values(std::move(other.Values))
{
}

TNamedSeries::TNamedSeries(const NZoom::NSignal::TSignalName& signalName, TVector<NZoom::NValue::TValue> values)
    : SignalName(signalName)
    , Values(std::move(values))
{
}

bool TNamedSeries::operator==(const TNamedSeries& other) const {
    return SignalName == other.SignalName && Values == other.Values;
}

const TSignalName& TNamedSeries::GetName() const {
    return SignalName;
}

const TVector<TValue>& TNamedSeries::GetValues() const {
    return Values;
}

void IContinuousSignalCallback::SetSignalsCount(const size_t /*count*/) {
}

TContinuousRecord::TContinuousRecord(TVector<TNamedSeries> values)
    : Inner(std::move(values))
{
}

TContinuousRecord::TContinuousRecord(TContinuousRecord&& other)
    : Inner(std::move(other.Inner))
{
}

const TVector<TNamedSeries>& TContinuousRecord::GetValue() const {
    return Inner;
}

void TContinuousRecord::Process(IContinuousSignalCallback& callback) const {
    callback.SetSignalsCount(Inner.size());
    for (const auto& s: Inner) {
        const auto& values = s.GetValues();
        IContinuousValueCallback* signalCallbackPtr = callback.OnSignal(s.GetName(), values.size());
        if (signalCallbackPtr == nullptr) {
            continue;
        }
        signalCallbackPtr->Start();
        for (const auto& v: values) {
            signalCallbackPtr->OnValue(v.GetValue());
        }
        signalCallbackPtr->Finish();
    }
}

bool TContinuousRecord::operator==(const TContinuousRecord& other) const {
    return Inner == other.Inner;
}

void TContinuousSignalToRecordCallback::SetSignalsCount(const size_t count) {
    Values.reserve(count);
}

IContinuousValueCallback* TContinuousSignalToRecordCallback::OnSignal(
    const NZoom::NSignal::TSignalName& name, const size_t valuesCount)
{
    ValueCallback.ConstructInPlace(name, Values, valuesCount);
    return ValueCallback.Get();
}

TContinuousSignalToRecordCallback::TContinuousValueCallback::TContinuousValueCallback(const NZoom::NSignal::TSignalName& name,
    TVector<TNamedSeries>& values, const size_t size
)
    : Name(name)
    , ParentValues(values)
    , Size(size)
{
}

void TContinuousSignalToRecordCallback::TContinuousValueCallback::Start() {
    Values.reserve(Size);
}

void TContinuousSignalToRecordCallback::TContinuousValueCallback::OnValue(const NZoom::NValue::TValueRef& value) {
    Values.push_back(value);
}

void TContinuousSignalToRecordCallback::TContinuousValueCallback::Finish() {
    ParentValues.emplace_back(Name, std::move(Values));
}

TContinuousRecord* TContinuousSignalToRecordCallback::BuildRecord() {
    return new TContinuousRecord(std::move(Values));
}

NZoom::NRecord::TRecord* TSignalValueToRecordStorageCallback::Create() {
    return new NZoom::NRecord::TRecord(std::move(Values));
}

void TSignalValueToRecordStorageCallback::SetObjectsCount(const size_t count) {
    Values.reserve(count);
}

void TSignalValueToRecordStorageCallback::OnSignalValue(const NZoom::NSignal::TSignalName& name,
    const NZoom::NValue::TValueRef& value)
{
    Values.emplace_back(name, value);
}

TWindowRecord::TWindowRecord(const size_t size)
    : Size(size)
{
}

void TWindowRecord::Process(IContinuousValueCallback& callback) const {
    callback.Start();
    for (const auto& [timestamp, value] : Values) {
        callback.OnValue(value.GetValue());
    }
    callback.Finish();
}

void TWindowRecord::Push(TInstant timestamp, const TValue& value) {
    Values.emplace(timestamp, value.GetValue());

    if (Values.size() > Size) {
        Values.erase(Values.begin());
    }
}

size_t TWindowRecord::Len() const noexcept {
    return Values.size();
}

TTimestampedNamedSeries::TTimestampedNamedSeries(const NZoom::NSignal::TSignalName& signalName,
                                                 TVector<NZoom::NValue::TValue> values, TInstant startTimestamp)
    : SignalName(signalName)
    , Values(std::move(values))
    , StartTimestamp(startTimestamp) {
}

const NZoom::NSignal::TSignalName& TTimestampedNamedSeries::GetName() const {
    return SignalName;
}

const TVector<NZoom::NValue::TValue>& TTimestampedNamedSeries::GetValues() const {
    return Values;
}

TVector<NZoom::NValue::TValue>& TTimestampedNamedSeries::GetValues() {
    return Values;
}

const TInstant TTimestampedNamedSeries::GetStartTimestamp() const {
    return StartTimestamp;
}

bool TTimestampedNamedSeries::operator==(const TTimestampedNamedSeries& other) const {
    return StartTimestamp == other.StartTimestamp && SignalName == other.SignalName && Values == other.Values;
}

TSingleMultiTaggedRecord::TSingleMultiTaggedRecord(TVector<NTags::TInstanceKey> keys, TRecord record)
    : Keys(std::move(keys))
    , Record(std::move(record)) {
}

const TVector<NTags::TInstanceKey>& TSingleMultiTaggedRecord::GetKeys() const {
    return Keys;
}

const TRecord& TSingleMultiTaggedRecord::GetRecord() const {
    return Record;
}
