#pragma once

#include <infra/yasm/common/labels/signal/signal_name.h>
#include <infra/yasm/zoom/components/subscription/subscription.h>
#include <infra/yasm/common/points/value/types.h>
#include <infra/yasm/common/labels/tags/instance_key.h>

#include <util/generic/deque.h>
#include <util/datetime/base.h>

namespace NZoom {
    namespace NRecord {

        class IRecordVisitor {
        public:
            virtual ~IRecordVisitor() = default;

            virtual void OnValue(NZoom::NValue::TValueRef value) = 0;
        };

        class IRecordIterable {
        public:
            virtual ~IRecordIterable() = default;

            virtual void Iterate(IRecordVisitor& visitor) const = 0;
        };


        struct ISignalValueCallback {
            virtual ~ISignalValueCallback() = default;

            virtual void SetObjectsCount(const size_t count);
            virtual void OnSignalValue(const NZoom::NSignal::TSignalName& name, const NZoom::NValue::TValueRef& value) = 0;
        };

        struct IContinuousValueCallback {
            virtual ~IContinuousValueCallback() = default;
            virtual void Start() = 0;
            virtual void OnValue(const NZoom::NValue::TValueRef& value) = 0;
            virtual void Finish() = 0;
        };

        struct IContinuousSignalCallback {
            virtual ~IContinuousSignalCallback() = default;

            virtual void SetSignalsCount(const size_t count);
            virtual IContinuousValueCallback* OnSignal(const NZoom::NSignal::TSignalName& name, const size_t valuesCount) = 0;
        };

        class TRecord {
        private:
            TVector<std::pair<NZoom::NSignal::TSignalName, NZoom::NValue::TValue>> Values;
        public:
            TRecord();
            TRecord(TVector<std::pair<NZoom::NSignal::TSignalName, NZoom::NValue::TValue>> values);

            const TVector<std::pair<NZoom::NSignal::TSignalName, NZoom::NValue::TValue>>& GetValues() const;
            TVector<std::pair<NZoom::NSignal::TSignalName, NZoom::NValue::TValue>>& GetValues();

            void Process(ISignalValueCallback& callback) const;
            size_t Len() const noexcept;

            bool operator==(const TRecord& other) const;
        };

        struct ITagRecordCallback {
            virtual ~ITagRecordCallback() = default;

            virtual void SetObjectsCount(const size_t count);
            virtual void OnTagRecord(NTags::TInstanceKey key, const TRecord& record) = 0;
        };

        class TTaggedRecord {
        private:
            TVector<std::pair<NTags::TInstanceKey, TRecord>> Values;
        public:
            TTaggedRecord();
            TTaggedRecord(TVector<std::pair<NTags::TInstanceKey, TRecord>> values);

            const TVector<std::pair<NTags::TInstanceKey, TRecord>>& GetValues() const;
            TVector<std::pair<NTags::TInstanceKey, TRecord>>& GetValues();

            void Process(ITagRecordCallback& callback) const;
            size_t Len() const noexcept;

            bool operator==(const TTaggedRecord& other) const;
        };

        using TTaggedRecordPtr = TTaggedRecord*;

        class TSingleMultiTaggedRecord {
        private:
            TVector<NTags::TInstanceKey> Keys;
            TRecord Record;
        public:
            TSingleMultiTaggedRecord() = default;
            TSingleMultiTaggedRecord(TSingleMultiTaggedRecord&&) = default;
            TSingleMultiTaggedRecord(TVector<NTags::TInstanceKey> keys, TRecord record);

            const TVector<NTags::TInstanceKey>& GetKeys() const;
            const TRecord& GetRecord() const;
        };

        class TNamedSeries: public TNonCopyable {
        private:
            NZoom::NSignal::TSignalName SignalName;
            TVector<NZoom::NValue::TValue> Values;

        public:
            TNamedSeries(TNamedSeries&& other);
            TNamedSeries(const NZoom::NSignal::TSignalName& signalName, TVector<NZoom::NValue::TValue> values); //by value for std::move

            const NZoom::NSignal::TSignalName& GetName() const;
            const TVector<NZoom::NValue::TValue>& GetValues() const;

            bool operator==(const TNamedSeries& other) const;
        };

        class TWindowRecord {
        private:
            TMap<TInstant, NZoom::NValue::TValue> Values;
            const size_t Size;

        public:
            TWindowRecord(const size_t size);

            void Process(IContinuousValueCallback& callback) const;
            void Push(TInstant timestamp, const NZoom::NValue::TValue& value);
            size_t Len() const noexcept;
        };

        class TTimestampedNamedSeries: public TMoveOnly {
        private:
            NZoom::NSignal::TSignalName SignalName;
            TVector<NZoom::NValue::TValue> Values;
            TInstant StartTimestamp;

        public:
            TTimestampedNamedSeries(const NZoom::NSignal::TSignalName& signalName,
                                    TVector<NZoom::NValue::TValue> values /* by value for std::move */,
                                    TInstant startTimestamp);

            const NZoom::NSignal::TSignalName& GetName() const;
            const TVector<NZoom::NValue::TValue>& GetValues() const;
            TVector<NZoom::NValue::TValue>& GetValues();
            const TInstant GetStartTimestamp() const;

            bool operator==(const TTimestampedNamedSeries& other) const;
        };

        class TContinuousRecord {
        private:
            TVector<TNamedSeries> Inner;

        public:
            TContinuousRecord(TVector<TNamedSeries> values);
            TContinuousRecord(TContinuousRecord&& other);

            const TVector<TNamedSeries>& GetValue() const;
            void Process(IContinuousSignalCallback& callback) const;

            bool operator==(const TContinuousRecord& other) const;
        };

        class TContinuousSignalToRecordCallback: public IContinuousSignalCallback {
        private:
            class TContinuousValueCallback: public IContinuousValueCallback {
                NZoom::NSignal::TSignalName Name;
                TVector<NZoom::NValue::TValue> Values;

                TVector<TNamedSeries>& ParentValues;
                size_t Size = 0;

            public:
                TContinuousValueCallback(const NZoom::NSignal::TSignalName& name, TVector<TNamedSeries>& values, const size_t size);

                void Start() override final;
                void OnValue(const NZoom::NValue::TValueRef& value) override final;
                void Finish() override final;
            };

            TMaybe<TContinuousValueCallback> ValueCallback;
            TVector<TNamedSeries> Values;
        public:
            void SetSignalsCount(const size_t count) override final;
            IContinuousValueCallback* OnSignal(const NZoom::NSignal::TSignalName& name, const size_t valuesCount) override final;

            TContinuousRecord* BuildRecord();
        };

        class TSignalValueToRecordStorageCallback: public NZoom::NRecord::ISignalValueCallback {
        private:
            TVector<std::pair<NZoom::NSignal::TSignalName, NZoom::NValue::TValue>> Values;

        public:
            // caller must destroy created record
            NZoom::NRecord::TRecord* Create();

            void SetObjectsCount(const size_t count) override final;
            void OnSignalValue(const NZoom::NSignal::TSignalName& name, const NZoom::NValue::TValueRef& value) override final;
        };

    }
}
