#pragma once

#include "constants.h"

#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>

#include <library/cpp/json/writer/json.h>

namespace NZoom {
    namespace NPython {

        namespace NJsonImpl {
            using namespace NZoom::NHgram;

            inline void FloatToJson(NJsonWriter::TBuf& packer, double v) noexcept {
                if (IsInteger(v)) {
                    packer.WriteLongLong(v);
                } else {
                    packer.WriteDouble(v);
                }
            }

            struct IUgramIteratorCallback {
                virtual ~IUgramIteratorCallback() = default;

                virtual void OnBucket(double lowerBound, double weight) = 0;
                virtual void OnDelimiter(double upperBound) = 0;
            };

            class TUgramWriter : public IUgramIteratorCallback {
            public:
                TUgramWriter(NJsonWriter::TBuf& packer)
                    : Packer(packer)
                {
                }

                void OnBucket(double lowerBound, double weight) override final {
                    Packer.BeginList();
                    FloatToJson(Packer, lowerBound);
                    FloatToJson(Packer, weight);
                    Packer.EndList();
                }

                void OnDelimiter(double upperBound) override final {
                    Packer.BeginList();
                    FloatToJson(Packer, upperBound);
                    Packer.WriteInt(0);
                    Packer.EndList();
                }

            private:
                NJsonWriter::TBuf& Packer;
            };

            template <typename TUgramCompressorSingleton>
            class THgramStorageCallback: public IHgramStorageCallback {
            public:
                THgramStorageCallback(NJsonWriter::TBuf& packer)
                    : Packer(packer)
                {
                }

                void OnStoreSmall(const TVector<double>& values, const size_t zeros) override final {
                    Packer.BeginList();
                    StoreIn3List(values, zeros);
                    Packer.WriteNull();
                    Packer.EndList();
                }

                void OnStoreNormal(const TVector<double>& values, const size_t zeros, const i16 startPower) override final {
                    Packer.BeginList();
                    StoreIn3List(values, zeros);
                    Packer.WriteInt(startPower);
                    Packer.EndList();
                }

                void OnStoreUgram(const TUgramBuckets& buckets) override final {
                    const TUgramBuckets& compressedBuckets = TUgramCompressorSingleton::GetInstance().Compress(buckets);

                    Packer.BeginList();
                    Packer.WriteString(TUgram::MARKER);

                    if (!compressedBuckets) {
                        Packer.BeginList();
                        Packer.EndList();
                    } else {
                        Packer.BeginList();
                        TUgramWriter writer(Packer);
                        IterateUgram(compressedBuckets, writer);
                        Packer.EndList();
                    }

                    Packer.EndList();
                }

            private:
                inline void StoreIn3List(const TVector<double>& values, const size_t zeros) {
                    Packer.BeginList();
                    for (const double v: values) {
                        FloatToJson(Packer, v);
                    }
                    Packer.EndList();
                    Packer.WriteULongLong(zeros);
                }

                inline void IterateUgram(const TUgramBuckets& buckets, IUgramIteratorCallback& callback) {
                    auto it = buckets.cbegin();
                    callback.OnBucket(it->LowerBound, it->Weight);
                    double lastUpperBound = it->UpperBound;

                    ++it;
                    for (; it != buckets.cend(); ++it) {
                        if (lastUpperBound != it->LowerBound) {
                            callback.OnDelimiter(lastUpperBound);
                        }

                        callback.OnBucket(it->LowerBound, it->Weight);
                        lastUpperBound = it->UpperBound;
                    }

                    callback.OnDelimiter(lastUpperBound);
                }

                NJsonWriter::TBuf& Packer;
            };
        }

        template <typename TUgramCompressorSingleton>
        class TJsonValueRefSerializer: public NZoom::NValue::IUpdatable {
        public:
            TJsonValueRefSerializer(NJsonWriter::TBuf& packer)
                : Packer(packer)
            {
            }

            void MulNone() override final {
                Packer.WriteNull();
            }

            void MulFloat(const double value) override final {
                NJsonImpl::FloatToJson(Packer, value);
            }

            void MulVec(const TVector<double>& value) override final {
                Packer.BeginList();
                if (value.size() != 2) {
                    for(const double v: value) {
                        NJsonImpl::FloatToJson(Packer, v);
                    }
                } else {
                    for(const double v: value) {
                        SafeWriteDouble(v);
                    }
                }
                Packer.EndList();
            }

            void MulCountedSum(const double sum, const ui64 count) override final {
                Packer.BeginList();
                Packer.WriteULongLong(count);
                SafeWriteDouble(sum);
                Packer.EndList();
            }

            void MulHgram(const NHgram::THgram& value) override final {
                NJsonImpl::THgramStorageCallback<TUgramCompressorSingleton> hgramCallback(Packer);
                value.Store(hgramCallback);
            }

        private:
            void SafeWriteDouble(double v) {
                if (IsInteger(v)) {
                    Packer.WriteDouble(v, EFloatToStringMode::PREC_POINT_DIGITS, 1);
                } else {
                    Packer.WriteDouble(v);
                }
            }

            NJsonWriter::TBuf& Packer;
        };
    }
}
