#pragma once

#include "constants.h"

#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

namespace NZoom {
    namespace NPython {

        namespace NMsgpackImpl {
            using namespace NZoom::NHgram;

            template<typename T>
            inline void FloatToMsgpack(msgpack::packer<T>& packer, double v) noexcept {
                if (IsInteger(v)) {
                    packer.pack_int64(v);
                } else {
                    packer.pack_double(v);
                }
            }

            struct IUgramIteratorCallback {
                virtual ~IUgramIteratorCallback() = default;

                virtual void OnBucket(double lowerBound, double weight) = 0;
                virtual void OnDelimiter(double upperBound) = 0;
            };

            class TUgramCounter : public IUgramIteratorCallback {
            public:
                void OnBucket(double, double) override final {
                    Elements++;
                }

                void OnDelimiter(double) override final {
                    Elements++;
                }

                size_t GetElements() {
                    return Elements;
                }

            private:
                size_t Elements = 0;
            };

            template<typename T>
            class TUgramWriter : public IUgramIteratorCallback {
            public:
                TUgramWriter(msgpack::packer<T>& packer)
                    : Packer(packer)
                {
                }

                void OnBucket(double lowerBound, double weight) override final {
                    Packer.pack_array(2);
                    FloatToMsgpack(Packer, lowerBound);
                    FloatToMsgpack(Packer, weight);
                }

                void OnDelimiter(double upperBound) override final {
                    Packer.pack_array(2);
                    FloatToMsgpack(Packer, upperBound);
                    Packer.pack_int8(0);
                }

            private:
                msgpack::packer<T>& Packer;
            };

            template<typename T, typename TUgramCompressorSingleton>
            class THgramStorageCallback: public IHgramStorageCallback {
            private:
                msgpack::packer<T>& Packer;

            public:
                THgramStorageCallback(msgpack::packer<T>& packer)
                    : Packer(packer)
                {
                }

                void OnStoreSmall(const TVector<double>& values, const size_t zeros) override final {
                    StoreIn3List(values, zeros);
                    Packer.pack_nil();
                }

                void OnStoreNormal(const TVector<double>& values, const size_t zeros, const i16 startPower) override final {
                    StoreIn3List(values, zeros);
                    Packer.pack_int16(startPower);
                }

                void OnStoreUgram(const TUgramBuckets& buckets) override final {
                    const TUgramBuckets& compressedBuckets = TUgramCompressorSingleton::GetInstance().Compress(buckets);

                    Packer.pack_array(2);
                    Packer.pack_bin(TUgram::MARKER.length());
                    Packer.pack_bin_body(TUgram::MARKER.data(), TUgram::MARKER.length());

                    if (!compressedBuckets) {
                        Packer.pack_array(0);
                    } else {
                        TUgramCounter counter;
                        IterateUgram(compressedBuckets, counter);
                        Packer.pack_array(counter.GetElements());

                        TUgramWriter<T> writer(Packer);
                        IterateUgram(compressedBuckets, writer);
                    }
                }

            private:
                inline void StoreIn3List(const TVector<double>& values, const size_t zeros) {
                    Packer.pack_array(3);
                    Packer.pack_array(values.size());
                    for (const double v: values) {
                        FloatToMsgpack(Packer, v);
                    }
                    Packer.pack_uint64(zeros);
                }

                inline void IterateUgram(const TUgramBuckets& buckets, IUgramIteratorCallback& callback) {
                    auto it = buckets.cbegin();
                    callback.OnBucket(it->LowerBound, it->Weight);
                    double lastUpperBound = it->UpperBound;

                    ++it;
                    for (; it != buckets.cend(); ++it) {
                        if (lastUpperBound != it->LowerBound) {
                            callback.OnDelimiter(lastUpperBound);
                        }

                        callback.OnBucket(it->LowerBound, it->Weight);
                        lastUpperBound = it->UpperBound;
                    }

                    callback.OnDelimiter(lastUpperBound);
                }
            };
        }

        template<typename T, typename TUgramCompressorSingleton>
        class TValueRefSerializer: public NZoom::NValue::IUpdatable {
        private:
            msgpack::packer<T>& Packer;

        public:
            TValueRefSerializer(msgpack::packer<T>& packer)
                : Packer(packer)
            {
            }

            void MulNone() override final {
                Packer.pack_nil();
            }

            void MulFloat(const double value) override final {
                NMsgpackImpl::FloatToMsgpack(Packer, value);
            }

            void MulVec(const TVector<double>& value) override final {
                Packer.pack_array(value.size());
                if (value.size() != 2) {
                    for(const double v: value) {
                        NMsgpackImpl::FloatToMsgpack(Packer, v);
                    }
                } else {
                    for(const double v: value) {
                        Packer.pack_double(v);
                    }
                }
            }

            void MulCountedSum(const double sum, const ui64 count) override final {
                Packer.pack_array(2);
                Packer.pack_uint64(count);
                Packer.pack_double(sum);
            }

            void MulHgram(const NHgram::THgram& value) override final {
                NMsgpackImpl::THgramStorageCallback<T, TUgramCompressorSingleton> hgramCallback(Packer);
                value.Store(hgramCallback);
            }
        };

        TStringBuf AsStrBuf(const msgpack::object& obj);
        i64 AsInt(const msgpack::object& obj);
    }
}
