#pragma once

#include <util/string/cast.h>

#include <util/generic/maybe.h>

namespace NZoom {
    namespace NPython {

        template<typename TValueType, typename TVecType>
        inline std::pair<double, double> TValueDeserializer<TValueType, TVecType>::VecToPair(const TVecType& ugramBucket) {
            if (ugramBucket.GetSize() != 2) {
                ythrow yexception() << "Ugram bucket should be vector of size 2";
            }
            const TValueType first = ugramBucket.GetItem(0);
            const TValueType second = ugramBucket.GetItem(1);
            return std::make_pair(ItemToFloat(first), ItemToFloat(second));
        }

        template<typename TValueType, typename TVecType>
        inline NZoom::NValue::TValue TValueDeserializer<TValueType, TVecType>::DeserializeUgram(const TVecType& ugramBuckets,
                                                                                                bool skipEmptyBuckets) {
            const size_t sourceCount = ugramBuckets.GetSize();
            if (sourceCount == 0) {
                return NZoom::NValue::TValue(NZoom::NHgram::THgram::EmptyUgram());
            }

            NZoom::NHgram::TUgramBuckets buckets;
            buckets.reserve(sourceCount);

            const TValueType firstItem = ugramBuckets.GetItem(0);
            if (firstItem.GetType() != EType::Vec) {
                ythrow yexception() << "Ugram bucket should be vector";
            }

            double prevLowerBound, prevWeight;
            std::tie(prevLowerBound, prevWeight) = VecToPair(firstItem.AsVec());
            if (prevWeight < 0.0) {
                prevWeight = 0.0;
            }

            for (size_t i = 1; i < sourceCount; ++i) {
                const TValueType item = ugramBuckets.GetItem(i);
                if (item.GetType() != EType::Vec) {
                    ythrow yexception() << "Ugram bucket should be vector";
                }
                double lowerBound, weight;
                std::tie(lowerBound, weight) = VecToPair(item.AsVec());
                if (lowerBound == prevLowerBound) {
                    buckets.push_back(NZoom::NHgram::TUgramBucket::Point(prevLowerBound, prevWeight));
                } else if (prevWeight > 0.0 || (!skipEmptyBuckets && prevWeight == 0.0)) {
                    buckets.push_back(NZoom::NHgram::TUgramBucket(prevLowerBound, lowerBound, prevWeight));
                }
                prevLowerBound = lowerBound;
                prevWeight = weight;
                if (prevWeight < 0.0) {
                    prevWeight = 0.0;
                }
            }
            if (prevWeight > 0.0) {
                buckets.push_back(NZoom::NHgram::TUgramBucket::Point(prevLowerBound, prevWeight));
            }
            return NZoom::NValue::TValue(NZoom::NHgram::THgram::Ugram(std::move(buckets)));
        }

        template<typename TValueType, typename TVecType>
        inline NZoom::NValue::TValue TValueDeserializer<TValueType, TVecType>::DeserializeHgram(const TVecType& values,
            const TValueType& zeros, const TValueType& startPower)
        {
            TVector<double> items;
            const size_t itemsSize = values.GetSize();
            items.reserve(itemsSize);

            TMaybe<size_t> zeroPosition;

            for (size_t i = 0; i < itemsSize; ++i) {
                const double v = ItemToFloat(values.GetItem(i));
                if (v == 0.0 && zeroPosition.Empty()) {
                    zeroPosition = i;
                }
                items.push_back(v);
            }

            const i64 zerosCount = ItemToInt(zeros);
            if (zerosCount < 0) {
                ythrow yexception() << "Negative zeros count";
            }
            const auto startPowerType = startPower.GetType();
            if (startPowerType == EType::None) { //Small
                if (zeroPosition.Defined()) {
                    ythrow yexception() << "Zeros in small hgram on position " << ToString(zeroPosition.GetRef());
                }
                return NZoom::NValue::TValue(NZoom::NHgram::THgram::Small(std::move(items), zerosCount));
            }
            if (startPowerType != EType::Int) {
                ythrow yexception() << "Non integer start power";
            }
            return NZoom::NValue::TValue(NZoom::NHgram::THgram::Normal(std::move(items), zerosCount, startPower.AsInt()));
        }

        template<typename TValueType, typename TVecType>
        inline double TValueDeserializer<TValueType, TVecType>::ItemToFloat(const TValueType& v) {
            const auto type = v.GetType();
            if (type == EType::Int) {
                return v.AsInt();
            } else if (type == EType::Float) {
                return v.AsFloat();
            } else {
                ythrow yexception() << "Wrong element type";
            }
        }

        template<typename TValueType, typename TVecType>
        inline i64 TValueDeserializer<TValueType, TVecType>::ItemToInt(const TValueType& v) {
            const auto type = v.GetType();
            if (type == EType::Int) {
                return v.AsInt();
            } else if (type == EType::Float) {
                return v.AsFloat();
            } else {
                ythrow yexception() << "Wrong element type";
            }
        }

        template<typename TValueType, typename TVecType>
        inline NZoom::NValue::TValue TValueDeserializer<TValueType, TVecType>::GetValueFromVec(const TVecType& item,
                                                                                               bool skipEmptyUgramBuckets) {
            const size_t itemSize = item.GetSize();
            if (itemSize == 2) { // vec? ugram, counted sum
                const TValueType first = item.GetItem(0);
                const TValueType second = item.GetItem(1);
                const auto firstType = first.GetType();

                if (firstType == EType::String) {
                    const TStringBuf firstStr = first.AsString();
                    if (NZoom::NHgram::TUgram::MARKER == firstStr)
                    {
                        if (second.GetType() == EType::Vec) {
                            return DeserializeUgram(second.AsVec(), skipEmptyUgramBuckets);
                        }
                        ythrow yexception() << "Wrong ugram value type";
                    }
                    ythrow yexception() << "Wrong marker: \"" << first.AsString() << "\"";
                } else if (firstType == EType::Int) { // Vec or counted sum
                    const i64 countCandidate = first.AsInt();
                    if (second.GetType() == EType::Float) {
                        const double sumCandidate = ItemToFloat(second);
                        if (countCandidate >= 0 && (countCandidate != 0 || sumCandidate == 0.0)) { //It's a counted sum
                            return NZoom::NValue::TValue(sumCandidate, countCandidate);
                        }
                    }
                }

                TVector<double> res;
                res.reserve(2);
                if (firstType == EType::Int || firstType == EType::Float) {
                    res.push_back(ItemToFloat(first));
                }
                if (second.GetType() == EType::Int || second.GetType() == EType::Float) {
                    res.push_back(ItemToFloat(second));
                }
                return NZoom::NValue::TValue(std::move(res));
            } else if (itemSize == 3) { // hgram, vec
                const TValueType first = item.GetItem(0);
                const auto firstType = first.GetType();
                if (firstType == EType::Vec) {
                    const TValueType second = item.GetItem(1);
                    const TValueType third = item.GetItem(2);
                    return DeserializeHgram(first.AsVec(), second, third);
                }
            }
            // vec
            TVector<double> res;
            res.reserve(itemSize);
            for (size_t i = 0; i < itemSize; ++i) {
                const TValueType value = item.GetItem(i);
                if (value.GetType() == EType::Int || value.GetType() == EType::Float) {
                    res.push_back(ItemToFloat(value));
                }
            }
            return NZoom::NValue::TValue(std::move(res));
        }

        template<typename TValueType, typename TVecType>
        inline NZoom::NValue::TValue TValueDeserializer<TValueType, TVecType>::GetValue(const TValueType& value,
                                                                                        bool skipEmptyUgramBuckets) {
            const EType type = value.GetType();
            if (type == EType::None) {
                return NZoom::NValue::TValue();
            }
            if (type == EType::Float) {
                return NZoom::NValue::TValue(value.AsFloat());
            }
            if (type == EType::Int) {
                return NZoom::NValue::TValue(value.AsInt());
            }
            if (type == EType::Vec) {
                return GetValueFromVec(value.AsVec(), skipEmptyUgramBuckets);
            }
            ythrow yexception() << "Unknown value type";
        }

    }
}
