#pragma once

#include <library/cpp/json/common/defs.h>

namespace NZoom {
    namespace NPython {

        class TAbstactJsonCallback {

        public:
            virtual ~TAbstactJsonCallback() = default;

            virtual bool OnNull() {
                return false;
            }

            virtual bool OnBoolean(bool) {
                return false;
            }

            virtual bool OnInteger(long long) {
                return false;
            }

            virtual bool OnUInteger(unsigned long long) {
                return false;
            }

            virtual bool OnDouble(double) {
                return false;
            }

            virtual bool OnString(const TStringBuf&) {
                return false;
            }

            virtual bool OnOpenMap() {
                return false;
            }

            virtual bool OnMapKey(const TStringBuf&) {
                return true;
            }

            virtual bool OnCloseMap() {
                return true;
            }

            virtual bool OnOpenArray() {
                return false;
            }

            virtual bool OnCloseArray() {
                return true;
            }

            virtual bool OnEnd() {
                return false;
            }

            virtual bool OnStringNoCopy(const TStringBuf& s) {
                return OnString(s);
            }

            virtual bool OnMapKeyNoCopy(const TStringBuf& s) {
                return OnMapKey(s);
            }
        };

        class TZoomJsonDeserializingCallback: public NJson::TJsonCallbacks {
        private:
            TAbstactJsonCallback* Consumer = nullptr;
        public:
            TZoomJsonDeserializingCallback(TAbstactJsonCallback* consumer)
                : NJson::TJsonCallbacks(true)
                , Consumer(consumer)
            {
            }

            void SetConsumer(TAbstactJsonCallback* consumer) {
                Consumer = consumer;
            }

            virtual bool OnNull() override final {
                return Consumer->OnNull();
            }

            virtual bool OnBoolean(bool value) override final {
                return Consumer->OnBoolean(value);
            }

            virtual bool OnInteger(long long value) override final {
                return Consumer->OnInteger(value);
            }

            virtual bool OnUInteger(unsigned long long value) override final {
                return Consumer->OnUInteger(value);
            }

            virtual bool OnDouble(double value) override final {
                return Consumer->OnDouble(value);
            }

            virtual bool OnString(const TStringBuf& value) override final {
                return Consumer->OnString(value);
            }

            virtual bool OnOpenMap() override final {
                return Consumer->OnOpenMap();
            }

            virtual bool OnMapKey(const TStringBuf& key) override final {
                return Consumer->OnMapKey(key);
            }

            virtual bool OnCloseMap() override final {
                return Consumer->OnCloseMap();
            }

            virtual bool OnOpenArray() override final {
                return Consumer->OnOpenArray();
            }

            virtual bool OnCloseArray() override final {
                return Consumer->OnCloseArray();
            }

            virtual bool OnEnd() override final {
                return Consumer->OnEnd();
            }

            bool OnStringNoCopy(const TStringBuf& s) override {
                return  Consumer->OnStringNoCopy(s);
            }

            bool OnMapKeyNoCopy(const TStringBuf& s) override {
                return  Consumer->OnMapKeyNoCopy(s);
            }
        };

        class TSkippingJsonCallback: public TAbstactJsonCallback {
        private:
            TZoomJsonDeserializingCallback& Callback;
            TAbstactJsonCallback* Parent = nullptr;

        protected:
            ui32 Depth = 0;

        public:
            TSkippingJsonCallback(TZoomJsonDeserializingCallback& callback, TAbstactJsonCallback& parent)
                : Callback(callback)
                , Parent(&parent)
            {
            }

            //Skip all
            TSkippingJsonCallback(TZoomJsonDeserializingCallback& callback)
                : Callback(callback)
                , Parent(nullptr)
                , Depth(1) // OnEnd() will decrease it by 1
            {
            }

            inline virtual bool OnNull() override {
                return true;
            }

            inline virtual bool OnBoolean(bool) override {
                return true;
            }

            inline virtual bool OnInteger(long long) override {
                return true;
            }

            inline virtual bool OnUInteger(unsigned long long) override {
                return true;
            }

            inline virtual bool OnDouble(double) override {
                return true;
            }

            inline virtual bool OnString(const TStringBuf&) override {
                return true;
            }

            inline virtual bool OnOpenMap() override {
                ++Depth;
                return true;
            }

            inline virtual bool OnMapKey(const TStringBuf&) override {
                return true;
            }

            virtual bool OnCloseMap() override {
                if (Depth == 0) {
                    Callback.SetConsumer(Parent);
                    return Parent->OnCloseMap();
                }
                --Depth;
                return true;
            }

            virtual bool OnOpenArray() override {
                ++Depth;
                return true;
            }

            virtual bool OnCloseArray() override {
                if (Depth == 0) {
                    Callback.SetConsumer(Parent);
                    return Parent->OnCloseArray();
                }
                --Depth;
                return true;
            }

            virtual bool OnEnd() override {
                if (Depth == 0) {
                    Callback.SetConsumer(Parent);
                    return Parent->OnEnd();
                }
                --Depth;
                return (Depth == 0);
            }

            virtual bool OnStringNoCopy(const TStringBuf&) override {
                return true;
            }

            virtual bool OnMapKeyNoCopy(const TStringBuf&) override {
                return true;
            }
        };
    }
}
