#include "history.h"

using namespace NZoom::NProtobuf;
using namespace NZoom::NValue;
using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;
using EReplicaSelector = NYasm::NInterfaces::NInternal::THistoryAggregatedQuery::EReplicaSelector;

namespace {
    struct TResponseVariants {
        TVector<THistoryResponse*> Good;
        TVector<THistoryResponse*> Bad;
    };

    THistoryResponse CombineResponses(TVector<THistoryResponse*>& responses) {
        Y_ASSERT(!responses.empty());
        THistoryResponse* earliestResponse(*MinElementBy(responses.begin(), responses.end(), [](const THistoryResponse* response) {
            return response->Series.GetStartTimestamp();
        }));
        TVector<TValue> values(Reserve(earliestResponse->Series.GetValues().size()));
        for (auto& response : responses) {
            size_t currentOffset(
                (response->Series.GetStartTimestamp() - earliestResponse->Series.GetStartTimestamp()).GetValue()
                / earliestResponse->Request.Period.GetValue()
            );
            for (auto& value : response->Series.GetValues()) {
                while (values.size() < currentOffset) {
                    values.emplace_back(TValue());
                }
                if (values.size() <= currentOffset) {
                    values.emplace_back(std::move(value));
                } else if (values[currentOffset].IsDefault()) {
                    values[currentOffset] = std::move(value);
                }
                ++currentOffset;
            }
        }
        return THistoryResponse(
            earliestResponse->Request,
            earliestResponse->Series.GetStartTimestamp(),
            std::move(values),
            EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );
    }
}

struct ::NZoom::NProtobuf::THistoryDeserializer {
    template <class T>
    THistoryDeserializer(const T& message)
        : HostNameDeserializer(message.GetHostNameTable())
        , RequestKeyDeserializer(message.GetRequestKeyTable())
        , SignalNameDeserializer(message.GetSignalNameTable())
    {
    }

    THistoryRequest RequestFromProto(const NYasm::NInterfaces::NInternal::THistoryAggregatedQuery& query) const {
        auto requestKey(RequestKeyDeserializer.Deserialize(query.GetRequestKey()));
        auto signalName(SignalNameDeserializer.Deserialize(query.GetSignalName()));
        auto hostName(HostNameDeserializer.Deserialize(query.GetHostName()));
        TReplicaIndex replicaIndex;
        switch (query.GetReplicaSelector()) {
          case EReplicaSelector::THistoryAggregatedQuery_EReplicaSelector_ANY:
            break;
          case EReplicaSelector::THistoryAggregatedQuery_EReplicaSelector_FIRST:
            replicaIndex.ConstructInPlace(0);
            break;
          case EReplicaSelector::THistoryAggregatedQuery_EReplicaSelector_SECOND:
            replicaIndex.ConstructInPlace(1);
            break;
          default:
            throw yexception() << "Parsing error: invalid replica selector given";
        }
        return THistoryRequest{
            .Start = TInstant::Seconds(query.GetStartTimestamp()),
            .End = TInstant::Seconds(query.GetEndTimestamp()),
            .Period = TDuration::Seconds(query.GetPeriod()),
            .HostName = hostName,
            .RequestKey = requestKey,
            .SignalName = signalName,
            .ReplicaIndex = replicaIndex,
            .DontAggregate = query.GetDontAggregate()
        };
    }

    NZoom::NProtobuf::THostNameDeserializer HostNameDeserializer;
    NZoom::NProtobuf::TInternedRequestKeyDeserializer RequestKeyDeserializer;
    NZoom::NProtobuf::TSignalNameDeserializer SignalNameDeserializer;
};

struct ::NZoom::NProtobuf::THistorySerializer {
    template <class T>
    THistorySerializer(T& message)
        : HostNameSerializer(message.MutableHostNameTable())
        , RequesKeySerializer(message.MutableRequestKeyTable())
        , SignalNameSerializer(message.MutableSignalNameTable())
    {
    }

    void RequestToProto(const THistoryRequest& request, NYasm::NInterfaces::NInternal::THistoryAggregatedQuery* query) {
        query->SetStartTimestamp(request.Start.Seconds());
        query->SetEndTimestamp(request.End.Seconds());
        query->SetPeriod(request.Period.Seconds());
        HostNameSerializer.Intern(request.HostName, query->MutableHostName());
        RequesKeySerializer.Intern(request.RequestKey, query->MutableRequestKey());
        SignalNameSerializer.Intern(request.SignalName, query->MutableSignalName());

        if (request.ReplicaIndex.Defined()) {
            query->SetReplicaSelector(static_cast<EReplicaSelector>(request.ReplicaIndex.GetRef() + 1));
        } else {
            query->SetReplicaSelector(EReplicaSelector::THistoryAggregatedQuery_EReplicaSelector_ANY);
        }

        query->SetDontAggregate(request.DontAggregate);
    }

    NZoom::NProtobuf::THostNameSerializer HostNameSerializer;
    NZoom::NProtobuf::TInternedRequestKeySerializer RequesKeySerializer;
    NZoom::NProtobuf::TSignalNameSerializer SignalNameSerializer;
};

size_t THistoryRequest::GetPoints() const {
    if (Period) {
        return (End + Period - Start).GetValue() / Period.GetValue();
    } else {
        return 0;
    }
}

size_t THistoryRequest::Hash() const {
    return MultiHash(Start, End, Period, HostName, RequestKey, SignalName);
}

bool THistoryRequest::operator==(const THistoryRequest& other) const {
    return (
        Start == other.Start
        && End == other.End
        && Period == other.Period
        && HostName == other.HostName
        && RequestKey == other.RequestKey
        && SignalName == other.SignalName
        && MatchReplicas(ReplicaIndex, other.ReplicaIndex)
    );
}

bool THistoryResponse::operator==(const THistoryResponse& other) const {
    return (
        Request == other.Request
        && Series == other.Series
        && StatusCode == other.StatusCode
    );
}

TVector<THistoryRequest> THistoryRequest::FromProto(const THistoryReadAggregatedRequest& request) {
    THistoryDeserializer deserializer(request);
    TVector<THistoryRequest> result(Reserve(request.QueriesSize()));
    for (const auto& seriesRequest : request.GetQueries()) {
        result.emplace_back(deserializer.RequestFromProto(seriesRequest));
    }
    return result;
}

TVector<THistoryResponse> THistoryResponse::FromProto(const THistoryReadAggregatedResponse& response) {
    THistoryDeserializer deserializer(response);
    TVector<THistoryResponse> result(Reserve(response.SeriesSize()));
    for (const auto& seriesResponse : response.GetSeries()) {
        auto& query(seriesResponse.GetQuery());
        THistoryRequest request(deserializer.RequestFromProto(query));
        TVector<NZoom::NValue::TValue> values(TValueSeriesDeserializer::Deserialize(seriesResponse.GetValueSeries()));
        result.emplace_back(request, TInstant::Seconds(seriesResponse.GetStartTimestamp()),
                            std::move(values), seriesResponse.GetStatusCode());
    }
    return result;
}

TVector<THistoryResponse> THistoryResponse::MergeMultiple(TVector<THistoryResponse>& responses) {
    THashMap<THistoryRequest, TResponseVariants> groupedResponses;
    groupedResponses.reserve(responses.size());
    for (auto& response : responses) {
        auto& variants(groupedResponses[response.Request]);
        if (response.StatusCode == EStatusCode::THistoryAggregatedSeries_EStatusCode_OK) {
            variants.Good.emplace_back(&response);
        } else {
            variants.Bad.emplace_back(&response);
        }
    }

    TVector<THistoryResponse> result(Reserve(responses.size()));
    for (auto& [request, variants] : groupedResponses) {
        if (variants.Good.size() >= 2) {
            result.emplace_back(CombineResponses(variants.Good));
        } else if (variants.Good.size() == 1) {
            result.emplace_back(std::move(*variants.Good.front()));
        } else if (variants.Bad.size() >= 2) {
            SortBy(variants.Bad.begin(), variants.Bad.end(), [](const THistoryResponse* response) {
                return response->StatusCode;
            });
            result.emplace_back(std::move(*variants.Bad.front()));
        } else if (variants.Bad.size() == 1) {
            result.emplace_back(std::move(*variants.Bad.front()));
        }
    }
    return result;
}

TVector<THistoryRequest> THistoryResponse::FilterRequests(TReplicaIndex replicaIndex, const TVector<THistoryRequest>& historyRequests) {
    TVector<THistoryRequest> result;
    for (size_t i = 0; i < historyRequests.size(); ++i) {
        if (MatchReplicas(replicaIndex, historyRequests[i].ReplicaIndex)) {
            result.push_back(historyRequests[i]);
        }
    }
    return result;
}


THistoryRequestWriter::THistoryRequestWriter(THistoryReadAggregatedRequest& request)
    : Request(request)
    , HistorySerializer(MakeHolder<THistorySerializer>(Request))
{
}

THistoryRequestWriter::~THistoryRequestWriter() {
}

void THistoryRequestWriter::Reserve(size_t size) {
    Request.MutableQueries()->Reserve(size);
}

void THistoryRequestWriter::Add(const THistoryRequest& request) {
    HistorySerializer->RequestToProto(request, Request.AddQueries());
}

THistoryResponseWriter::THistoryResponseWriter(THistoryReadAggregatedResponse& response)
    : Response(response)
    , HistorySerializer(MakeHolder<THistorySerializer>(Response))
{
}

THistoryResponseWriter::~THistoryResponseWriter() {
}

void THistoryResponseWriter::Reserve(size_t size) {
    Response.MutableSeries()->Reserve(size);
}

void THistoryResponseWriter::Add(const THistoryResponse& response) {
    auto* series(Response.AddSeries());
    HistorySerializer->RequestToProto(response.Request, series->MutableQuery());
    series->SetStartTimestamp(response.Series.GetStartTimestamp().Seconds());
    NZoom::NProtobuf::TValueSeriesSerializer::Serialize(response.Series.GetValues(), series->MutableValueSeries());
    series->SetStatusCode(response.StatusCode);
}

template <>
void Out<EStatusCode>(IOutputStream& stream,
                      TTypeTraits<EStatusCode>::TFuncParam statusCode) {
    stream << "EStatusCode::";
    switch (statusCode) {
        case EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR: {
            stream << "INTERNAL_ERROR";
            return;
        }
        case EStatusCode::THistoryAggregatedSeries_EStatusCode_LIMIT_EXCEEDED: {
            stream << "LIMIT_EXCEEDED";
            return;
        }
        case EStatusCode::THistoryAggregatedSeries_EStatusCode_TIME_EXCEEDED: {
            stream << "TIME_EXCEEDED";
            return;
        }
        case EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND: {
            stream << "NOT_FOUND";
            return;
        }
        case EStatusCode::THistoryAggregatedSeries_EStatusCode_PARTIAL: {
            stream << "PARTIAL";
            return;
        }
        case EStatusCode::THistoryAggregatedSeries_EStatusCode_OK: {
            stream << "OK";
            return;
        }
        case EStatusCode::THistoryAggregatedSeries_EStatusCode_UNKNOWN: {
            stream << "UNKNOWN";
            return;
        }
        default: {
            stream << "__WRONG__";
            return;
        }
    };
}

bool MatchReplicas(const TReplicaIndex first, const TReplicaIndex second) {
    return !first.Defined() || !second.Defined() || first.GetRef() == second.GetRef();
}
