#pragma once

#include <infra/yasm/common/points/value/types.h>
#include <infra/yasm/zoom/components/subscription/request_key.h>
#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/host_name.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/signal_name.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/request_key.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/value_series.h>
#include <infra/yasm/interfaces/internal/history_api.pb.h>

#include <util/datetime/base.h>

// NOTE: undefined TMaybe in this case means reading data from any shard
using TReplicaIndex = TMaybe<uint8_t>;

namespace NZoom::NProtobuf {
    using THistoryReadAggregatedRequest = NYasm::NInterfaces::NInternal::THistoryReadAggregatedRequest;
    using THistoryReadAggregatedResponse = NYasm::NInterfaces::NInternal::THistoryReadAggregatedResponse;

    struct THistoryDeserializer;
    struct THistorySerializer;

    struct THistoryRequest {
        TInstant Start;
        TInstant End;
        TDuration Period;
        NZoom::NHost::THostName HostName;
        NZoom::NSubscription::TInternedRequestKey RequestKey;
        NZoom::NSignal::TSignalName SignalName;
        TReplicaIndex ReplicaIndex;
        bool DontAggregate;

        const TString& GetHostName() const {
            return HostName.GetName();
        }

        size_t GetPoints() const;

        size_t Hash() const;

        bool operator==(const THistoryRequest& other) const;

        static TVector<THistoryRequest> FromProto(const THistoryReadAggregatedRequest& request);
    };

    struct THistoryResponse {
        THistoryResponse(const THistoryRequest& request,
                         TInstant timestamp,
                         TVector<NZoom::NValue::TValue> values,
                         NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode statusCode)
            : Request(request)
            , Series(Request.SignalName, std::move(values), timestamp)
            , StatusCode(statusCode)
        {
        }

        THistoryResponse(const THistoryRequest& request,
                         NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode statusCode)
            : Request(request)
            , Series(Request.SignalName, TVector<NZoom::NValue::TValue>{}, request.Start)
            , StatusCode(statusCode)
        {
        }

        bool operator==(const THistoryResponse& other) const;

        THistoryRequest Request;
        NZoom::NRecord::TTimestampedNamedSeries Series;
        NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode StatusCode;

        static TVector<THistoryResponse> FromProto(const THistoryReadAggregatedResponse& response);
        static TVector<THistoryResponse> MergeMultiple(TVector<THistoryResponse>& responses);
        static TVector<NZoom::NProtobuf::THistoryRequest> FilterRequests(TReplicaIndex replicaIndex, const TVector<NZoom::NProtobuf::THistoryRequest>& historyRequests);
    };

    class THistoryRequestWriter {
    public:
        THistoryRequestWriter(THistoryReadAggregatedRequest& request);
        ~THistoryRequestWriter();

        void Reserve(size_t size);

        void Add(const THistoryRequest& request);

    private:
        THistoryReadAggregatedRequest& Request;
        THolder<THistorySerializer> HistorySerializer;
    };

    class THistoryResponseWriter {
    public:
        THistoryResponseWriter(THistoryReadAggregatedResponse& response);
        ~THistoryResponseWriter();

        void Reserve(size_t size);

        void Add(const THistoryResponse& response);

    private:
        THistoryReadAggregatedResponse& Response;
        THolder<THistorySerializer> HistorySerializer;
    };
}

template<>
struct THash<NZoom::NProtobuf::THistoryRequest> {
    inline size_t operator()(const NZoom::NProtobuf::THistoryRequest& request) const noexcept {
        return request.Hash();
    }
};

bool MatchReplicas(const TReplicaIndex first, const TReplicaIndex second);
