#include "history.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NProtobuf;
using namespace NZoom::NHost;
using namespace NZoom::NSubscription;
using namespace NZoom::NValue;
using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

Y_UNIT_TEST_SUITE(THistorySerialization) {
    Y_UNIT_TEST(TestRequest) {
        THistoryReadAggregatedRequest message;

        THistoryRequest originalRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(105),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=history"),
            .SignalName = TStringBuf("signal_summ")
        };
        UNIT_ASSERT_VALUES_EQUAL(originalRequest.GetPoints(), 2);

        {
            THistoryRequestWriter requestWriter(message);
            requestWriter.Reserve(1);
            requestWriter.Add(originalRequest);
        }

        auto decodedRequests(THistoryRequest::FromProto(message));
        UNIT_ASSERT_VALUES_EQUAL(decodedRequests.size(), 1);
        UNIT_ASSERT_EQUAL(decodedRequests.back(), originalRequest);
    }

    Y_UNIT_TEST(TestResponse) {
        THistoryReadAggregatedResponse message;

        TVector<TValue> values;
        values.emplace_back(TValue(1.));
        THistoryRequest originalRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=history"),
            .SignalName = TStringBuf("signal_summ")
        };
        THistoryResponse originalResponse(
            originalRequest,
            originalRequest.Start,
            std::move(values),
            EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );

        {
            THistoryResponseWriter responseWriter(message);
            responseWriter.Reserve(1);
            responseWriter.Add(originalResponse);
        }

        auto decodedResponses(THistoryResponse::FromProto(message));
        UNIT_ASSERT_VALUES_EQUAL(decodedResponses.size(), 1);
        UNIT_ASSERT_EQUAL(decodedResponses.back(), originalResponse);
        UNIT_ASSERT_EQUAL(decodedResponses.back().Request, originalRequest);
    }

    Y_UNIT_TEST(TestMerge) {
        THistoryRequest request{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=history"),
            .SignalName = TStringBuf("signal_summ")
        };

        TVector<THistoryResponse> responses;
        responses.emplace_back(
            request,
            request.Start,
            TVector<TValue>{},
            EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR
        );

        TVector<TValue> values;
        values.emplace_back(TValue(1.));
        responses.emplace_back(
            request,
            request.Start,
            std::move(values),
            EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );

        TVector<THistoryResponse> mergedResponses(THistoryResponse::MergeMultiple(responses));
        UNIT_ASSERT_VALUES_EQUAL(mergedResponses.size(), 1);

        THistoryResponse response(std::move(mergedResponses.back()));
        UNIT_ASSERT_EQUAL(response.Request, request);
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        TVector<TValue> expectedValues;
        expectedValues.emplace_back(TValue(1.));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
    }

    Y_UNIT_TEST(TestMergeBestValues) {
        THistoryRequest request{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(110),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=history"),
            .SignalName = TStringBuf("signal_summ")
        };

        TVector<THistoryResponse> responses;
        {
            TVector<TValue> values;
            values.emplace_back(TValue());
            values.emplace_back(TValue(2));
            values.emplace_back(TValue(0));
            responses.emplace_back(
                request,
                request.Start,
                std::move(values),
                EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
            );
        }
        {
            TVector<TValue> values;
            values.emplace_back(TValue(1));
            values.emplace_back(TValue());
            values.emplace_back(TValue(3));
            responses.emplace_back(
                request,
                request.Start,
                std::move(values),
                EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
            );
        }

        TVector<THistoryResponse> mergedResponses(THistoryResponse::MergeMultiple(responses));
        UNIT_ASSERT_VALUES_EQUAL(mergedResponses.size(), 1);

        THistoryResponse response(std::move(mergedResponses.back()));
        UNIT_ASSERT_EQUAL(response.Request, request);
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        TVector<TValue> values;
        values.emplace_back(TValue(1));
        values.emplace_back(TValue(2));
        values.emplace_back(TValue(3));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), values);
    }

    Y_UNIT_TEST(TestMergePartialValues) {
        THistoryRequest request{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(110),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=history"),
            .SignalName = TStringBuf("signal_summ")
        };

        TVector<THistoryResponse> responses;
        {
            TVector<TValue> values;
            values.emplace_back(TValue(1));
            responses.emplace_back(
                request,
                request.Start,
                std::move(values),
                EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
            );
        }
        {
            TVector<TValue> values;
            values.emplace_back(TValue(2));
            values.emplace_back(TValue(3));
            responses.emplace_back(
                request,
                request.Start + request.Period,
                std::move(values),
                EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
            );
        }

        TVector<THistoryResponse> mergedResponses(THistoryResponse::MergeMultiple(responses));
        UNIT_ASSERT_VALUES_EQUAL(mergedResponses.size(), 1);

        THistoryResponse response(std::move(mergedResponses.back()));
        UNIT_ASSERT_EQUAL(response.Request, request);
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        TVector<TValue> values;
        values.emplace_back(TValue(1));
        values.emplace_back(TValue(2));
        values.emplace_back(TValue(3));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), values);
    }
}
