#include "py_to_zoom.h"
#include "pydict.h"

#include <infra/yasm/zoom/components/serialization/deserializers/json.h>

#include <library/cpp/pybind/ptr.h>

using namespace NZoom::NPython;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;

namespace {
    static constexpr TStringBuf HANDLED_KEY = "__handled__";
}

TRecord* TPyDeserializer::DeserializeDict(PyObject* value, bool skipEmptyUgramBuckets) {
    if (!PyDict_Check(value)) {
        ythrow yexception() << "Value is not dict";
    }
    const size_t valuesSize = PyDict_Size(value);

    TVector<std::pair<TSignalName, NZoom::NValue::TValue>> items;
    items.reserve(valuesSize);

    PyObject *itemKey, *itemValue;
    Py_ssize_t pos = 0;
    while (PyDict_Next(value, &pos, &itemKey, &itemValue)) {
        char* buf;
        Py_ssize_t len;
        if (PyString_AsStringAndSize(itemKey, &buf, &len)) {
            ythrow yexception() << "Key is not string";
        }
        items.emplace_back(TStringBuf(buf, len), TPythonValueHierarchy::DeserializeValue(itemValue, skipEmptyUgramBuckets));
    }
    return new TRecord(std::move(items));
}

NZoom::NRecord::TRecord* TPyDeserializer::DeserializeIterable(PyObject* value, bool skipEmptyUgramBuckets) {
    NPyBind::TPyObjectPtr iterator(PyObject_GetIter(value), true);
    if (iterator.Get() == nullptr) {
        ythrow yexception() << "Value is not iterable";
    }
    TVector<std::pair<TSignalName, NZoom::NValue::TValue>> items;
    PyObject *item;
    while (item = PyIter_Next(iterator.Get())) {
        NPyBind::TPyObjectPtr pairPtr(item, true);

        PyObject *itemKey, *itemValue;
        if (PyTuple_Check(item)) {
            if (PyTuple_GET_SIZE(item) != 2) {
                continue;
            }
            itemKey = PyTuple_GET_ITEM(item, 0);
            itemValue = PyTuple_GET_ITEM(item, 1);
        } else if (PyList_Check(item)) {
            if (PyList_GET_SIZE(item) != 2) {
                continue;
            }
            itemKey = PyList_GET_ITEM(item, 0);
            itemValue = PyList_GET_ITEM(item, 1);
        } else {
            ythrow yexception() << "Value item is not tuple nor list";
        }
        char* buf;
        Py_ssize_t len;
        if (PyString_AsStringAndSize(itemKey, &buf, &len)) {
            ythrow yexception() << "Key is not string";
        }
        items.emplace_back(TStringBuf(buf, len), TPythonValueHierarchy::DeserializeValue(itemValue, skipEmptyUgramBuckets));
    }

    if (PyErr_Occurred()) {
        ythrow yexception() << "Exception happend";
    }
    return new TRecord(std::move(items));
}

NZoom::NValue::TValue* NZoom::NPython::PyToValue(PyObject* value, bool skipEmptyUgramBuckets) {
    return new NZoom::NValue::TValue(TPythonValueHierarchy::DeserializeValue(value, skipEmptyUgramBuckets));
}

NZoom::NValue::TValue NZoom::NPython::PyToOwnedValue(PyObject* value, bool skipEmptyUgramBuckets) {
    return TPythonValueHierarchy::DeserializeValue(value, skipEmptyUgramBuckets);
}

NZoom::NRecord::TRecord TJsonPyDeserializer::Deserialize(const NJson::TJsonValue& value, bool skipEmptyUgramBuckets) {
    if (!value.IsMap()) {
        ythrow yexception() << "Json is not map";
    }
    const auto& jsonMap = value.GetMap();
    const size_t dataSize = jsonMap.size();
    TVector<std::pair<NZoom::NSignal::TSignalName, NZoom::NValue::TValue>> items;
    items.reserve(dataSize);

    for (const auto& kvPair: jsonMap) {
        if (kvPair.first == HANDLED_KEY) {
            continue;
        }
        items.emplace_back(kvPair.first, TJsonValueHierarchy::DeserializeValue(kvPair.second, skipEmptyUgramBuckets));
    }
    return {std::move(items)};
}
