#include "zoom_to_py.h"

#include <infra/yasm/common/points/value/types.h>
#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>
#include <infra/yasm/zoom/components/serialization/common/constants.h>

#include <util/string/cast.h>
#include <util/system/tls.h>

#include <cmath>

using namespace NZoom::NPython;
using namespace NZoom::NRecord;
using namespace NZoom::NHgram;
using namespace NZoom::NHost;
using namespace NZoom::NSubscription;
using namespace NZoom::NSignal;

namespace {
    PyObject* FloatToPy(double v) {
        double intpart;
        if (v > MIN_WHOLE_F64 && v < MAX_WHOLE_F64 && std::modf(v, &intpart) == 0.0) {
            return PyInt_FromSsize_t(i64(v));
        }
        return PyFloat_FromDouble(v);
    }

    NPyBind::TPyObjectPtr Vec2List(const TVector<double>& value) {
        NPyBind::TPyObjectPtr list(PyList_New(value.size()), true);
        for (size_t i = 0; i < value.size(); ++i) {
            PyList_SetItem(list.Get(), i, FloatToPy(value[i]));
        }
        return list;
    }

    template <class TUgramCompressorSingleton>
    class THgramStorageCallback: public IHgramStorageCallback {
    private:
        NPyBind::TPyObjectPtr Result;

    public:
        void OnStoreSmall(const TVector<double>& values, const size_t zeros) final {
            NPyBind::TPyObjectPtr list = StoreIn3List(values, zeros);
            Py_INCREF(Py_None);
            PyList_SetItem(list.Get(), 2, Py_None);
            Result = std::move(list);
        }

        void OnStoreNormal(const TVector<double>& values, const size_t zeros, const i16 startPower) final {
            NPyBind::TPyObjectPtr list = StoreIn3List(values, zeros);
            PyList_SetItem(list.Get(), 2, PyInt_FromSsize_t(startPower));
            Result = std::move(list);
        }

        void OnStoreUgram(const TUgramBuckets& buckets) final {
            const TUgramBuckets& compressedBuckets = TUgramCompressorSingleton::GetInstance().Compress(buckets);
            size_t count = 0;
            if (!compressedBuckets.empty()) {
                count = compressedBuckets.size() + 1;
                for (size_t i = 1; i < compressedBuckets.size(); ++i) {
                    if (compressedBuckets[i - 1].UpperBound != compressedBuckets[i].LowerBound) {
                        ++count;
                    }
                }
            }

            NPyBind::TPyObjectPtr bucketsList(PyList_New(count), true);
            size_t curIdx = 0;
            for (size_t i = 0; i < compressedBuckets.size(); ++i) {
                const auto& currentBucket = compressedBuckets[i];
                PyList_SetItem(bucketsList.Get(), curIdx,
                    MakePair(currentBucket.LowerBound, currentBucket.Weight).RefGet()
                );
                ++curIdx;
                const size_t nextIdx = i + 1;
                if (nextIdx == compressedBuckets.size() ||
                    (currentBucket.UpperBound != compressedBuckets[nextIdx].LowerBound))
                {
                    PyList_SetItem(bucketsList.Get(), curIdx,
                        MakePair(currentBucket.UpperBound, 0.0).RefGet());
                    ++curIdx;
                }
            }
            NPyBind::TPyObjectPtr list(PyList_New(2), true);
            PyList_SetItem(list.Get(), 0, PyString_FromStringAndSize(TUgram::MARKER.data(), TUgram::MARKER.size()));
            PyList_SetItem(list.Get(), 1, bucketsList.RefGet());
            Result = std::move(list);
        }

        NPyBind::TPyObjectPtr GetSerialized() {
            return Result;
        }

    private:
        static NPyBind::TPyObjectPtr StoreIn3List(const TVector<double>& values, const size_t zeros) {
            NPyBind::TPyObjectPtr list(PyList_New(3), true);
            PyList_SetItem(list.Get(), 0, Vec2List(values).RefGet());
            PyList_SetItem(list.Get(), 1, PyInt_FromSize_t(zeros));
            return list;
        }

        static NPyBind::TPyObjectPtr MakePair(const double bound, const double weight) {
            NPyBind::TPyObjectPtr tuple(PyTuple_New(2), true);
            PyTuple_SetItem(tuple.Get(), 0, FloatToPy(bound));
            PyTuple_SetItem(tuple.Get(), 1, FloatToPy(weight));
            return tuple;
        }
    };

    template <class TUgramCompressorSingleton>
    class TValueToPy: public NZoom::NValue::IUpdatable {
    private:
        NPyBind::TPyObjectPtr ObjectHolder;

    public:
        void MulNone() final {
            ObjectHolder = NPyBind::TPyObjectPtr(Py_None);
        }

        void MulFloat(const double value) final {
            ObjectHolder =  NPyBind::TPyObjectPtr(FloatToPy(value), true);
        }

        void MulVec(const TVector<double>& value) final {
            ObjectHolder = Vec2List(value);
        }

        void MulCountedSum(const double sum, const ui64 count) final {
            NPyBind::TPyObjectPtr list(PyList_New(2), true);
            PyList_SetItem(list.Get(), 0, PyInt_FromSize_t(count));
            PyList_SetItem(list.Get(), 1, PyFloat_FromDouble(sum));
            ObjectHolder = std::move(list);
        }

        void MulHgram(const THgram& value) final {
            THgramStorageCallback<TUgramCompressorSingleton> callback;
            value.Store(callback);
            ObjectHolder = callback.GetSerialized();
        }

        NPyBind::TPyObjectPtr GetObject() noexcept {
            return ObjectHolder;
        }
    };

}

PyObject* NZoom::NPython::ValueToPy(const NZoom::NValue::TValueRef& value, bool skipEmptyUgramBuckets) {
    if (skipEmptyUgramBuckets) {
        TValueToPy<TUgramCompressor> valueToPy;
        value.Update(valueToPy);
        return valueToPy.GetObject().RefGet();
    } else {
        TValueToPy<TLazyUgramCompressor> valueToPy;
        value.Update(valueToPy);
        return valueToPy.GetObject().RefGet();
    }
}

void TSignalValueToDictStorage::Init(bool skipEmptyUgramBuckets) {
    ObjectHolder = NPyBind::TPyObjectPtr(PyDict_New(), true);
    SkipEmptyUgramBuckets = skipEmptyUgramBuckets;
}

PyObject* TSignalValueToDictStorage::GetValue() {
    return ObjectHolder.RefGet();
}

PyObject* TSignalValueToDictStorage::GetValueRef() {
    return ObjectHolder.Get();
}

void TSignalValueToDictStorage::OnSignalValue(const NZoom::NSignal::TSignalName& name, const NZoom::NValue::TValueRef& value) {
    NPyBind::TPyObjectPtr v;
    if (SkipEmptyUgramBuckets) {
        TValueToPy<TUgramCompressor> valueToPy;
        value.Update(valueToPy);
        v = valueToPy.GetObject();
    } else {
        TValueToPy<TLazyUgramCompressor> valueToPy;
        value.Update(valueToPy);
        v = valueToPy.GetObject();
    }
    PyDict_SetItemString(ObjectHolder.Get(), name.GetName().data(), v.Get());
}

void TTagRecordToDictStorage::Init() {
    ObjectHolder = NPyBind::TPyObjectPtr(PyDict_New(), true);
}

PyObject* TTagRecordToDictStorage::GetValue() {
    return ObjectHolder.RefGet();
}

void TTagRecordToDictStorage::OnTagRecord(NTags::TInstanceKey key, const NZoom::NRecord::TRecord& record) {
    TSignalValueToDictStorage storage;
    storage.Init(true);
    record.Process(storage);
    PyDict_SetItemString(ObjectHolder.Get(), key.ToNamed().data(), storage.GetValueRef());
}

TPyListIterator::TPyListIterator() {
}

void TPyListIterator::Init(PyObject* list) {
    if (!PyList_Check(list)) {
        ythrow yexception() << "Passed object is not list";
    }
    List = list;
    Size = PyList_Size(list);
    Position = 0;
}

TStringBuf TPyListIterator::GetCurrent() const {
    PyObject* value = PyList_GET_ITEM(List, Position);
    char* buf;
    Py_ssize_t len;
    if (PyString_AsStringAndSize(value, &buf, &len)) {
        ythrow yexception() << "Not String item on position " << ToString(Position);
    }
    return TStringBuf(buf, len);
}

bool TPyListIterator::IsValid() const {
    return Position != Size;
}

void TPyListIterator::Next() {
    ++Position;
}

TContinuousValueToAccumulatorCallback::TContinuousValueToAccumulatorCallback(NZoom::NAccumulators::TAccumulator& accumulator)
    : Accumulator(accumulator)
{
}

void TContinuousValueToAccumulatorCallback::Start() {
    Accumulator.Clean();
}

void TContinuousValueToAccumulatorCallback::OnValue(const NZoom::NValue::TValueRef& value) {
    Accumulator.Mul(value);
}

void TContinuousValueToAccumulatorCallback::Finish() {
}

TSubscriptionMergedDataPySerializer::TSubscriptionMergedDataPySerializer(bool allowLegacyTypes)
    : ValueDictHolder(PyDict_New(), true)
    , BorderDictHolder(PyDict_New(), true)
    , ConvertedSignalsHolder(PyDict_New(), true)
    , AllowLegacyTypes(allowLegacyTypes) {
}

void TSubscriptionMergedDataPySerializer::TraverseMergerData(const NZoom::NSubscription::TSubscriptionValueSeriesMerger& merger) {
    ValueDictHolder = NPyBind::TPyObjectPtr(PyDict_New(), true);
    BorderDictHolder = NPyBind::TPyObjectPtr(PyDict_New(), true);
    ConvertedSignalsHolder = NPyBind::TPyObjectPtr(PyDict_New(), true);

    for (const auto& [reqKey, reqKeyData]: merger.GetMergedDataTable()) {
        auto reqKeyValuesDict = NPyBind::TPyObjectPtr(PyDict_New(), true);
        auto reqKeyBordersDict = NPyBind::TPyObjectPtr(PyDict_New(), true);
        auto reqKeyConvertedDict = NPyBind::TPyObjectPtr(PyDict_New(), true);
        auto reqKeyDictKey = NPyBind::TPyObjectPtr(PyString_FromString(reqKey.GetName().data()), true);
        PyDict_SetItem(ValueDictHolder.Get(), reqKeyDictKey.Get(), reqKeyValuesDict.Get());
        PyDict_SetItem(BorderDictHolder.Get(), reqKeyDictKey.Get(), reqKeyBordersDict.Get());
        PyDict_SetItem(ConvertedSignalsHolder.Get(), reqKeyDictKey.Get(), reqKeyConvertedDict.Get());

        for (const auto& [signal, signalData]: reqKeyData) {
            bool smallHgramConverted = false;
            auto valuesList = NPyBind::TPyObjectPtr(PyList_New(signalData.SeriesLength), true);
            for (size_t pos = 0; pos < signalData.SeriesLength; ++pos) {
                TValueToPy<TUgramCompressor> valueToPy;
                if (!signalData.Accumulators.Empty()) {
                    const auto& value = signalData.Accumulators[pos].GetValue();
                    if (!AllowLegacyTypes && value.GetType() == NValue::EValueType::SMALL_HGRAM) {
                        NZoom::NAccumulators::TAccumulator convertingAccumulator(NAccumulators::EAccumulatorType::Hgram, false);
                        smallHgramConverted = true;
                        convertingAccumulator.Mul(value);
                        convertingAccumulator.GetValue().Update(valueToPy);
                    } else {
                        value.Update(valueToPy);
                    }
                } else {
                    valueToPy.MulNone();
                }
                PyList_SetItem(valuesList.Get(), pos, valueToPy.GetObject().RefGet()); // SetItem steals reference, so RefGet()
            }
            auto bordersPair = NPyBind::TPyObjectPtr(PyTuple_New(2), true);
            PyTuple_SetItem(bordersPair.Get(), 0, PyInt_FromSize_t(signalData.LeftBorder));
            PyTuple_SetItem(bordersPair.Get(), 1, PyInt_FromSize_t(signalData.RightBorder));

            auto signalDictKey = NPyBind::TPyObjectPtr(PyString_FromString(signal.GetName().data()), true);
            PyDict_SetItem(reqKeyValuesDict.Get(), signalDictKey.Get(), valuesList.Get());
            PyDict_SetItem(reqKeyBordersDict.Get(), signalDictKey.Get(), bordersPair.Get());

            auto flag = NPyBind::TPyObjectPtr(PyBool_FromLong(smallHgramConverted), true);
            PyDict_SetItem(reqKeyConvertedDict.Get(), signalDictKey.Get(), flag.Get());
        }
    }
}

PyObject* TSubscriptionMergedDataPySerializer::GetValues() {
    return ValueDictHolder.Get();
}

PyObject* TSubscriptionMergedDataPySerializer::GetBorders() {
    return BorderDictHolder.Get();
}

PyObject* TSubscriptionMergedDataPySerializer::GetConvertedSignals() {
    return ConvertedSignalsHolder.Get();
}
