#include "zoom_to_json.h"

#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>
#include <infra/yasm/zoom/components/serialization/deserializers/json.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/json/json_reader.h>

using namespace NZoom::NRecord;
using namespace NZoom::NPython;
using namespace NZoom::NHgram;
using namespace NZoom::NValue;
using namespace NZoom::NSignal;

Y_UNIT_TEST_SUITE(TZoomToJsonTest) {

    TRecord CreateOriginalRecord() {
        TVector<std::pair<TSignalName, TValue>> values;
        values.emplace_back(TString("name0"), TValue());
        values.emplace_back(TString("name1"), TValue(1.0));
        values.emplace_back(TString("name2"), TValue(2, 2));
        values.emplace_back(TString("name3"), TValue({1, 2, 3}));
        values.emplace_back(TString("name4"), TValue(THgram::Small({1, 2, 3}, 10)));
        values.emplace_back(TString("name5"), TValue(THgram::Normal({0.0, 51.0, 1.0, 1.0, 50.0}, 0, -1)));

        values.emplace_back(TString("name6"), TValue(
            THgram::Ugram({
                TUgramBucket(1.0, 2.0, 2.0),
                TUgramBucket(2.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 2.0)
             })));
        values.emplace_back(TString("name7"), TValue(TVector<double>({2, 2})));

        return std::move(values);
    }

    Y_UNIT_TEST(RecordCopyingViaJsonSerialization) {
        TRecord orig(CreateOriginalRecord());
        NJsonWriter::TBuf jsonBuf;
        TJsonSerializer<TUgramCompressor> callback(jsonBuf);
        orig.Process(callback);

        NJson::TJsonValue root;
        UNIT_ASSERT(NJson::ReadJsonTree(jsonBuf.Str(), &root));

        THashMap<TSignalName, TValueRef> values;
        for (const auto& [signalName, value] : orig.GetValues()) {
            values.emplace(signalName, value.GetValue());
        }

        UNIT_ASSERT_VALUES_EQUAL(values.size(), root.GetMap().size());
        for (const auto& [signalName, valueRoot] : root.GetMap()) {
            UNIT_ASSERT_VALUES_EQUAL(TValue(values.at(signalName)), TJsonValueHierarchy::DeserializeValue(valueRoot, true));
        }
    }

    Y_UNIT_TEST(EmptyRecord) {
        TVector<std::pair<TSignalName, TValue>> values;
        TRecord record(std::move(values));

        NJsonWriter::TBuf jsonBuf;
        TJsonSerializer<TUgramCompressor> callback(jsonBuf);
        record.Process(callback);

        NJson::TJsonValue root;
        UNIT_ASSERT(NJson::ReadJsonTree(jsonBuf.Str(), &root));
        UNIT_ASSERT(root.IsMap());
        UNIT_ASSERT(root.GetMap().empty());
    }
}
