#include "zoom_to_msgpack.h"

#include <infra/yasm/zoom/components/serialization/common/msgpack_utils.h>
#include <infra/yasm/zoom/components/serialization/deserializers/abstract/abstract_value.h>
#include <infra/yasm/zoom/components/serialization/deserializers/msgpack.h>

#include <util/string/cast.h>

using namespace NZoom::NPython;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;
using namespace NZoom::NHgram;

TMsgPackRefSerializer::TMsgPackRefSerializer(msgpack::packer<msgpack::sbuffer>& packer, bool skipEmptyUgramBuckets)
    : Packer(packer)
    , SkipEmptyUgramBuckets(skipEmptyUgramBuckets)
{
}

void TMsgPackRefSerializer::SetObjectsCount(const size_t count) {
    if (TargetCount != 0) {
        ythrow yexception() << "SetObjectsCount() invoked twice";
    }
    Packer.pack_array(count);
    TargetCount = count;
}

void TMsgPackRefSerializer::OnSignalValue(const TSignalName& name, const NZoom::NValue::TValueRef& value) {
    const TString& nameStr = name.GetName();
    Packer.pack_array(2);
    Packer.pack_bin(nameStr.size());
    Packer.pack_bin_body(nameStr.data(), nameStr.size());

    if (SkipEmptyUgramBuckets) {
        TValueRefSerializer<msgpack::sbuffer, TUgramCompressor> valueSerializer(Packer);
        value.Update(valueSerializer);
    } else {
        TValueRefSerializer<msgpack::sbuffer, TLazyUgramCompressor> valueSerializer(Packer);
        value.Update(valueSerializer);
    }

    ++ActualCount;
    if (TargetCount < ActualCount) {
        ythrow yexception() << "OnSignalValue() invoked more than requested";
    }
}

bool TMsgPackRefSerializer::IsFinished() const {
    return TargetCount == ActualCount;
}

TMsgPackSerializer::TMsgPackSerializer(bool skipEmptyUgramBuckets)
    : Packer(&Buffer)
    , Impl(Packer, skipEmptyUgramBuckets)
{
}

void TMsgPackSerializer::SetObjectsCount(const size_t count) {
    Impl.SetObjectsCount(count);
}

void TMsgPackSerializer::OnSignalValue(const TSignalName& name, const NZoom::NValue::TValueRef& value) {
    Impl.OnSignalValue(name, value);
}

TStringBuf TMsgPackSerializer::GetValue() const {
    if (!Impl.IsFinished()) {
        ythrow yexception() << "GetValue() count mismatch";
    }
    return TStringBuf(Buffer.data(), Buffer.size());
}

TRecord* TMsgPackDeserializer::Deserialize(TStringBuf value, bool skipEmptyUgramBuckets) {
    msgpack::unpacked msg;
    msgpack::unpack(msg, value.data(), value.size());
    msgpack::object obj = msg.get();
    if (obj.type != msgpack::type::ARRAY) {
        ythrow yexception() << "MsgPack is not array";
    }
    const size_t dataSize = obj.via.array.size;

    TVector<std::pair<NZoom::NSignal::TSignalName, NZoom::NValue::TValue>> values;
    values.reserve(dataSize);

    for (size_t i = 0; i < dataSize; ++i) {
        const msgpack::object& item = obj.via.array.ptr[i];
        if (item.type != msgpack::type::ARRAY) {
            ythrow yexception() << "MsgPack item " << ToString(i) << " is not array";
        }
        if (item.via.array.size != 2) {
            ythrow yexception() << "MsgPack item " << ToString(i) << " array has wrong size " << ToString(item.via.array.size);
        }

        values.emplace_back(AsStrBuf(item.via.array.ptr[0]),
            TMsgPackValueHierarchy::DeserializeValue(item.via.array.ptr[1], skipEmptyUgramBuckets));
    }
    return new TRecord(std::move(values));
}
