#include "record.h"

#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/xrange.h>

using namespace NZoom::NProtobuf;
using namespace NZoom::NRecord;
using namespace NZoom::NHgram;
using namespace NZoom::NValue;
using namespace NZoom::NSignal;

Y_UNIT_TEST_SUITE(TRecordToProtobufTest) {

    TRecord CreateOriginalRecord() {
        TVector<std::pair<TSignalName, TValue>> values;
        values.emplace_back(TString("name0"), TValue());
        values.emplace_back(TString("name1"), TValue(1.0));
        values.emplace_back(TString("name2"), TValue(2, 2));
        values.emplace_back(TString("name3"), TValue({1, 2, 3}));
        values.emplace_back(TString("name4"), TValue(THgram::Small({1, 2, 3}, 10)));
        values.emplace_back(TString("name5"), TValue(THgram::Normal({0.0, 51.0, 1.0, 1.0, 50.0}, 0, -1)));

        values.emplace_back(TString("name6"), TValue(
            THgram::Ugram({
                TUgramBucket(1.0, 2.0, 2.0),
                TUgramBucket(2.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 2.0)
             })));
        values.emplace_back(TString("name7"), TValue(TVector<double>({2, 2})));

        return std::move(values);
    }

    Y_UNIT_TEST(RecordCopyingViaProtobufSerialization) {
        NYasm::NInterfaces::NInternal::TSignalNameTable signalTable;
        TSignalNameSerializer signalSerializer(&signalTable);

        TRecord orig(CreateOriginalRecord());
        NYasm::NInterfaces::NInternal::TRecord protoRecord;
        TProtobufRecordSerializer<TUgramCompressor> serializer(protoRecord, signalSerializer);
        orig.Process(serializer);

        TSignalNameDeserializer signalDeserializer(signalTable);
        TRecord copy(DeserializeProtobufRecord(protoRecord, signalDeserializer, true));
        UNIT_ASSERT_EQUAL(orig, copy);
    }

    Y_UNIT_TEST(EmptyRecord) {
        NYasm::NInterfaces::NInternal::TSignalNameTable signalTable;
        TSignalNameSerializer signalSerializer(&signalTable);

        TVector<std::pair<TSignalName, TValue>> values;
        TRecord orig(std::move(values));
        NYasm::NInterfaces::NInternal::TRecord protoRecord;
        TProtobufRecordSerializer<TUgramCompressor> serializer(protoRecord, signalSerializer);
        orig.Process(serializer);

        TSignalNameDeserializer signalDeserializer(signalTable);
        TRecord copy(DeserializeProtobufRecord(protoRecord, signalDeserializer, true));
        UNIT_ASSERT_EQUAL(orig, copy);
    }

    Y_UNIT_TEST(UgramSkipEmptyBuckets) {
        NYasm::NInterfaces::NInternal::TSignalNameTable signalTable;
        TSignalNameSerializer signalSerializer(&signalTable);

        TVector<std::pair<TSignalName, TValue>> valuesProvided;
        valuesProvided.emplace_back(TString("ugram_name_hgram"), TValue(
            THgram::Ugram({
                TUgramBucket(0.0, 1.0, 0.0),
                TUgramBucket(1.0, 2.0, 2.0),
                TUgramBucket(2.0, 3.0, 0.0),
                TUgramBucket(3.0, 4.0, 2.0)
            })));

        TVector<std::pair<TSignalName, TValue>> valuesExpected;
        valuesExpected.emplace_back(TString("ugram_name_hgram"), TValue(
            THgram::Ugram({
                TUgramBucket(1.0, 2.0, 2.0),
                TUgramBucket(3.0, 4.0, 2.0)
            })));

        TRecord originalRecord(std::move(valuesProvided));
        TRecord expectedRecord(std::move(valuesExpected));

        NYasm::NInterfaces::NInternal::TRecord protoRecord;
        TProtobufRecordSerializer<TUgramCompressor> serializer(protoRecord, signalSerializer);
        originalRecord.Process(serializer);

        TSignalNameDeserializer signalDeserializer1(signalTable);
        TRecord result1 = DeserializeProtobufRecord(protoRecord, signalDeserializer1, false);
        UNIT_ASSERT_EQUAL(result1, expectedRecord);

        TSignalNameDeserializer signalDeserializer2(signalTable);
        TRecord result2 = DeserializeProtobufRecord(protoRecord, signalDeserializer2, false);
        UNIT_ASSERT_EQUAL(result2, expectedRecord);
    }

    Y_UNIT_TEST(UgramDontSkipEmptyBuckets) {
        NYasm::NInterfaces::NInternal::TSignalNameTable signalTable;
        TSignalNameSerializer signalSerializer(&signalTable);

        TVector<std::pair<TSignalName, TValue>> valuesProvided;
        valuesProvided.emplace_back(TString("ugram_name_hgram"), TValue(
            THgram::Ugram({
                TUgramBucket(0.0, 1.0, 0.0),
                TUgramBucket(1.0, 2.0, 2.0),
                TUgramBucket(2.0, 3.0, 0.0),
                TUgramBucket(3.0, 4.0, 2.0)
            })));

        TVector<std::pair<TSignalName, TValue>> valuesExpected1;
        valuesExpected1.emplace_back(TString("ugram_name_hgram"), TValue(
            THgram::Ugram({
                TUgramBucket(0.0, 1.0, 0.0),
                TUgramBucket(1.0, 2.0, 2.0),
                TUgramBucket(2.0, 3.0, 0.0),
                TUgramBucket(3.0, 4.0, 2.0)
            })));

        TVector<std::pair<TSignalName, TValue>> valuesExpected2;
        valuesExpected2.emplace_back(TString("ugram_name_hgram"), TValue(
            THgram::Ugram({
                TUgramBucket(1.0, 2.0, 2.0),
                TUgramBucket(3.0, 4.0, 2.0)
            })));

        TRecord originalRecord(std::move(valuesProvided));
        TRecord expectedRecord1(std::move(valuesExpected1));
        TRecord expectedRecord2(std::move(valuesExpected2));

        NYasm::NInterfaces::NInternal::TRecord protoRecord;
        TProtobufRecordSerializer<TLazyUgramCompressor> serializer(protoRecord, signalSerializer);
        originalRecord.Process(serializer);

        TSignalNameDeserializer signalDeserializer1(signalTable);
        TRecord result1 = DeserializeProtobufRecord(protoRecord, signalDeserializer1, false);
        UNIT_ASSERT_EQUAL(result1, expectedRecord1);

        TSignalNameDeserializer signalDeserializer2(signalTable);
        TRecord result2 = DeserializeProtobufRecord(protoRecord, signalDeserializer2, true);
        UNIT_ASSERT_EQUAL(result2, expectedRecord2);
    }
}
