#include "subscription.h"

namespace NZoom {
    using namespace NProtobuf;
    using namespace NSubscription;
    namespace NYasmProto = NYasm::NInterfaces::NInternal;

    TSubscriptionSerializer::TSubscriptionSerializer(THostNameSerializer& hostNameSerializer,
                                                     TInternedRequestKeySerializer& requestKeySerializer,
                                                     TSignalExpressionSerializer& signalNameSerializer)
        : HostNameSerializer(hostNameSerializer)
        , RequestKeySerializer(requestKeySerializer)
        , SignalNameSerializer(signalNameSerializer) {
    }

    void TSubscriptionSerializer::Intern(const TSubscription& source, NYasmProto::TSubscription* dest) {
        HostNameSerializer.Intern(source.GetHostName(), dest->MutableHostName());
        RequestKeySerializer.Intern(source.GetRequestKey(), dest->MutableRequestKey());
        SignalNameSerializer.Intern(source.GetSignalExpression(), dest->MutableSignalExpression());
    }

    TSubscriptionDeserializer::TSubscriptionDeserializer(const THostNameDeserializer& hostNameDeserializer,
                                                         const TInternedRequestKeyDeserializer& requestKeyDeserializer,
                                                         const TSignalExpressionDeserializer& signalNameDeserializer)
         : HostNameDeserializer(hostNameDeserializer)
         , RequestKeyDeserializer(requestKeyDeserializer)
         , SignalNameDeserializer(signalNameDeserializer) {
    }

    TSubscription TSubscriptionDeserializer::Deserialize(const NYasmProto::TSubscription& subscription) const {
        return TSubscription(
            HostNameDeserializer.Deserialize(subscription.GetHostName()),
            RequestKeyDeserializer.Deserialize(subscription.GetRequestKey()),
            SignalNameDeserializer.Deserialize(subscription.GetSignalExpression()));
    }

    std::pair<NSubscription::TSubscription, TString> TSubscriptionDeserializer::DeserializeWithFullInfo(
        const NYasm::NInterfaces::NInternal::TSubscription& subscription) const {

        const auto& [requestKey, requestKeyAsProvided] = RequestKeyDeserializer.DeserializeWithFullInfo(
            subscription.GetRequestKey());

        return std::pair<TSubscription, TString>(
            TSubscription(
                HostNameDeserializer.Deserialize(subscription.GetHostName()),
                RequestKeyDeserializer.Deserialize(subscription.GetRequestKey()),
                SignalNameDeserializer.Deserialize(subscription.GetSignalExpression())),
            requestKeyAsProvided);
    }

    TSubscriptionWithValueSeriesSerializer::TSubscriptionWithValueSeriesSerializer(
        THostNameSerializer& hostNameSerializer,
        TInternedRequestKeySerializer& requestKeySerializer,
        TSignalExpressionSerializer& signalNameSerializer)
        : SubscriptionSerializer(hostNameSerializer, requestKeySerializer, signalNameSerializer) {
    }

    void TSubscriptionWithValueSeriesSerializer::Intern(const TSource& source, TTargetProto* dest) {
        Intern(source.Subscription, source.ValueSeries.FirstValueTimestamp, source.ValueSeries.Values, dest);
    }

    TSubscriptionWithValueSeriesDeserializer::TSubscriptionWithValueSeriesDeserializer(
        const THostNameDeserializer& hostNameDeserializer,
        const TInternedRequestKeyDeserializer& requestKeyDeserializer,
        const TSignalExpressionDeserializer& signalNameDeserializer)
        : SubscriptionDeserializer(hostNameDeserializer, requestKeyDeserializer, signalNameDeserializer) {
    }

    TSubscriptionWithValueSeries TSubscriptionWithValueSeriesDeserializer::Deserialize(
        const NYasmProto::TSubscriptionWithValueSeries& subWithValuesProto) const {

        TInstant firstValueTimestamp = TInstant::Seconds(subWithValuesProto.GetStartTimestamp());
        TValueSeries valueSeries(firstValueTimestamp, TValueSeriesDeserializer::Deserialize(subWithValuesProto.GetValueSeries()));

        return TSubscriptionWithValueSeries(
            SubscriptionDeserializer.Deserialize(subWithValuesProto.GetSubscription()),
            std::move(valueSeries));
    }
}
