#include "subscription.h"

#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/xrange.h>

using namespace NZoom::NProtobuf;
using namespace NZoom::NSubscription;
using namespace NZoom::NValue;

namespace {
    // Treats 0 as None.
    TValueSeries DoublesToValueSeries(TVector<double> doubles, TInstant firstValueTimestamp) {
        TVector<TValue> result;
        for (auto curDouble: doubles) {
            if (0 != curDouble) {
                result.emplace_back(curDouble);
            } else {
                result.emplace_back();
            }
        }
        return TValueSeries(firstValueTimestamp, std::move(result));
    }
}

Y_UNIT_TEST_SUITE(TSubscriptionToProtobufTest) {
    Y_UNIT_TEST(Serialization) {
        NYasm::NInterfaces::NInternal::THostNameTable hostNameTable;
        NYasm::NInterfaces::NInternal::TRequestKeyTable requestKeyTable;
        NYasm::NInterfaces::NInternal::TSignalNameTable signalNameTable;
        THostNameSerializer hostNameSerializer(&hostNameTable);
        TInternedRequestKeySerializer requesKeySerializer(&requestKeyTable);
        TSignalExpressionSerializer signalNameSerializer(&signalNameTable);
        TSubscriptionSerializer serializer(hostNameSerializer, requesKeySerializer, signalNameSerializer);

        NYasm::NInterfaces::NInternal::TSubscription subscription;
        serializer.Intern(TSubscription(
            TStringBuf("ASEARCH"),
            TStringBuf("itype=app;ctype=prod;prj=project;geo=sas"),
            TSubscription::TSignalExpression("signal_summ")), &subscription);

        THostNameDeserializer hostNameDeserializer(hostNameTable);
        TInternedRequestKeyDeserializer requestKeyDeserializer(requestKeyTable);
        TSignalExpressionDeserializer signalNameDeserializer(signalNameTable);
        TSubscriptionDeserializer deserializer(hostNameDeserializer, requestKeyDeserializer, signalNameDeserializer);
        UNIT_ASSERT_VALUES_EQUAL(deserializer.Deserialize(subscription),
            TSubscription(
                TStringBuf("ASEARCH"),
                TStringBuf("ctype=prod;itype=app;prj=project;geo=sas"),
                TSubscription::TSignalExpression("signal_summ")));
    }

    Y_UNIT_TEST(DeserializeWithFullInfo) {
        NYasm::NInterfaces::NInternal::THostNameTable hostNameTable;
        NYasm::NInterfaces::NInternal::TRequestKeyTable requestKeyTable;
        NYasm::NInterfaces::NInternal::TSignalNameTable signalNameTable;
        THostNameSerializer hostNameSerializer(&hostNameTable);
        TInternedRequestKeySerializer requesKeySerializer(&requestKeyTable);
        TSignalExpressionSerializer signalNameSerializer(&signalNameTable);
        TSubscriptionSerializer serializer(hostNameSerializer, requesKeySerializer, signalNameSerializer);

        NYasm::NInterfaces::NInternal::TSubscription subscription;
        serializer.Intern(TSubscription(
            TStringBuf("ASEARCH"),
            TStringBuf("itype=app;ctype=prod;prj=project;geo=sas"),
            TSubscription::TSignalExpression("signal_summ")), &subscription);

        // modify request key table to emulate deserialization with non-normalized request key
        TString requestKeyAsProvided = TString("itype=app;ctype=prod;prj=project;geo=sas");
        requestKeyTable.SetName(0, requestKeyAsProvided);

        THostNameDeserializer hostNameDeserializer(hostNameTable);
        TInternedRequestKeyDeserializer requestKeyDeserializer(requestKeyTable);
        TSignalExpressionDeserializer signalNameDeserializer(signalNameTable);
        TSubscriptionDeserializer deserializer(hostNameDeserializer, requestKeyDeserializer, signalNameDeserializer);
        auto [actualSubscription, actualRequestKeyAsProvided] = deserializer.DeserializeWithFullInfo(subscription);
        UNIT_ASSERT_VALUES_EQUAL(
            TSubscription(
                TStringBuf("ASEARCH"),
                TStringBuf("ctype=prod;itype=app;prj=project;geo=sas"),
                TSubscription::TSignalExpression("signal_summ")),
            actualSubscription);
        UNIT_ASSERT_VALUES_EQUAL(requestKeyAsProvided, actualRequestKeyAsProvided);
    }

    Y_UNIT_TEST(Missing) {
        NYasm::NInterfaces::NInternal::THostNameTable hostNameTable;
        NYasm::NInterfaces::NInternal::TRequestKeyTable requestKeyTable;
        NYasm::NInterfaces::NInternal::TSignalNameTable signalNameTable;

        NYasm::NInterfaces::NInternal::TSubscription subscription;
        THostNameDeserializer hostNameDeserializer(hostNameTable);
        TInternedRequestKeyDeserializer requestKeyDeserializer(requestKeyTable);
        TSignalExpressionDeserializer signalNameDeserializer(signalNameTable);
        TSubscriptionDeserializer deserializer(hostNameDeserializer, requestKeyDeserializer, signalNameDeserializer);
        UNIT_ASSERT_EXCEPTION(deserializer.Deserialize(subscription), std::out_of_range);
    }

    Y_UNIT_TEST(SeriesSerialization) {
        NYasm::NInterfaces::NInternal::THostNameTable hostNameTable;
        NYasm::NInterfaces::NInternal::TRequestKeyTable requestKeyTable;
        NYasm::NInterfaces::NInternal::TSignalNameTable signalNameTable;
        THostNameSerializer hostNameSerializer(&hostNameTable);
        TInternedRequestKeySerializer requesKeySerializer(&requestKeyTable);
        TSignalExpressionSerializer signalNameSerializer(&signalNameTable);
        TSubscriptionWithValueSeriesSerializer serializer(hostNameSerializer, requesKeySerializer, signalNameSerializer);

        NYasm::NInterfaces::NInternal::TSubscriptionWithValueSeries subscriptionWithValuesProto;
        TSubscriptionWithValueSeries subscriptionWithValues(
            TSubscription(
                TStringBuf("ASEARCH"),
                TStringBuf("itype=app;ctype=prod;prj=project;geo=sas"),
                TSubscription::TSignalExpression("signal_summ")),
            DoublesToValueSeries({0, 3, 5, 6, 0, 0}, TInstant::Seconds(1234567)));

        serializer.Intern(subscriptionWithValues, &subscriptionWithValuesProto);

        THostNameDeserializer hostNameDeserializer(hostNameTable);
        TInternedRequestKeyDeserializer requestKeyDeserializer(requestKeyTable);
        TSignalExpressionDeserializer signalNameDeserializer(signalNameTable);
        TSubscriptionWithValueSeriesDeserializer deserializer(hostNameDeserializer, requestKeyDeserializer, signalNameDeserializer);
        UNIT_ASSERT_EQUAL(deserializer.Deserialize(subscriptionWithValuesProto), subscriptionWithValues);
    }

    Y_UNIT_TEST(SeriesMissing) {
        NYasm::NInterfaces::NInternal::THostNameTable hostNameTable;
        NYasm::NInterfaces::NInternal::TRequestKeyTable requestKeyTable;
        NYasm::NInterfaces::NInternal::TSignalNameTable signalNameTable;

        NYasm::NInterfaces::NInternal::TSubscriptionWithValueSeries subscriptionWithValuesProto;
        subscriptionWithValuesProto.SetStartTimestamp(777);

        THostNameDeserializer hostNameDeserializer(hostNameTable);
        TInternedRequestKeyDeserializer requestKeyDeserializer(requestKeyTable);
        TSignalExpressionDeserializer signalNameDeserializer(signalNameTable);
        TSubscriptionWithValueSeriesDeserializer deserializer(hostNameDeserializer, requestKeyDeserializer, signalNameDeserializer);
        UNIT_ASSERT_EXCEPTION(deserializer.Deserialize(subscriptionWithValuesProto), std::out_of_range);
    }
}
