#include "value.h"

using namespace NZoom::NProtobuf;
using namespace NZoom::NValue;
using namespace NZoom::NHgram;

namespace {
    template <class T>
    TVector<T> CreateVector(const ::google::protobuf::RepeatedField<T>& source) {
        TVector<T> result(Reserve(source.size()));
        for (const auto& item : source) {
            result.emplace_back(item);
        }
        return result;
    }

    TUgramBuckets CreateUgramBuckets(
        const ::google::protobuf::RepeatedPtrField< ::NYasm::NInterfaces::NInternal::TUgramBucket >& buckets,
        bool skipEmptyUgramBuckets)
    {
        TUgramBuckets result(Reserve(buckets.size()));
        for (const auto& bucket : buckets) {
            if (bucket.GetWeight() > 0.0 || (!skipEmptyUgramBuckets && bucket.GetWeight() == 0.0)) {
                result.emplace_back(
                    bucket.GetLowerBound(),
                    bucket.GetUpperBound(),
                    bucket.GetWeight()
                );
            }
        }
        return result;
    }
}

namespace NZoom::NProtobuf {
    TValue DeserializeProtobufValue(const NYasm::NInterfaces::NInternal::TValue& value, bool skipEmptyUgramBuckets) {
        using EValueCase = NYasm::NInterfaces::NInternal::TValue::ValueCase;
        switch (value.GetValueCase()) {
            case EValueCase::VALUE_NOT_SET:
            case EValueCase::kNoneValue: {
                return TValue();
            }
            case EValueCase::kFloatValue: {
                return TValue(value.GetFloatValue().GetValue());
            }
            case EValueCase::kCountedSum: {
                return TValue(
                    value.GetCountedSum().GetSum(),
                    value.GetCountedSum().GetCount()
                );
            }
            case EValueCase::kVec: {
                return TValue(CreateVector(value.GetVec().GetValues()));
            }
            case EValueCase::kHgramSmall: {
                const auto& hgram(value.GetHgramSmall());
                return TValue(THgram::Small(
                    CreateVector(hgram.GetValues()),
                    hgram.GetZeros()
                ));
            }
            case EValueCase::kHgramNormal: {
                const auto& hgram(value.GetHgramNormal());
                return TValue(THgram::Normal(
                    CreateVector(hgram.GetValues()),
                    hgram.GetZeros(),
                    hgram.GetStartPower()
                ));
            }
            case EValueCase::kUgram: {
                return TValue(THgram::Ugram(CreateUgramBuckets(value.GetUgram().GetBuckets(), skipEmptyUgramBuckets)));
            }
            case EValueCase::kHyperLogLog: {
                ythrow yexception() << "hyperloglog not implemented";
            }
        }
    }
}

void TProtobufWrappingValue::UpdateFlat(IMultiUpdatable& accumulator) const {
    using EValueCase = NYasm::NInterfaces::NInternal::TValue::ValueCase;
    switch (ProtoValue.GetValueCase()) {
        case EValueCase::VALUE_NOT_SET:
        case EValueCase::kHyperLogLog: // NOT IMPLEMENTED
        case EValueCase::kNoneValue: {
            accumulator.MulNone();
            break;
        }
        case EValueCase::kFloatValue: {
            accumulator.MulFloat(ProtoValue.GetFloatValue().GetValue());
            break;
        }
        case EValueCase::kCountedSum: {
            accumulator.MulCountedSum(
                ProtoValue.GetCountedSum().GetSum(),
                ProtoValue.GetCountedSum().GetCount());
            break;
        }
        case EValueCase::kVec: {
            accumulator.MulVec(CreateVector(ProtoValue.GetVec().GetValues()));
            break;
        }
        case EValueCase::kHgramSmall: {
            const auto& hgram = ProtoValue.GetHgramSmall();
            auto valuesArray = TArrayRef<const double>(hgram.GetValues().data(), hgram.GetValues().size());
            accumulator.OnStoreSmall(valuesArray, hgram.GetZeros());
            break;
        }
        case EValueCase::kHgramNormal: {
            const auto& hgram = ProtoValue.GetHgramNormal();
            auto valuesArray = TArrayRef<const double>(hgram.GetValues().data(), hgram.GetValues().size());
            accumulator.OnStoreNormal(valuesArray, hgram.GetZeros(), hgram.GetStartPower());
            break;
        }
        case EValueCase::kUgram: {
            accumulator.OnStoreUgram(CreateUgramBuckets(ProtoValue.GetUgram().GetBuckets(), true));
            break;
        }
    }
}

EValueType TProtobufWrappingValue::GetType() const noexcept {
    using EValueCase = NYasm::NInterfaces::NInternal::TValue::ValueCase;
    switch (ProtoValue.GetValueCase()) {
        case EValueCase::VALUE_NOT_SET:
        case EValueCase::kHyperLogLog: // NOT IMPLEMENTED
        case EValueCase::kNoneValue: {
            return EValueType::NONE;
        }
        case EValueCase::kFloatValue: {
            return EValueType::FLOAT;
        }
        case EValueCase::kCountedSum: {
            return EValueType::COUNTED_SUM;
        }
        case EValueCase::kVec: {
            return EValueType::VEC;
        }
        case EValueCase::kHgramSmall: {
            return EValueType::SMALL_HGRAM;
        }
        case EValueCase::kHgramNormal: {
            return EValueType::NORMAL_HGRAM;
        }
        case EValueCase::kUgram: {
            return EValueType::USER_HGRAM;
        }
    }
}

bool TProtobufWrappingValue::IsDefault() const noexcept {
    using EValueCase = NYasm::NInterfaces::NInternal::TValue::ValueCase;
    switch (ProtoValue.GetValueCase()) {
        case EValueCase::VALUE_NOT_SET:
        case EValueCase::kHyperLogLog: // NOT IMPLEMENTED
        case EValueCase::kNoneValue: {
            return true;
        }
        case EValueCase::kFloatValue: {
            return 0.0 == ProtoValue.GetFloatValue().GetValue();
        }
        case EValueCase::kCountedSum: {
            return 0.0 == ProtoValue.GetCountedSum().GetSum() &&
                0 == ProtoValue.GetCountedSum().GetCount();
        }
        case EValueCase::kVec: {
            return 0 == ProtoValue.GetVec().ValuesSize();
        }
        case EValueCase::kHgramSmall: {
            return 0 == ProtoValue.GetHgramSmall().ValuesSize() &&
                0 == ProtoValue.GetHgramSmall().GetZeros();
        }
        case EValueCase::kHgramNormal: {
            return 0 == ProtoValue.GetHgramNormal().ValuesSize() &&
                0 == ProtoValue.GetHgramNormal().GetZeros();
        }
        case EValueCase::kUgram: {
            return 0 == ProtoValue.GetUgram().BucketsSize();
        }
    }
}
