#pragma once

#include <infra/yasm/interfaces/internal/value.pb.h>
#include <infra/yasm/common/points/value/types.h>
#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>

namespace NZoom::NProtobuf {

    template <class TUgramCompressorSingleton>
    class TProtobufValueSerializer: public NZoom::NValue::IUpdatable, public NZoom::NHgram::IHgramStorageCallback {
    public:
        TProtobufValueSerializer(NYasm::NInterfaces::NInternal::TValue& target)
            : Target(target)
        {
        }

        void MulNone() override final {
            Target.MutableNoneValue();
        }

        void MulFloat(const double value) override final {
            Target.MutableFloatValue()->SetValue(value);
        }

        void MulVec(const TVector<double>& values) override final {
            Copy(values, *Target.MutableVec()->MutableValues());
        }

        void MulCountedSum(const double sum, const ui64 count) override final {
            auto* dest(Target.MutableCountedSum());
            dest->SetSum(sum);
            dest->SetCount(count);
        }

        void MulHgram(const NHgram::THgram& value) override final {
            value.Store(*this);
        }

        void OnStoreSmall(const TVector<double>& values, const size_t zeros) override final {
            auto* dest(Target.MutableHgramSmall());
            Copy(values, *dest->MutableValues());
            dest->SetZeros(zeros);
        }

        void OnStoreNormal(const TVector<double>& values, const size_t zeros, const i16 startPower) override final {
            auto* dest(Target.MutableHgramNormal());
            Copy(values, *dest->MutableValues());
            dest->SetZeros(zeros);
            dest->SetStartPower(startPower);
        }

        void OnStoreUgram(const NZoom::NHgram::TUgramBuckets& buckets) override final {
            const NZoom::NHgram::TUgramBuckets& compressedBuckets = TUgramCompressorSingleton::GetInstance().Compress(buckets);
            auto* dest(Target.MutableUgram()->MutableBuckets());
            dest->Reserve(compressedBuckets.size());
            for (const auto& bucket : compressedBuckets) {
                auto* newBucket(dest->Add());
                newBucket->SetLowerBound(bucket.LowerBound);
                newBucket->SetUpperBound(bucket.UpperBound);
                newBucket->SetWeight(bucket.Weight);
            }
        }

    private:
        template <class T>
        void Copy(const TVector<T>& source, ::google::protobuf::RepeatedField<T>& dest) {
            dest.Reserve(source.size());
            for (const auto& item : source) {
                dest.Add(item);
            }
        }

        NYasm::NInterfaces::NInternal::TValue& Target;
    };

    NZoom::NValue::TValue DeserializeProtobufValue(const NYasm::NInterfaces::NInternal::TValue& value,
                                                   bool skipEmptyUgramBuckets);

    class TProtobufWrappingValue: public NValue::IFlatValue {
    public:
        explicit TProtobufWrappingValue(const NYasm::NInterfaces::NInternal::TValue& protoValue)
            : ProtoValue(protoValue) {
        }
        TProtobufWrappingValue(const TProtobufWrappingValue&) = default;

        void UpdateFlat(NValue::IMultiUpdatable& accumulator) const final;
        NValue::EValueType GetType() const noexcept final;
        bool IsDefault() const noexcept final;

    private:
        const NYasm::NInterfaces::NInternal::TValue& ProtoValue;
    };

    TProtobufWrappingValue DeserializeProtobufValueByWrapping(const NYasm::NInterfaces::NInternal::TValue& value);
}
