#include "subscription.h"

#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/xrange.h>

using namespace NZoom::NProtobuf;
using namespace NZoom::NSubscription;
using namespace NZoom::NValue;
using namespace NZoom::NHgram;

namespace {
    // Treats 0 as None.
    TVector<TValue> DoublesToValueSeries(TVector<double> doubles) {
        TVector<TValue> result;
        for (auto curDouble: doubles) {
            if (0 != curDouble) {
                result.emplace_back(curDouble);
            } else {
                result.emplace_back();
            }
        }
        return result;
    }

    class TUgramOnlyVisitor : public IUpdatable, public IHgramStorageCallback {
    public:
        void MulNone() override {
            Y_FAIL();
        }

        void MulFloat(const double) override {
            Y_FAIL();
        }

        void MulVec(const TVector<double>&) override {
            Y_FAIL();
        }

        void MulCountedSum(const double, const ui64) override {
            Y_FAIL();
        }

        void MulHgram(const NZoom::NHgram::THgram& hgram) override {
            hgram.Store(*this);
        }

        void OnStoreSmall(const TVector<double>&, const size_t) override {
            Y_FAIL();
        }

        void OnStoreNormal(const TVector<double>&, const size_t, const i16) override {
            Y_FAIL();
        }

        void OnStoreUgram(const TUgramBuckets& buckets) override {
            UNIT_ASSERT_LE(buckets.size(), UGRAM_LIMIT);
        }
    };
}

Y_UNIT_TEST_SUITE(TValueSeriesToProtobufTest) {
    Y_UNIT_TEST(SeriesSerialization) {
        NYasm::NInterfaces::NInternal::TValueSeries valueSeriesProto;
        auto valueSeries = DoublesToValueSeries({0, 3, 5, 6, 0, 0});

        TValueSeriesSerializer::Serialize(valueSeries, &valueSeriesProto);
        UNIT_ASSERT_VALUES_EQUAL(valueSeries, TValueSeriesDeserializer::Deserialize(valueSeriesProto));
    }

    Y_UNIT_TEST(HgramSeriesSerialization) {
        TVector<TValue> valuesEncoded;
        valuesEncoded.emplace_back(NZoom::NHgram::THgram::Small(TVector<double>{2, 3, 4}, 2));
        valuesEncoded.emplace_back(2);
        valuesEncoded.emplace_back(3);
        valuesEncoded.emplace_back(4);
        valuesEncoded.emplace_back(NZoom::NHgram::THgram::Small(TVector<double>{5, 6, 7}, 5));
        valuesEncoded.emplace_back();
        valuesEncoded.emplace_back();

        NYasm::NInterfaces::NInternal::TValueSeries valueSeriesProto;
        TValueSeriesSerializer::Serialize(valuesEncoded, &valueSeriesProto);

        TVector<NZoom::NValue::TValue> valuesExpected;
        valuesExpected.emplace_back(NZoom::NHgram::THgram::Small(TVector<double>{2, 3, 4}, 2));
        valuesExpected.emplace_back();
        valuesExpected.emplace_back();
        valuesExpected.emplace_back();
        valuesExpected.emplace_back(NZoom::NHgram::THgram::Small(TVector<double>{5, 6, 7}, 5));
        valuesExpected.emplace_back();
        valuesExpected.emplace_back();

        UNIT_ASSERT_VALUES_EQUAL(valuesExpected, TValueSeriesDeserializer::Deserialize(valueSeriesProto));
    }

    Y_UNIT_TEST(BigUgramSerialization) {
        TUgramBuckets buckets;
        for (const auto idx : xrange(500)) {
            buckets.emplace_back(TUgramBucket(idx, idx + 1, 1));
        }
        TVector<TValue> valuesEncoded;
        valuesEncoded.emplace_back(THgram::Ugram(std::move(buckets)));

        NYasm::NInterfaces::NInternal::TValueSeries valueSeriesProto;
        TValueSeriesSerializer::Serialize(valuesEncoded, &valueSeriesProto);

        TVector<TValue> decodedValues(TValueSeriesDeserializer::Deserialize(valueSeriesProto));
        UNIT_ASSERT_VALUES_EQUAL(decodedValues.size(), valuesEncoded.size());

        TUgramOnlyVisitor ugramVisitor;
        decodedValues.back().Update(ugramVisitor);
    }

    Y_UNIT_TEST(NoneSeriesSerialization) {
        NYasm::NInterfaces::NInternal::TValueSeries valueSeriesProto;
        auto valueSeries = DoublesToValueSeries({0, 0, 0, 0, 0, 0});

        TValueSeriesSerializer::Serialize(valueSeries, &valueSeriesProto);
        UNIT_ASSERT_VALUES_EQUAL(valueSeries, TValueSeriesDeserializer::Deserialize(valueSeriesProto));
    }
}
