#include "store.h"

using namespace NZoom::NSubscription;
using namespace NZoom::NValue;
using namespace NYasm::NCommon;

static const constexpr TDuration SUBSCRIPTION_TTL = TDuration::Seconds(60);

bool TSubscriptionValues::PushValue(TInstant timestamp, TValue value) {
    TLightWriteGuard guard(RWLock);

    Messages.clear(); // messages are not supposed to be used with PushValues
    Messages.shrink_to_fit();

    auto currentTimestamp = TimestampToInternal(timestamp);

    if (!Values.empty()) {
        TInternalTimestamp lastElementTimestamp = StartTimestamp + Values.size() - 1;
        if (currentTimestamp <= lastElementTimestamp) {
            return false; // pushed value is older than we have already
        }
        size_t gapSize = currentTimestamp - lastElementTimestamp - 1;
        while (gapSize != 0) {
            // fill gap with None values
            Values.emplace_back();
            --gapSize;
        }
    } else {
        StartTimestamp = currentTimestamp;
    }

    Values.push_back(std::move(value));
    CleanOldValues();
    return true;
}

bool TSubscriptionValues::SetValues(TValueSeries valueSeries, TVector<TString> messages) {
    TLightWriteGuard guard(RWLock);

    Messages = std::move(messages);
    Messages.shrink_to_fit();

    // we should not empty our series because other code can rely on it being non-empty after IsEmpty returned true
    if (!valueSeries.Values.empty()) {
        StartTimestamp = TimestampToInternal(valueSeries.FirstValueTimestamp);
        Values.clear();
        for (auto& value: valueSeries.Values) {
            Values.push_back(std::move(value));
        }
        CleanOldValues();
        return true;
    } else {
        return false;
    }
}

void TSubscriptionValues::CleanOldValues() {
    // Need to keep list not bigger than max_length
    while (!Values.empty() && (Values.size() > SERIES_MAX_LENGTH)) {
        Values.pop_front();
        ++StartTimestamp;
    }
    if (Values.empty()) {
        StartTimestamp = 0;
    }
}

void TSubscriptionState::Prolong(TInstant now) {
    AtomicSet(Deadline, (now + SUBSCRIPTION_TTL).GetValue());
}

bool TPurifyingStore::Add(const TSubscription& subscription, TInstant now) {
    // check if subscription itype is blocked
    const auto& subscriptionItype = subscription.GetRequestKey().GetItype();
    if (FastConfig && FastConfig->Has(NYasm::NCommon::EFastConfigFlag::BAN_ITYPE_SUBSCRIPTIONS)) {
        for (const auto& value: FastConfig->Get(NYasm::NCommon::EFastConfigFlag::BAN_ITYPE_SUBSCRIPTIONS)) {
            if (subscriptionItype == value) {
                return false;
            }
        }
    }
    // try to prolong an existing entry without write-locking the entire structure
    if (ProlongSubscriptionState(subscription, now)) {
        return false;
    } else {
        TLightWriteGuard guard(Mutex);
        TSubscriptionMap::insert_ctx ctx;
        auto& subs = Subscriptions[subscription.GetHostName()];
        const auto it = subs.find(subscription, ctx);
        if (it != subs.end()) {
            it->second.Prolong(now);
            return false;
        }
        Revision++;
        subs.emplace_direct(ctx, subscription, now);
        return true;
    }
}

TVersionedSubscriptions TPurifyingStore::List(const TVector<THostName>& hostNames, TInstant now) {
    if (AtomicSwap(&NeedCleanup, 0) == 1) {
        Cleanup(now);
    }

    TLightReadGuard guard(Mutex);
    TVector<TSubscriptionWithTime> result;

    for (const auto& hostName: hostNames) {
        const auto it = Subscriptions.find(hostName);
        if (it != Subscriptions.end()) {
            FillHostSubscriptions(it->second, now, result);
        }
    }

    return TVersionedSubscriptions{
        .Revision = Revision,
        .Subscriptions = std::move(result)
    };
}

TVersionedSubscriptions TPurifyingStore::ListAll(TInstant now) {
    if (AtomicSwap(&NeedCleanup, 0) == 1) {
        Cleanup(now);
    }

    TLightReadGuard guard(Mutex);
    TVector<TSubscriptionWithTime> result;
    for (const auto& subs: Subscriptions) {
        FillHostSubscriptions(subs.second, now, result);
    }
    return TVersionedSubscriptions{
        .Revision = Revision,
        .Subscriptions = std::move(result)
    };
}

TVersionedSubscriptions TPurifyingStore::ListAllNoGroups(TInstant now) {
    if (AtomicSwap(&NeedCleanup, 0) == 1) {
        Cleanup(now);
    }

    TLightReadGuard guard(Mutex);
    TVector<TSubscriptionWithTime> result;
    for (const auto& subs: Subscriptions) {
        if (!subs.first.IsGroup()) {
            FillHostSubscriptions(subs.second, now, result);
        }
    }
    return TVersionedSubscriptions{
        .Revision = Revision,
        .Subscriptions = std::move(result)
    };
};

TVector<THostName> TPurifyingStore::ListHosts() const {
    TLightReadGuard guard(Mutex);
    TVector<THostName> result;
    result.reserve(Subscriptions.size());
    for (const auto& subs: Subscriptions) {
        result.emplace_back(subs.first);
    }
    return result;
}

void TPurifyingStore::Cleanup(TInstant now) {
    THashMap<THostName, TVector<TSubscription>> markedSubscriptions;

    {
        TLightReadGuard guard(Mutex);
        for (const auto& hostSubscriptions: Subscriptions) {
            auto& subs(markedSubscriptions[hostSubscriptions.first]);
            for (const auto& subscriptionState: hostSubscriptions.second) {
                if (subscriptionState.second.GetDeadline() <= now) {
                    subs.emplace_back(subscriptionState.first);
                }
            }
        }
    }

    if (markedSubscriptions.empty()) {
        return;
    }

    TLightWriteGuard guard(Mutex);
    for (const auto& hostSubscriptions: markedSubscriptions) {
        auto subsIt = Subscriptions.find(hostSubscriptions.first);

        // host could be already removed here since we unlocked Mutex for a short
        // period of time
        if (subsIt != Subscriptions.end()) {
            auto& subs = subsIt->second;
            for (const auto& subscription: hostSubscriptions.second) {
                // subscription can be already removed here too
                const auto it = subs.find(subscription);
                if (it != subs.end() && it->second.GetDeadline() <= now) {
                    subs.erase(it);
                    Revision++;
                }
            }
            if (subs.empty()) {
                Subscriptions.erase(subsIt);
            }
        }
    }
}

TDuration TPurifyingStore::GetLifetime() const {
    return SUBSCRIPTION_TTL;
}

void TPurifyingStore::Flush() {
    TLightWriteGuard guard(Mutex);
    Revision = 0;
    Subscriptions.clear();
}

size_t TPurifyingStore::GetRevision() const {
    TLightReadGuard guard(Mutex);
    return Revision;
}

bool TPurifyingStore::ProlongSubscriptionState(const TSubscription& subscription, TInstant now) {
    TLightReadGuard guard(Mutex);
    const auto subsIt = Subscriptions.find(subscription.GetHostName());
    if (subsIt != Subscriptions.end()) {
        auto& subs = subsIt->second;
        const auto it = subs.find(subscription);
        if (it != subs.end()) {
            it->second.Prolong(now);
            return true;
        }
    }
    return false;
}

void TPurifyingStore::FillHostSubscriptions(const TSubscriptionMap& subs, TInstant now,
                                            TVector<TSubscriptionWithTime>& result)
{
    result.reserve(subs.size());
    bool hasOutdatedState = false;
    for (const auto& subscriptionState: subs) {
        if (subscriptionState.second.GetDeadline() <= now) {
            hasOutdatedState = true;
        }
        result.emplace_back(TSubscriptionWithTime{subscriptionState.first, subscriptionState.second.GetCreationTime()});
    }
    if (hasOutdatedState) {
        AtomicSet(NeedCleanup, 1);
    }
}

size_t TPurifyingStore::PushValues(TVector<TSubscriptionWithValue> values, TInstant timestamp) {
    TLightReadGuard guard(Mutex); // read guard is enough since we are not modifying the structure of the store
    size_t result = 0;
    for (auto& subscriptionWithValue: values) {
        const auto hostIt = Subscriptions.find(subscriptionWithValue.Subscription.GetHostName());
        if (hostIt != Subscriptions.end()) {
            const auto subsIt = hostIt->second.find(subscriptionWithValue.Subscription);
            if (subsIt != hostIt->second.end()) {
                if (subsIt->second.PushValue(timestamp, std::move(subscriptionWithValue.Value))) {
                    ++result;
                }
            }
        }
    }
    return result;
}

void TPurifyingStore::VisitSubscriptionValues(const TSubscriptionWithRawKey& sub, IValueSeriesVisitor& visitor) const {
    TVector<TSubscriptionWithRawKey> subs{sub};
    VisitSubscriptionValues(subs, visitor);
}

void TPurifyingStore::VisitSubscriptionValues(const TVector<TSubscriptionWithRawKey>& subs, IValueSeriesVisitor& visitor) const  {
    TLightReadGuard guard(Mutex);
    // It is safe to store raw pointers while we are holding the lock.
    TVector<std::pair<const TSubscriptionWithRawKey*, const TSubscriptionState*>> toVisit;
    toVisit.reserve(subs.size());
    for (const auto& subscriptionWithRawKey: subs) {
        const auto hostIt = Subscriptions.find(subscriptionWithRawKey.Subscription.GetHostName());
        if (hostIt != Subscriptions.end()) {
            const auto subsIt = hostIt->second.find(subscriptionWithRawKey.Subscription);
            if (subsIt != hostIt->second.end()) {
                if (!subsIt->second.HasNoValues()) {
                    toVisit.emplace_back(&subscriptionWithRawKey, &subsIt->second);
                }
            }
        }
    }
    visitor.OnSize(toVisit.size());
    for (const auto& subAndState: toVisit) {
        subAndState.second->ViewValues([&visitor, &subAndState](auto timestamp, const auto& values, const auto& messages) {
            visitor.OnSubscriptionValues(*subAndState.first, timestamp, values, messages);
        });
    }
}

size_t TPurifyingStore::SetValues(TVector<TSubscriptionWithValueSeries> valueSeries) {
    TLightReadGuard guard(Mutex); // read guard is enough since we are not modifying the structure of the store
    size_t setForSubsCount = 0;
    const TVector<TString> emptyHostMessages;
    for (auto& subscriptionWithValueSeries: valueSeries) {
        const auto hostIt = Subscriptions.find(subscriptionWithValueSeries.Subscription.GetHostName());
        if (hostIt != Subscriptions.end()) {
            auto& messages = subscriptionWithValueSeries.Messages;

            const auto subsIt = hostIt->second.find(subscriptionWithValueSeries.Subscription);
            if (subsIt != hostIt->second.end()) {
                if (subsIt->second.SetValues(std::move(subscriptionWithValueSeries.ValueSeries), std::move(messages))) {
                    ++setForSubsCount;
                }
            }
        }
    }
    return setForSubsCount;
}

void TPurifyingStore::SetFastConfig(NYasm::NCommon::TFastConfig* fastConfig) {
    FastConfig = fastConfig;
}

template <>
void Out<TVersionedSubscriptions>(IOutputStream& stream, TTypeTraits<TVersionedSubscriptions>::TFuncParam result) {
    stream << "TVersionedSubscriptions{Revision=" << result.Revision << ", "
           << "Subscriptions=[";
    for (auto it = result.Subscriptions.begin(); it != result.Subscriptions.end(); ++it) {
        if (it != result.Subscriptions.begin()) {
            stream << ", ";
        }
        stream << "TSubscriptionWithTime{Subscription=" << it->Subscription << ", "
               << "Timestamp=" << it->Timestamp << "}";
    }
    stream << "]}";
}
