#pragma once

#include <infra/yasm/common/labels/host/host.h>
#include <infra/yasm/zoom/components/subscription/subscription.h>
#include <infra/yasm/common/points/value/types.h>
#include <infra/yasm/common/config/fast_config.h>

#include <util/datetime/base.h>
#include <util/generic/hash.h>
#include <util/generic/list.h>
#include <util/generic/noncopyable.h>

namespace NZoom::NSubscription {
    using NZoom::NHost::THostName;

    class TSubscriptionValues: TNonCopyable {
    public:
        static constexpr size_t TIMESTAMP_SECONDS_GRANULARITY = 5;
        static constexpr size_t SERIES_MAX_LENGTH = 10;
        using TValuesContainer = TList<NZoom::NValue::TValue>;

        TSubscriptionValues()
            : Values()
            , StartTimestamp(0)
            , Messages()
            , RWLock() {
        }

        bool PushValue(TInstant timestamp, NZoom::NValue::TValue value);
        bool SetValues(NZoom::NSubscription::TValueSeries valueSeries, TVector<TString> messages);

        // Checks whether there are some values or not. Once there are, it will never become empty.
        bool IsEmpty() const {
            TLightReadGuard guard(RWLock);
            return Values.empty();
        }

        // View values through visitor. Do not lock any locks inside of the visitor to avoid deadlocks.
        template <typename Visitor>
        void ViewValues(Visitor visitor) const {
            TLightReadGuard guard(RWLock);
            visitor(InternalTimestampToNormal(StartTimestamp), Values, Messages);
        }

    private:
        using TInternalTimestamp = size_t; // Internal timestamps have TIMESTAMP_SECONDS_GRANULARITY resolution
        static TInternalTimestamp TimestampToInternal(TInstant timestamp) {
            return timestamp.Seconds() / TIMESTAMP_SECONDS_GRANULARITY;
        }
        static TInstant InternalTimestampToNormal(TInternalTimestamp timestamp) {
            return TInstant::Seconds(timestamp * TIMESTAMP_SECONDS_GRANULARITY);
        }

        void CleanOldValues();

        TValuesContainer Values;
        TInternalTimestamp StartTimestamp;
        TVector<TString> Messages;
        TLightRWLock RWLock;
    };

    class TSubscriptionState {
    public:
        TSubscriptionState(TInstant now)
            : Deadline(0)
            , CreationTime(now)
            , LastValues()
        {
            Prolong(now);
        }

        TInstant GetDeadline() const {
            return TInstant::FromValue(AtomicGet(Deadline));
        }

        TInstant GetCreationTime() const {
            return CreationTime;
        }

        void Prolong(TInstant now);

        bool PushValue(TInstant timestamp, NZoom::NValue::TValue value) {
            return LastValues.PushValue(timestamp, std::move(value));
        }
        bool SetValues(NZoom::NSubscription::TValueSeries valueSeries, TVector<TString>&& messages) {
            return LastValues.SetValues(std::move(valueSeries), std::move(messages));
        }
        bool HasNoValues() const {
            return LastValues.IsEmpty();
        }

        // View values through visitor. Do not lock any locks inside of the visitor to avoid deadlocks.
        template <typename Visitor>
        void ViewValues(Visitor visitor) const {
            LastValues.ViewValues(visitor);
        }

    private:
        TAtomic Deadline;
        const TInstant CreationTime;
        TSubscriptionValues LastValues;
    };

    struct TSubscriptionWithTime {
        TSubscription Subscription;
        TInstant Timestamp;

        bool operator==(const TSubscriptionWithTime& other) const noexcept {
            return Timestamp == other.Timestamp && Subscription == other.Subscription;
        }
    };

    struct TSubscriptionWithValue: public TMoveOnly {
        TSubscription Subscription;
        NZoom::NValue::TValue Value;

        bool operator==(const TSubscriptionWithValue& other) const noexcept {
            return Subscription == other.Subscription && Value == other.Value;
        }
    };

    struct TVersionedSubscriptions {
        size_t Revision;
        TVector<TSubscriptionWithTime> Subscriptions;

        bool operator==(const TVersionedSubscriptions& other) const noexcept {
            return Revision == other.Revision && Subscriptions == other.Subscriptions;
        }
    };

    struct TSubscriptionWithRawKey {
        TSubscription Subscription;
        TString RawKey;
    };

    class IValueSeriesVisitor {
    public:
        virtual ~IValueSeriesVisitor() = default;

        virtual void OnSubscriptionValues(
            const TSubscriptionWithRawKey& subscription, TInstant startTs,
            const TSubscriptionValues::TValuesContainer& values, const TVector<TString>& messages) = 0;

        virtual void OnSize(size_t) {
        }
    };

    class TPurifyingStore {
    public:
        TPurifyingStore()
            : Revision(0)
            , NeedCleanup(0)
            , FastConfig(nullptr)
        {
        }

        bool Add(const TSubscription& subscription, TInstant now);
        TVersionedSubscriptions List(const TVector<THostName>& hostNames, TInstant now);
        TVersionedSubscriptions ListAll(TInstant now);
        TVersionedSubscriptions ListAllNoGroups(TInstant now);
        TVector<THostName> ListHosts() const;
        void SetFastConfig(NYasm::NCommon::TFastConfig* fastConfig);

        // Push values to subscriptions. Moves values, so the vector is not const&
        size_t PushValues(TVector<TSubscriptionWithValue> values, TInstant timestamp);
        // Visit subscription states with values. Do not lock any locks inside of the visitor methods to avoid deadlocks.
        void VisitSubscriptionValues(const TSubscriptionWithRawKey& sub, IValueSeriesVisitor& visitor) const;
        void VisitSubscriptionValues(const TVector<TSubscriptionWithRawKey>& subs, IValueSeriesVisitor& visitor) const;
        // Set value series to subscriptions. Moves values, so the vector is not const&
        size_t SetValues(TVector<NZoom::NSubscription::TSubscriptionWithValueSeries> values);

        void Cleanup(TInstant now);
        TDuration GetLifetime() const;
        void Flush();

        size_t GetRevision() const;

    private:
        using TSubscriptionMap = THashMap<TSubscription, TSubscriptionState>;
        using THostSubscriptionMap = THashMap<THostName, TSubscriptionMap>;

        bool ProlongSubscriptionState(const TSubscription& subscription, TInstant now);
        void FillHostSubscriptions(const TSubscriptionMap& subs, TInstant now, TVector<TSubscriptionWithTime>& result);

        size_t Revision;
        THostSubscriptionMap Subscriptions;
        TLightRWLock Mutex;
        TAtomic NeedCleanup;
        NYasm::NCommon::TFastConfig* FastConfig;
    };
}
