#include "store.h"

#include <infra/yasm/common/config/fast_config_ut.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/tempfile.h>

using namespace NZoom::NSubscription;
using namespace NYasm::NCommon;
using NZoom::NSignal::TSignalName;
using NZoom::NValue::TValue;

namespace {

    // Converts vector of doubles to container of TValue. Treats 0 as None.
    template <typename Container>
    Container DoublesToValues(TVector<double> doubles) {
        Container result;
        for (auto curDouble: doubles) {
            if (0 != curDouble) {
                result.emplace_back(curDouble);
            } else {
                result.emplace_back();
            }
        }
        return result;
    }

    struct TSubscriptionValuesCopyVisitor: public IValueSeriesVisitor, TNonCopyable {
        TVector<TSubscriptionWithValueSeries> CopiedValues;
        TVector<TString> CopiedRawKeys;
        TVector<TVector<TString>> CopiedMessages;

        size_t CopiedSize;

        TSubscriptionValuesCopyVisitor()
            : CopiedValues()
            , CopiedSize(0) {
        }

        void Clear() {
            CopiedValues.clear();
            CopiedRawKeys.clear();
            CopiedMessages.clear();
            CopiedSize = 0;
        }

        void OnSize(size_t size) final {
            CopiedSize = size;
        }

        void OnSubscriptionValues(const TSubscriptionWithRawKey& subscriptionWithRawKey, TInstant timestamp,
            const TSubscriptionValues::TValuesContainer& values, const TVector<TString>& messages) final {
            TVector<TValue> valuesCopy;
            for (const auto& value: values) {
                valuesCopy.push_back(TValue(value.GetValue()));
            }
            CopiedValues.emplace_back(subscriptionWithRawKey.Subscription, TValueSeries(timestamp, std::move(valuesCopy)));
            CopiedRawKeys.push_back(subscriptionWithRawKey.RawKey);
            CopiedMessages.push_back(messages);
        }
    };
}

Y_UNIT_TEST_SUITE(TestSubscriptionStore) {
    Y_UNIT_TEST(TestCleanup) {
        const TInstant now(TInstant::Now());
        const THostName hostName(TStringBuf("SAS.000"));
        const TSubscription subscription(
            hostName,
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test_summ")
        );

        TPurifyingStore store;
        UNIT_ASSERT_VALUES_EQUAL(store.ListHosts(), TVector<THostName>{});

        store.Add(subscription, now);
        UNIT_ASSERT_VALUES_EQUAL(store.ListHosts(), TVector<THostName>{hostName});
        {
            TVersionedSubscriptions expected{
                .Revision = 1,
                .Subscriptions = {{subscription, now}}
            };
            UNIT_ASSERT_VALUES_EQUAL(store.List({hostName}, now), expected);
            UNIT_ASSERT_VALUES_EQUAL(store.List({hostName}, now + store.GetLifetime()), expected);
        }

        {
            TVersionedSubscriptions expected{
                .Revision = 2,
                .Subscriptions = {}
            };
            UNIT_ASSERT_VALUES_EQUAL(store.List({hostName}, now + store.GetLifetime()), expected);
            UNIT_ASSERT_VALUES_EQUAL(store.List({hostName}, now), expected);
        }
        UNIT_ASSERT_VALUES_EQUAL(store.ListHosts(), TVector<THostName>{});
    }

    Y_UNIT_TEST(TestCleanupTwoHosts) {
        const auto subscriptionCmp = [](const TSubscriptionWithTime& left, const TSubscriptionWithTime& right) {
            return left.Subscription.GetHostName().GetName() < right.Subscription.GetHostName().GetName();
        };
        const THostName hostName1(TStringBuf("SAS.001"));
        const THostName hostName2(TStringBuf("SAS.002"));
        const TSubscription subscription1(
                hostName1,
                TInternedRequestKey(TStringBuf("itype=base")),
                TStringBuf("test_summ")
        );
        const TSubscription subscription2(
                hostName2,
                TInternedRequestKey(TStringBuf("itype=base")),
                TStringBuf("test_summ")
        );

        TPurifyingStore store;
        UNIT_ASSERT_VALUES_EQUAL(store.ListHosts(), TVector<THostName>{});

        const auto addFirstTime = TInstant::Now();
        store.Add(subscription1, addFirstTime);
        UNIT_ASSERT_VALUES_EQUAL(store.ListHosts(), TVector<THostName>{hostName1});

        const auto smallTimeShift = TDuration::Seconds(5);
        const auto addSecondTime = addFirstTime + smallTimeShift;
        store.Add(subscription2, addSecondTime);
        auto hostsActual = store.ListHosts();
        THashSet<THostName> twoHostsExpected = {hostName1, hostName2};
        UNIT_ASSERT_VALUES_EQUAL(THashSet<THostName>(hostsActual.begin(), hostsActual.end()), twoHostsExpected);

        {
            TVersionedSubscriptions expected = {
                .Revision = 2,
                .Subscriptions = {{subscription1, addFirstTime}, {subscription2, addSecondTime}}
            };
            auto actual = store.ListAll(addSecondTime);
            auto repeated = store.ListAll(addSecondTime);
            Sort(actual.Subscriptions, subscriptionCmp);
            Sort(repeated.Subscriptions, subscriptionCmp);
            UNIT_ASSERT_VALUES_EQUAL(actual, expected);
            UNIT_ASSERT_VALUES_EQUAL(repeated, expected);
        }

        {
            TVersionedSubscriptions expected = {
                .Revision = 3,
                .Subscriptions = {{subscription2, addSecondTime}}
            };
            store.ListAll(addFirstTime + store.GetLifetime());
            auto repeated = store.ListAll(addFirstTime + store.GetLifetime());
            Sort(repeated.Subscriptions, subscriptionCmp);
            UNIT_ASSERT_VALUES_EQUAL(repeated, expected);
        }

        {
            TVersionedSubscriptions expected = {
                .Revision = 4,
                .Subscriptions = {}
            };
            store.ListAll(addSecondTime + store.GetLifetime());
            auto repeated = store.ListAll(addSecondTime + store.GetLifetime());
            UNIT_ASSERT_VALUES_EQUAL(repeated, expected);
        }
    }

    Y_UNIT_TEST(TestProlong) {
        TInstant now(TInstant::Now());
        const THostName hostName(TStringBuf("SAS.000"));
        const TSubscription subscription(
            hostName,
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test_summ")
        );

        TPurifyingStore store;
        UNIT_ASSERT(store.Add(subscription, now));
        UNIT_ASSERT(!store.Add(subscription, now + TDuration::Seconds(1)));
        now += store.GetLifetime();
        store.Cleanup(now);
        UNIT_ASSERT(!store.List({hostName}, now).Subscriptions.empty());

        now += TDuration::Seconds(1);
        store.Cleanup(now);
        UNIT_ASSERT(store.List({hostName}, now).Subscriptions.empty());
        UNIT_ASSERT(store.ListAll(now).Subscriptions.empty());
        UNIT_ASSERT(store.Add(subscription, now));
    }

    Y_UNIT_TEST(TestNotFound) {
        TInstant now(TInstant::Now());
        THostName hostName(TStringBuf("SAS.000"));
        TPurifyingStore store;
        TVersionedSubscriptions expected{
            .Revision = 0,
            .Subscriptions = {}
        };
        UNIT_ASSERT_VALUES_EQUAL(store.List({hostName}, now), expected);
        UNIT_ASSERT_VALUES_EQUAL(store.ListAll(now), expected);
    }

    Y_UNIT_TEST(TestPointsVisitors) {
        const THostName hostName1(TStringBuf("SAS.001"));
        const THostName hostName2(TStringBuf("SAS.002"));
        const TSubscription subscription1(
            hostName1,
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test1_summ")
        );
        const TSubscription subscription2(
            hostName2,
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test2_summ")
        );
        const TSubscription subscriptionNotInStore(
            TStringBuf("SAS.003"),
            TInternedRequestKey(TStringBuf("itype=common")),
            TStringBuf("test3_summ")
        );
        TPurifyingStore store;

        auto startTime = TInstant::Zero();
        UNIT_ASSERT(store.Add(subscription1, startTime));
        UNIT_ASSERT(store.Add(subscription2, startTime));

        TVector<TSubscriptionWithValue> values;
        values.push_back(TSubscriptionWithValue{
            .Subscription = subscription1,
            .Value = TValue(1)
        });
        values.push_back(TSubscriptionWithValue{
            .Subscription = subscription2,
            .Value = TValue(2)
        });
        values.push_back(TSubscriptionWithValue{
            .Subscription = subscriptionNotInStore,
            .Value = TValue(3)
        });
        UNIT_ASSERT(store.PushValues(std::move(values), startTime + TDuration::Seconds(10)));

        TVector<TSubscriptionWithRawKey> subscriptionsToVisit = {
            TSubscriptionWithRawKey{
                .Subscription = subscription1,
                .RawKey = "key1"
            },
            TSubscriptionWithRawKey{
                .Subscription = subscriptionNotInStore,
                .RawKey = "key2"
            },
            TSubscriptionWithRawKey{
                .Subscription = subscription1,
                .RawKey = "key3"
            },
            TSubscriptionWithRawKey{
                .Subscription = subscription2,
                .RawKey = "key4"
            },
            TSubscriptionWithRawKey{
                .Subscription = subscription2,
                .RawKey = "key5"
            },
            TSubscriptionWithRawKey{
                .Subscription = subscriptionNotInStore,
                .RawKey = "key6"
            }
        };
        TVector<TString> expectedRawKeys = {"key1", "key3", "key4", "key5"};
        TVector<TSubscriptionWithValueSeries> expectedResponse;
        expectedResponse.emplace_back(subscription1, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({1})));
        expectedResponse.emplace_back(subscription1, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({1})));
        expectedResponse.emplace_back(subscription2, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({2})));
        expectedResponse.emplace_back(subscription2, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({2})));
        TSubscriptionValuesCopyVisitor actualResponseVisitor;
        store.VisitSubscriptionValues(subscriptionsToVisit, actualResponseVisitor);
        UNIT_ASSERT_EQUAL(expectedResponse, actualResponseVisitor.CopiedValues);
        UNIT_ASSERT_EQUAL(expectedRawKeys, actualResponseVisitor.CopiedRawKeys);
        for (const auto& subMessages: actualResponseVisitor.CopiedMessages) {
            UNIT_ASSERT(subMessages.empty());
        }
    }

    Y_UNIT_TEST(TestMessages) {
        const THostName hostName1(TStringBuf("META1"));
        const THostName hostName2(TStringBuf("META2"));
        const THostName hostName3(TStringBuf("META3"));
        const TSubscription subscription1(
            hostName1,
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test1_summ")
        );
        const TSubscription subscription2(
            hostName2,
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test2_summ")
        );
        const TSubscription subscription3(
            hostName3,
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test3_summ")
        );
        const TSubscription subscription4(
            hostName3,
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test4_summ")
        );

        TVector<TString> messages1 = {"MSG1", "MSG2"};
        TVector<TString> messages2 = {"MSG3", "MSG4"};

        TPurifyingStore store;
        auto startTime = TInstant::Zero();
        UNIT_ASSERT(store.Add(subscription1, startTime));
        UNIT_ASSERT(store.Add(subscription2, startTime));
        UNIT_ASSERT(store.Add(subscription3, startTime));
        UNIT_ASSERT(store.Add(subscription4, startTime));

        TVector<TSubscriptionWithValueSeries> values;
        values.emplace_back(subscription1, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({1})), messages1);
        values.emplace_back(subscription2, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({1})));
        values.emplace_back(subscription3, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({1})), messages2);
        values.emplace_back(subscription4, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({1})), messages2);

        store.SetValues(std::move(values));
        TSubscriptionValuesCopyVisitor copyVisitor;

        TVector<TSubscriptionWithRawKey> subscriptionsToVisit = {
            TSubscriptionWithRawKey{
                .Subscription = subscription1,
            },
            TSubscriptionWithRawKey{
                .Subscription = subscription2,
            },
            TSubscriptionWithRawKey{
                .Subscription = subscription3,
            },
            TSubscriptionWithRawKey{
                .Subscription = subscription4,
            },
        };
        store.VisitSubscriptionValues(subscriptionsToVisit, copyVisitor);
        TVector<TVector<TString>> expectedMessages = {
            {TString("MSG1"), TString("MSG2")},
            {},
            {TString("MSG3"), TString("MSG4")},
            {TString("MSG3"), TString("MSG4")}
        };
        UNIT_ASSERT_VALUES_EQUAL(expectedMessages, copyVisitor.CopiedMessages);

        TVector<TSubscriptionWithValueSeries> values2;
        values2.emplace_back(subscription1, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({1})), messages1);
        values2.emplace_back(subscription2, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({1})));
        values2.emplace_back(subscription3, TValueSeries(startTime + TDuration::Seconds(10),
            DoublesToValues<TVector<TValue>>({1})));
        copyVisitor.Clear();
        store.SetValues(std::move(values2));
        store.VisitSubscriptionValues(subscriptionsToVisit, copyVisitor);
        TVector<TVector<TString>> expectedMessages2 = {
            {TString("MSG1"), TString("MSG2")},
            {},
            {},
            {TString("MSG3"), TString("MSG4")}
        };
        UNIT_ASSERT_VALUES_EQUAL(expectedMessages2, copyVisitor.CopiedMessages);
    }

    Y_UNIT_TEST(TestAddBannedItype) {
        TLog log {};
        TString localFileName {"fast_config.json"};
        TTempFile tempFile {localFileName};
        TFastConfigSettings settings {
            "oauth_token",
            "locke",
            "ytPath",
            localFileName,
            NTags::TInstanceKey::FromNamed("itype|ctype=prod")
        };
        TPurifyingStore store;
        NTest::TYtMockedFastConfig fastConfig(log, settings);
        fastConfig.Start();
        store.SetFastConfig(&fastConfig);

        const THostName hostName(TStringBuf("META1"));
        TInstant now(TInstant::Now());

        const TSubscription subscription(
            hostName,
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test1_summ")
        );

        const TSubscription subscriptionWithBannedItype(
            hostName,
            TInternedRequestKey(TStringBuf("itype=banneditype")),
            TStringBuf("test1_summ")
        );

        UNIT_ASSERT(store.Add(subscription, now));
        UNIT_ASSERT(!store.Add(subscriptionWithBannedItype, now));
        fastConfig.Stop();
    }

    Y_UNIT_TEST(TestListAllHosts) {
        const THostName hostName1(TStringBuf("SAS.001"));
        const THostName hostName2(TStringBuf("bylcf2kcwkknnhq6.sas.yp-c.yandex.net"));
        const TSubscription subscription1(
                hostName1,
                TInternedRequestKey(TStringBuf("itype=base")),
                TStringBuf("test_summ")
        );
        const TSubscription subscription2(
                hostName2,
                TInternedRequestKey(TStringBuf("itype=base")),
                TStringBuf("test_summ")
        );
        const auto addTime = TInstant::Now();
        TPurifyingStore store;
        TVersionedSubscriptions expectedEmptySubscriptions = {
            .Revision = 0,
            .Subscriptions = {}
        };
        UNIT_ASSERT_VALUES_EQUAL(store.ListAllNoGroups(addTime), expectedEmptySubscriptions);
        store.Add(subscription1, addTime);
        store.Add(subscription2, addTime);
        TVersionedSubscriptions expectedHostSubscriptions = {
            .Revision = 2,
            .Subscriptions = {{subscription2, addTime}}
        };
        UNIT_ASSERT_VALUES_EQUAL(store.ListAllNoGroups(addTime), expectedHostSubscriptions);
    }
}

Y_UNIT_TEST_SUITE(TestSubscriptionValues) {
    Y_UNIT_TEST(TestPushWithGap) {
        auto startTimestamp = TInstant::Zero();
        TSubscriptionValues valueSeries;

        UNIT_ASSERT(valueSeries.PushValue(startTimestamp, TValue(1)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(5), TValue(2)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(20), TValue(3)));
        valueSeries.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto&) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp);
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({1, 2, 0, 0, 3}), values);
        });
    }

    Y_UNIT_TEST(TestPushExpiredValue) {
        auto preStartTimestamp = TInstant::Zero();
        auto startTimestamp = preStartTimestamp + TDuration::Seconds(20);
        TSubscriptionValues valueSeries;

        UNIT_ASSERT(valueSeries.PushValue(startTimestamp, TValue(10)));
        UNIT_ASSERT(!valueSeries.PushValue(preStartTimestamp, TValue(1)));
        valueSeries.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto&) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp);
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({10}), values);
        });
    }

    Y_UNIT_TEST(TestPushNone) {
        auto startTimestamp = TInstant::Zero() + TDuration::Seconds(20);
        TSubscriptionValues valueSeries;

        UNIT_ASSERT(valueSeries.PushValue(startTimestamp, TValue()));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(10), TValue(2)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(20), TValue(4)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(45), TValue(9)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(50), TValue()));


        valueSeries.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto&) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp + TDuration::Seconds(5));
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({0, 2, 0, 4, 0, 0, 0, 0, 9, 0}), values);
        });
    }

    Y_UNIT_TEST(TestCleanup) {
        auto startTimestamp = TInstant::Zero();
        TSubscriptionValues valueSeries;

        UNIT_ASSERT(valueSeries.PushValue(startTimestamp, TValue(1)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(5), TValue(2)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(10), TValue(3)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(15), TValue(4)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(20), TValue(5)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(25), TValue(6)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(30), TValue(7)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(35), TValue(8)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(40), TValue(9)));
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(45), TValue(10)));

        valueSeries.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto&) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp);
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({1, 2, 3, 4, 5, 6, 7, 8, 9, 10}), values);
        });
        // check that cleanup is performed on PushValue
        UNIT_ASSERT(valueSeries.PushValue(startTimestamp + TDuration::Seconds(80), TValue(11)));
        valueSeries.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto&) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp + TDuration::Seconds(35));
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({8, 9, 10, 0, 0, 0, 0, 0, 0, 11}), values);
        });
    }

    Y_UNIT_TEST(TestSetValues) {
        auto startTimestamp = TInstant::Zero();
        TSubscriptionValues subscriptionValues;

        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp, TValue(1)));
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(5), TValue(2)));
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(10), TValue(3)));
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(15), TValue(4)));
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(20), TValue(5)));

        subscriptionValues.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto&) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp);
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({1, 2, 3, 4, 5}), values);
        });

        TValueSeries valueSeries(startTimestamp + TDuration::Seconds(13), DoublesToValues<TVector<TValue>>({6, 7, 8}));

        UNIT_ASSERT(subscriptionValues.SetValues(std::move(valueSeries), TVector<TString>()));
        subscriptionValues.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto&) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp + TDuration::Seconds(10));
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({6, 7, 8}), values);
        });
    }

    Y_UNIT_TEST(TestSetValuesWithMessages) {
        auto startTimestamp = TInstant::Zero();
        TSubscriptionValues subscriptionValues;

        TValueSeries valueSeries(startTimestamp + TDuration::Seconds(13), DoublesToValues<TVector<TValue>>({6, 7, 8}));

        TVector<TString> messages1 = {"Err1", "Err2"};

        UNIT_ASSERT(subscriptionValues.SetValues(std::move(valueSeries), std::move(messages1)));
        subscriptionValues.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto& messages) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp + TDuration::Seconds(10));
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({6, 7, 8}), values);
            UNIT_ASSERT_EQUAL(TVector<TString>({"Err1", "Err2"}), messages);
        });

        // push should drop messages
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(25), TValue(9)));
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(30), TValue(10)));
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(35), TValue(11)));

        subscriptionValues.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto& messages) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp + TDuration::Seconds(10));
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({6, 7, 8, 9, 10, 11}), values);
            UNIT_ASSERT(messages.empty());
        });

        TVector<TString> messages2 = {"Err1", "Err2"};
        TValueSeries valueSeries2(startTimestamp + TDuration::Seconds(13), DoublesToValues<TVector<TValue>>({6, 7, 8}));
        UNIT_ASSERT(subscriptionValues.SetValues(std::move(valueSeries2), std::move(messages2)));
        subscriptionValues.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto& messages) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp + TDuration::Seconds(10));
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({6, 7, 8}), values);
            UNIT_ASSERT_EQUAL(TVector<TString>({"Err1", "Err2"}), messages);
        });

        // set with no messages should drop existing messages
        TValueSeries valueSeries3(startTimestamp + TDuration::Seconds(13), DoublesToValues<TVector<TValue>>({6, 7, 8}));
        UNIT_ASSERT(subscriptionValues.SetValues(std::move(valueSeries3), TVector<TString>()));
        subscriptionValues.ViewValues([startTimestamp](auto timestamp, const auto& values, const auto& messages) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp + TDuration::Seconds(10));
            UNIT_ASSERT_EQUAL(DoublesToValues<TList<TValue>>({6, 7, 8}), values);
            UNIT_ASSERT(messages.empty());
        });
    }

    Y_UNIT_TEST(TestSetEmptyValues) {
        auto startTimestamp = TInstant::Zero();
        TSubscriptionValues subscriptionValues;

        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp, TValue(1)));
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(5), TValue(2)));
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(10), TValue(3)));
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(15), TValue(4)));
        UNIT_ASSERT(subscriptionValues.PushValue(startTimestamp + TDuration::Seconds(20), TValue(5)));

        auto expectedValues = DoublesToValues<TList<TValue>>({1, 2, 3, 4, 5});

        subscriptionValues.ViewValues([startTimestamp, &expectedValues](auto timestamp, const auto& values, const auto&) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp);
            UNIT_ASSERT_EQUAL(expectedValues, values);
        });

        TValueSeries valueSeries(startTimestamp + TDuration::Seconds(13), TVector<TValue>());

        UNIT_ASSERT(!subscriptionValues.SetValues(std::move(valueSeries), TVector<TString>()));
        subscriptionValues.ViewValues([startTimestamp, &expectedValues](auto timestamp, const auto& values, const auto&) {
            UNIT_ASSERT_EQUAL(timestamp, startTimestamp);
            UNIT_ASSERT_EQUAL(expectedValues, values);
        });
    }
}
