#pragma once

#include <infra/yasm/common/labels/host/host.h>
#include <infra/yasm/common/labels/signal/signal_name.h>
#include <infra/yasm/common/labels/tags/request_key.h>
#include <infra/yasm/common/points/value/types.h>
#include <infra/yasm/zoom/components/subscription/request_key.h>
#include <library/cpp/consistent_hashing/consistent_hashing.h>
#include <util/datetime/base.h>
#include <util/digest/city.h>
#include <util/digest/multi.h>
#include <util/digest/numeric.h>

namespace NZoom {
    namespace NSubscription {
        // Fields used to pack/unpack TSubscription
        constexpr TStringBuf HOST_FIELD_NAME = "host";
        constexpr TStringBuf TAGS_FIELD_NAME = "tags";
        constexpr TStringBuf SIGNAL_FIELD_NAME = "signal";
        constexpr TStringBuf ALLOW_LEGACY_TYPES_FIELD_NAME = "allow_legacy_types";
        constexpr TStringBuf TAGS_REQUESTED_FIELD_NAME = "tags_requested"; // optional
        constexpr TStringBuf MESSAGES_FIELD_NAME = "messages"; // optional

        // Top level fields for packing/unpacking subscription related data.
        constexpr TStringBuf SUBSCRIPTION_LIST_FIELD_NAME = "subscriptions";
        constexpr TStringBuf ITERATION_TIMESTAMP_FIELD_NAME = "timestamp";

        // Fields used to pack/unpack TSubscription extensions
        constexpr TStringBuf TIMESTAMP_FIELD_NAME = "timestamp";
        constexpr TStringBuf VALUE_FIELD_NAME = "value";
        constexpr TStringBuf VALUE_SERIES_START_FIELD_NAME = "st";
        constexpr TStringBuf VALUE_SERIES_FIELD_NAME = "values";

        class TSubscription {
        public:
            using TSignalExpression = TString;
            static constexpr bool ALLOW_LEGACY_TYPES_DEFAULT = true;

            TSubscription(NZoom::NHost::THostName hostName, TInternedRequestKey requestKey, TSignalExpression signalExpression, bool allowLegacyTypes = ALLOW_LEGACY_TYPES_DEFAULT)
                : HostName(hostName)
                , RequestKey(requestKey)
                , SignalExpression(std::move(signalExpression))
                , AllowLegacyTypes(allowLegacyTypes)
            {
            }

            TSubscription(NZoom::NHost::THostName hostName, TInternedRequestKey requestKey, TStringBuf signalExpression, bool allowLegacyTypes = ALLOW_LEGACY_TYPES_DEFAULT)
                : HostName(hostName)
                , RequestKey(requestKey)
                , SignalExpression(signalExpression)
                , AllowLegacyTypes(allowLegacyTypes)
            {
            }

            const NZoom::NHost::THostName& GetHostName() const {
                return HostName;
            }

            const TInternedRequestKey& GetRequestKey() const {
                return RequestKey;
            }

            const TSignalExpression& GetSignalExpression() const {
                return SignalExpression;
            }

            bool GetAllowLegacyTypes() const {
                return AllowLegacyTypes;
            }

            inline bool operator==(const TSubscription& other) const noexcept {
                return HostName == other.HostName &&
                       RequestKey == other.RequestKey &&
                       SignalExpression == other.SignalExpression &&
                       AllowLegacyTypes == other.AllowLegacyTypes;
            }

            inline size_t Hash() const noexcept {
                return MultiHash(HostName, RequestKey, SignalExpression, AllowLegacyTypes);
            }

        private:
            NZoom::NHost::THostName HostName;
            TInternedRequestKey RequestKey;
            TSignalExpression SignalExpression;
            bool AllowLegacyTypes;
        };

        struct TValueSeries: public TMoveOnly {
            TInstant FirstValueTimestamp;
            TVector<NZoom::NValue::TValue> Values;

            TValueSeries()
                : FirstValueTimestamp(TInstant::Zero())
                , Values() {
            }

            TValueSeries(TInstant firstValueTimestamp, TVector<NZoom::NValue::TValue> values)
                : FirstValueTimestamp(firstValueTimestamp)
                , Values(std::move(values)) {
            }

            bool operator==(const TValueSeries& other) const noexcept {
                return Values == other.Values &&
                       FirstValueTimestamp == other.FirstValueTimestamp;
            }
        };

        struct TSubscriptionWithValueSeries: public TMoveOnly {
            TSubscription Subscription;
            TValueSeries ValueSeries;
            TVector<TString> Messages;

            TSubscriptionWithValueSeries(const TSubscription& subscription, TValueSeries valueSeries)
                : Subscription(subscription)
                , ValueSeries(std::move(valueSeries)) {
            }

            TSubscriptionWithValueSeries(const TSubscription& subscription, TValueSeries valueSeries, TVector<TString> messages)
                : Subscription(subscription)
                , ValueSeries(std::move(valueSeries))
                , Messages(std::move(messages)) {
            }

            bool operator==(const TSubscriptionWithValueSeries& other) const noexcept {
                return Subscription == other.Subscription &&
                       ValueSeries == other.ValueSeries;
            }
        };

        using TTagSignals = THashMap<TInternedRequestKey, TVector<NZoom::NSignal::TSignalName>>;
        using THostTagSignals = THashMap<NZoom::NHost::THostName, TTagSignals>;

        TTagSignals SubscriptionsToTagSignals(const TVector<TSubscription>& subscriptions);
        THostTagSignals SubscriptionsToHostTagSignals(const TVector<TSubscription>& subscriptions);

        // Subscription consistent hashing based on host and tags pair
        size_t GetSubscriptionBucket(const NHost::THostName& host, const TInternedRequestKey& tags, size_t bucketCount);
        size_t GetSubscriptionBucket(const TSubscription& subscription, size_t bucketCount);
    }
}

template<>
struct THash<NZoom::NSubscription::TSubscription> {
    inline size_t operator()(const NZoom::NSubscription::TSubscription& v) const noexcept {
        return v.Hash();
    }
};
