#include "subscription.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NSubscription;

Y_UNIT_TEST_SUITE(TZoomSubscriptionTest) {
    Y_UNIT_TEST(TestEq) {
        TSubscription s(TStringBuf("h1"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TSubscription s1(TStringBuf("h2"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TSubscription s2(TStringBuf("h1"), TStringBuf("itype=b2"), TStringBuf("s1_dmmm"));
        TSubscription s3(TStringBuf("h1"), TStringBuf("itype=b1"), TStringBuf("s2_dmmm"));
        TSubscription s4(TStringBuf("h1"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"), false);
        TSubscription s_copy(s);
        UNIT_ASSERT_EQUAL(s, s_copy);
        UNIT_ASSERT_UNEQUAL(s, s1);
        UNIT_ASSERT_UNEQUAL(s, s2);
        UNIT_ASSERT_UNEQUAL(s, s3);
        UNIT_ASSERT_UNEQUAL(s, s4);
        UNIT_ASSERT_UNEQUAL(s1, s2);
        UNIT_ASSERT_UNEQUAL(s1, s3);
        UNIT_ASSERT_UNEQUAL(s2, s3);
    }

    Y_UNIT_TEST(TestHash) {
        TSubscription s1(TStringBuf("h1"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TSubscription s2(TStringBuf("h1"), TStringBuf("itype=b1"), TString::Join("s1_", "dmmm"));
        TSubscription s3(TStringBuf("h1"), TStringBuf("itype=b1"), TString::Join("s1_", "dmmm"), false);
        UNIT_ASSERT_EQUAL(s1.Hash(), s2.Hash());
        UNIT_ASSERT_UNEQUAL(s1.Hash(), s3.Hash());
    }

    Y_UNIT_TEST(TestConsistentHash) {
        TSubscription s1(TStringBuf("h1"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TSubscription s2(TStringBuf("h1"), TStringBuf("itype=b1"), TString::Join("s1_", "dmmm"));
        for (size_t busketCount = 1; busketCount < 20; ++busketCount) {
            UNIT_ASSERT_EQUAL(GetSubscriptionBucket(s1, busketCount), GetSubscriptionBucket(s2, busketCount));
        }
    }

    Y_UNIT_TEST(TestConsistentHashDistribution) {
        TVector<TString> hosts = {"SAS.000", "SAS.001", "SAS.002", "SAS.003", "SAS.004", "SAS.005", "SAS.006", "SAS.007",
            "SAS.008", "IMGS6", "IMGS7", "IMGS8", "IMGS13", "IMGS17", "QLOUD_GOLOVAN_SAS.0", "QLOUD_GOLOVAN_SAS.1",
            "QLOUD_GOLOVAN_SAS.2", "QLOUD_GOLOVAN_SAS.11"};
        TVector<TString> tags;
        for (size_t i = 0; i < 1798; ++i) {
            TStringStream s;
            s << "itype=app" << i << ";ctype=prod" << i << ";geo=sas" << i;
            if (i % 5 == 0) {
                s << ";prj=some-prj" << i;
            }
            tags.push_back(s.Str());
        }

        TVector<TSubscription> subscriptions;
        for (size_t index = 0; index <= 123456; ++index) {
            subscriptions.emplace_back(hosts[index % hosts.size()],
                tags[index % tags.size()], TSubscription::TSignalExpression("some_signal_summ")); // signal is not used in
                                                                                                  // bucket distribution for now
        }

        for (size_t busketCount = 2; busketCount < 20; ++busketCount) {
            TVector<size_t> busketSizes(busketCount, 0);
            for (const auto& subscription: subscriptions) {
                busketSizes[GetSubscriptionBucket(subscription, busketCount)]++;
            }

            size_t averSize = subscriptions.size() / busketCount;
            auto averSizePlus = static_cast<size_t>(averSize * 1.1); // 10% is better than what we had in python implementation
            auto averSizeMinus = static_cast<size_t>(averSize * 0.9);
            for (auto busketSize: busketSizes) {
                if (!(busketSize > averSizeMinus && busketSize < averSizePlus)) {
                    TStringStream err;
                    err << "Busket size " << busketSize << " is not in (" << averSizeMinus << ", " << averSizePlus << ") range "
                        << "for busket count " << busketCount;
                    UNIT_FAIL(err.Str());
                }
            }
        }
    }

}
