#include "value_series_merger.h"

namespace NZoom::NSubscription {
    using namespace NAccumulators;

    TSubscriptionValueSeriesMerger::TSubscriptionValueSeriesMerger(size_t seriesLength, size_t resolution, ui64 startTimestamp)
        : MergedDataTable()
        , SeriesLength(seriesLength)
        , Resolution(resolution)
        , StartTimestamp(startTimestamp) {
    }

    void TSubscriptionValueSeriesMerger::MulSubscriptionsValueSeries(const TVector<TSubscriptionWithValueSeries>& subs) {
        for (const auto& subscriptionWithValues: subs) {
            auto& mergedData = FindOrInsert(subscriptionWithValues.Subscription);
            ProcessValues(subscriptionWithValues.ValueSeries, mergedData, false);
        }
    }

    void TSubscriptionValueSeriesMerger::MergeSubscriptionsValueSeries(const TVector<TSubscriptionWithValueSeries>& subs) {
        for (const auto& subscriptionWithValues: subs) {
            auto& mergedData = FindOrInsert(subscriptionWithValues.Subscription);
            ProcessValues(subscriptionWithValues.ValueSeries, mergedData, true);
        }
    }

    void TSubscriptionValueSeriesMerger::ProcessValues(const TValueSeries& valueSeries,
                                                       TSubscriptionMergedData& mergeTo,
                                                       bool merge) {
        auto& accumulators = mergeTo.Accumulators;
        if (!accumulators.Empty()) {
            const auto& values = valueSeries.Values;

            const i64 signedCurStartTimestamp = valueSeries.FirstValueTimestamp.Seconds();
            i64 offset = (signedCurStartTimestamp - StartTimestamp) / Resolution;
            if (offset >= 0) {
                if (offset >= i64(accumulators.Len())) {
                    mergeTo.SetEmptyBorders();
                    return;
                }
                mergeTo.LeftBorder = Max<size_t>(mergeTo.LeftBorder, size_t(offset));
                mergeTo.RightBorder = Min<size_t>(mergeTo.RightBorder, values.size() + offset);

            } else {
                if (-offset >= i64(values.size())) {
                    mergeTo.SetEmptyBorders();
                    return;
                }
                mergeTo.RightBorder = Min<size_t>(mergeTo.RightBorder, values.size() + offset);
            }

            const i64 leftOffset = (offset >= 0) ? offset : 0;
            const i64 rightOffset = (offset < 0) ? -offset : 0;

            auto innerIt = accumulators.begin() + leftOffset;
            auto valuesIt = values.cbegin() + rightOffset;

            while (innerIt != accumulators.end() && valuesIt != values.cend()) {
                if (!merge || innerIt->GetValue().IsDefault()) {
                    innerIt->Mul(*valuesIt);
                }
                ++innerIt;
                ++valuesIt;
            }
        }
    }

    TSubscriptionMergedData& TSubscriptionValueSeriesMerger::FindOrInsert(const TSubscription& subscription) {
        NSignal::TSignalName signalName = subscription.GetSignalExpression();
        auto& reqKeyData = MergedDataTable[subscription.GetRequestKey()];
        TSignalToMergedDataMap::insert_ctx ctx;
        auto signalDataIt = reqKeyData.find(signalName, ctx);
        if (signalDataIt == reqKeyData.end()) {
            const TAggregationRules* aggregationRulesPtr = signalName.GetAggregationRules();
            if (aggregationRulesPtr) {
                // new signal
                auto type = aggregationRulesPtr->GetAccumulatorType(EAggregationMethod::MetaGroup);
                signalDataIt = reqKeyData.emplace_direct(ctx, signalName, TSubscriptionMergedData(type, SeriesLength));
            } else {
                // old signal
                signalDataIt = reqKeyData.emplace_direct(ctx, signalName, TSubscriptionMergedData(SeriesLength));
            }
        }
        return signalDataIt->second;
    }
}
