#pragma once

#include <infra/yasm/common/points/accumulators/accumulators.h>
#include <infra/yasm/zoom/components/subscription/subscription.h>

namespace NZoom::NSubscription {

    struct TSubscriptionMergedData: public TMoveOnly {
        TSubscriptionMergedData(NAccumulators::EAccumulatorType type, size_t seriesLength)
            : Accumulators(type, seriesLength)
            , SeriesLength(seriesLength)
            , LeftBorder(0)
            , RightBorder(seriesLength) {
        }

        explicit TSubscriptionMergedData(size_t seriesLength)
            : Accumulators(NAccumulators::EAccumulatorType::Counter, 0) // type does not matter since array is empty
            , SeriesLength(seriesLength)
            , LeftBorder(0)
            , RightBorder(seriesLength) {
        }

        void SetEmptyBorders() {
            LeftBorder = SeriesLength;
            RightBorder = 0;
        }

        NAccumulators::TAccumulatorsArray Accumulators; // when empty, should be treated as array of SeriesLength None values
        size_t SeriesLength;
        size_t LeftBorder;
        size_t RightBorder;
    };

    using TSignalToMergedDataMap = THashMap<NSignal::TSignalName, TSubscriptionMergedData>;
    using TMergedDataTable = THashMap<TInternedRequestKey, TSignalToMergedDataMap>;

    /**
     * Class for merging value series received for multiple hosts. Merges by request key and signal name.
     */
    class TSubscriptionValueSeriesMerger {
    public:
        /**
         * @param seriesLength length of a merged time series
         * @param resolution seconds inbetween points in value series
         * @param startTimestamp desired start timestamp of a merged time series
         */
        TSubscriptionValueSeriesMerger(size_t seriesLength, size_t resolution, ui64 startTimestamp);

        /**
         * Merges subscription values with the ones that are already in the merger.
         * @param subscriptionsWithValues list of subscriptions with value series. Note:
         *     - old signals and signal expressions will be ignored
         *     - host field of a subscription will be ignored
         */
        void MulSubscriptionsValueSeries(const TVector<TSubscriptionWithValueSeries>& subscriptionsWithValues);
        void MergeSubscriptionsValueSeries(const TVector<TSubscriptionWithValueSeries>& subscriptionsWithValues);

        const TMergedDataTable& GetMergedDataTable() const noexcept {
            return MergedDataTable;
        }

    private:
        TMergedDataTable MergedDataTable;

        TSubscriptionMergedData& FindOrInsert(const TSubscription& subscription);
        void ProcessValues(const TValueSeries& valueSeries,
                           TSubscriptionMergedData& mergeTo,
                           bool merge);

        const size_t SeriesLength;

        // types are signed so it would be easier to handle negative offsets
        const i64 Resolution;
        const i64 StartTimestamp;
    };
}
