#include "value_series_merger.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NSubscription;
using namespace NZoom::NValue;
using namespace NZoom::NHgram;

namespace {
    // Converts vector of doubles to container of TValue.
    TVector<NZoom::NValue::TValue> DoublesToValues(TVector<double> doubles) {
        TVector<NZoom::NValue::TValue> result;
        for (auto curDouble: doubles) {
            result.emplace_back(curDouble);
        }
        return result;
    }

    TVector<NZoom::NValue::TValue> ExtractValues(const TSubscription& sub, const TSubscriptionValueSeriesMerger& merger) {
        auto reqKeyIt = merger.GetMergedDataTable().find(sub.GetRequestKey());
        if (reqKeyIt == merger.GetMergedDataTable().end()) {
            ythrow yexception() << "No values for request key " << sub.GetRequestKey().GetName();
        }
        auto signalIt = reqKeyIt->second.find(sub.GetSignalExpression());
        if (signalIt == reqKeyIt->second.end()) {
            ythrow yexception() << "No value for signal " << sub.GetSignalExpression();
        }

        TVector<NZoom::NValue::TValue> result;
        for (const auto& acc: signalIt->second.Accumulators) {
            result.emplace_back(acc.GetValue());
        }
        return result;
    }

    std::pair<size_t, size_t> ExtractBorders(const TSubscription& sub, const TSubscriptionValueSeriesMerger& merger) {
        auto reqKeyIt = merger.GetMergedDataTable().find(sub.GetRequestKey());
        if (reqKeyIt == merger.GetMergedDataTable().end()) {
            ythrow yexception() << "No borders for request key " << sub.GetRequestKey().GetName();
        }
        auto signalIt = reqKeyIt->second.find(sub.GetSignalExpression());
        if (signalIt == reqKeyIt->second.end()) {
            ythrow yexception() << "No borders for signal " << sub.GetSignalExpression();
        }

        return std::make_pair(signalIt->second.LeftBorder, signalIt->second.RightBorder);
    }
}

Y_UNIT_TEST_SUITE(TZoomValueSeriesMergerTest) {
    Y_UNIT_TEST(TestTooLeftShift) {
        TSubscription subscription1(TStringBuf("h"), TStringBuf("itype=b1"), TStringBuf("s_dmmm"));
        TVector<TSubscriptionWithValueSeries> subsWithValues;
        subsWithValues.emplace_back(
            subscription1,
            TValueSeries(TInstant::Seconds(100),
                DoublesToValues({1, 2, 3, 4, 5})));
        TSubscriptionValueSeriesMerger merger(5, 5, 125);
        merger.MulSubscriptionsValueSeries(subsWithValues);
        UNIT_ASSERT_VALUES_EQUAL(1, merger.GetMergedDataTable().size()); // it is maybe better if it was empty, but it is ok
        UNIT_ASSERT_VALUES_EQUAL(DoublesToValues({0, 0, 0, 0, 0}), ExtractValues(subscription1, merger));
        auto [left1, right1] = ExtractBorders(subscription1, merger);
        UNIT_ASSERT(left1 >= right1); // empty borders

        TSubscription subscription2(TStringBuf("h"), TStringBuf("itype=b2"), TStringBuf("s_dmmm"));
        // Add another series with smaller shift and check that processing of a too big shift does not affect others
        subsWithValues.emplace_back(
            subscription2,
            TValueSeries(TInstant::Seconds(105),
                DoublesToValues({1, 2, 3, 4, 5})));

        merger.MulSubscriptionsValueSeries(subsWithValues);
        UNIT_ASSERT_VALUES_EQUAL(2, merger.GetMergedDataTable().size());
        UNIT_ASSERT_VALUES_EQUAL(DoublesToValues({5, 0, 0, 0, 0}), ExtractValues(subscription2, merger));
        auto [left2, right2] = ExtractBorders(subscription2, merger);
        UNIT_ASSERT_VALUES_EQUAL(0, left2);
        UNIT_ASSERT_VALUES_EQUAL(1, right2);
    }

    Y_UNIT_TEST(TestTooRightShift) {
        TSubscription subscription1(TStringBuf("h1"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TVector<TSubscriptionWithValueSeries> subsWithValues;
        subsWithValues.emplace_back(
            subscription1,
            TValueSeries(TInstant::Seconds(100),
                DoublesToValues({1, 2, 3, 4, 5})));
        TSubscriptionValueSeriesMerger merger(5, 5, 75);
        merger.MulSubscriptionsValueSeries(subsWithValues);
        UNIT_ASSERT_VALUES_EQUAL(1, merger.GetMergedDataTable().size()); // it is maybe better if it was empty, but it is ok
        UNIT_ASSERT_VALUES_EQUAL(DoublesToValues({0, 0, 0, 0, 0}), ExtractValues(subscription1, merger));
        auto [left1, right1] = ExtractBorders(subscription1, merger);
        UNIT_ASSERT(left1 >= right1); // empty borders

        TSubscription subscription2(TStringBuf("h"), TStringBuf("itype=b2"), TStringBuf("s_dmmm"));
        // Add another series with smaller shift and check that processing of a too big shift does not affect others
        subsWithValues.emplace_back(
            subscription2,
            TValueSeries(TInstant::Seconds(95),
                DoublesToValues({1, 2, 3, 4, 5})));

        merger.MulSubscriptionsValueSeries(subsWithValues);
        UNIT_ASSERT_VALUES_EQUAL(2, merger.GetMergedDataTable().size());
        UNIT_ASSERT_VALUES_EQUAL(DoublesToValues({0, 0, 0, 0, 1}), ExtractValues(subscription2, merger));
        auto [left2, right2] = ExtractBorders(subscription2, merger);
        UNIT_ASSERT_VALUES_EQUAL(4, left2);
        UNIT_ASSERT_VALUES_EQUAL(5, right2);
    }

    Y_UNIT_TEST(TestMergedData) {
        TSubscription subscription1(TStringBuf("h1"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TSubscription subscription2(TStringBuf("h2"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TVector<TSubscriptionWithValueSeries> subsWithValues;
        subsWithValues.emplace_back(
            subscription1,
            TValueSeries(TInstant::Seconds(100),
                DoublesToValues({1, 1, 1, 1, 1})));
        subsWithValues.emplace_back(
            subscription2,
            TValueSeries(TInstant::Seconds(105),
                DoublesToValues({2, 2, 2})));
        subsWithValues.emplace_back(
            subscription2,
            TValueSeries(TInstant::Seconds(95),
                DoublesToValues({1, 1})));
        subsWithValues.emplace_back(
            subscription2,
            TValueSeries(TInstant::Seconds(120),
                DoublesToValues({1, 1})));
        TSubscriptionValueSeriesMerger merger(5, 5, 100);
        merger.MulSubscriptionsValueSeries(subsWithValues);

        UNIT_ASSERT_VALUES_EQUAL(DoublesToValues({2, 3, 3, 3, 2}), ExtractValues(subscription1, merger));
        UNIT_ASSERT_VALUES_EQUAL(DoublesToValues({2, 3, 3, 3, 2}), ExtractValues(subscription2, merger));
        auto [left, right] = ExtractBorders(subscription1, merger);
        UNIT_ASSERT(left >= right); // empty borders
    }

    Y_UNIT_TEST(TestBorders) {
        TSubscription subscription1(TStringBuf("h1"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TSubscription subscription2(TStringBuf("h2"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TVector<TSubscriptionWithValueSeries> subsWithValues;
        subsWithValues.emplace_back(
            subscription1,
            TValueSeries(TInstant::Seconds(100),
                DoublesToValues({1, 1, 1, 1, 1})));
        subsWithValues.emplace_back(
            subscription2,
            TValueSeries(TInstant::Seconds(105),
                DoublesToValues({1, 1, 1})));
        subsWithValues.emplace_back(
            subscription2,
            TValueSeries(TInstant::Seconds(95),
                DoublesToValues({1, 1, 1, 1})));
        subsWithValues.emplace_back(
            subscription2,
            TValueSeries(TInstant::Seconds(110),
                DoublesToValues({1, 1, 1, 1})));
        TSubscriptionValueSeriesMerger merger(5, 5, 100);
        merger.MulSubscriptionsValueSeries(subsWithValues);

        UNIT_ASSERT_VALUES_EQUAL(DoublesToValues({2, 3, 4, 3, 2}), ExtractValues(subscription1, merger));
        UNIT_ASSERT_VALUES_EQUAL(DoublesToValues({2, 3, 4, 3, 2}), ExtractValues(subscription2, merger));
        auto [left, right] = ExtractBorders(subscription1, merger);
        UNIT_ASSERT_VALUES_EQUAL(2, left);
        UNIT_ASSERT_VALUES_EQUAL(3, right);
    }

    Y_UNIT_TEST(TestSingleHostMergedData) {
        TSubscription subscription1(TStringBuf("h1"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TVector<TSubscriptionWithValueSeries> subsWithValues;
        subsWithValues.emplace_back(
            subscription1,
            TValueSeries(TInstant::Seconds(100),
                DoublesToValues({1, 1, 1, 0, 0, 0})));
        subsWithValues.emplace_back(
            subscription1,
            TValueSeries(TInstant::Seconds(100),
                DoublesToValues({1, 1, 1, 2, 2, 2})));
        TSubscriptionValueSeriesMerger merger(6, 5, 100);
        merger.MergeSubscriptionsValueSeries(subsWithValues);

        UNIT_ASSERT_VALUES_EQUAL(DoublesToValues({1, 1, 1, 2, 2, 2}), ExtractValues(subscription1, merger));
        auto [left, right] = ExtractBorders(subscription1, merger);
        UNIT_ASSERT_VALUES_EQUAL(0, left);
        UNIT_ASSERT_VALUES_EQUAL(6, right);
    }

    Y_UNIT_TEST(TestSingleHostMergedDataRelocate) {
        TSubscription subscription1(TStringBuf("h1"), TStringBuf("itype=b1"), TStringBuf("s1_dmmm"));
        TVector<TSubscriptionWithValueSeries> subsWithValues;
        subsWithValues.emplace_back(
            subscription1,
            TValueSeries(TInstant::Seconds(100),
                DoublesToValues({1, 1, 0, 0, 0})));
        subsWithValues.emplace_back(
            subscription1,
            TValueSeries(TInstant::Seconds(100),
                DoublesToValues({0, 0, 0, 2, 2})));
        TSubscriptionValueSeriesMerger merger(5, 5, 100);
        merger.MergeSubscriptionsValueSeries(subsWithValues);

        UNIT_ASSERT_VALUES_EQUAL(DoublesToValues({1, 1, 0, 2, 2}), ExtractValues(subscription1, merger));
        auto [left, right] = ExtractBorders(subscription1, merger);
        UNIT_ASSERT_VALUES_EQUAL(0, left);
        UNIT_ASSERT_VALUES_EQUAL(5, right);
    }
}
