#include "yasmconf.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_value.h>
#include <util/memory/blob.h>

using namespace NZoom::NYasmConf;
using namespace NZoom::NSignal;
using namespace NZoom::NAccumulators;

namespace {
    static constexpr TStringBuf COMMON = "common";

    static const NJson::TJsonValue DEFAULT_MAP(NJson::JSON_MAP);

    template<typename T>
    void ProcessSignals(const NJson::TJsonValue& valueElem, const NJson::TJsonValue& periodsElem,
        TVector<std::pair<T, TAggregationRules>>& values,
        TVector<std::pair<T, TAggregationRules>>& periodicValues)
    {
        for (const auto& kv: valueElem.GetMap()) {
            const TString& signalName = kv.first;
            const NJson::TJsonValue& signalDescr = kv.second;

            //It's wild pornography, but also it is the existing config format.
            const TString& groupAccumulatorStr = signalDescr[1][0].GetString();

            const TMaybe<TAggregationRules> aggRulesMaybe = TAggregationRules::FromYasmConfName(groupAccumulatorStr);
            if (aggRulesMaybe.Empty()) {
                ythrow TParsingError() << "Wrong aggregation rules " + groupAccumulatorStr;
            }
            values.emplace_back(signalName, aggRulesMaybe.GetRef());

            const auto it = periodsElem.GetMap().find(signalName);
            if (it != periodsElem.GetMap().cend()) {
                const TMaybe<TAggregationRules> periodicAggRulesMaybe = TAggregationRules::FromYasmConfName(it->second.GetString());
                if (periodicAggRulesMaybe.Empty()) {
                    ythrow TParsingError() << "Wrong periodic aggregation rules " + it->second.GetString();
                }
                periodicValues.emplace_back(signalName, periodicAggRulesMaybe.GetRef());
            }
        }
    }

    struct TJsonRefs {
        const NJson::TJsonValue& Periods;
        const NJson::TJsonValue& Patterns;
        const NJson::TJsonValue& Signals;

        TJsonRefs(const NJson::TJsonValue& value)
            : Periods(value.Has("periods") ? value["periods"] : DEFAULT_MAP)
            , Patterns(value.Has("patterns") ? value["patterns"] : DEFAULT_MAP)
            , Signals(value.Has("signals") ? value["signals"] : DEFAULT_MAP)
        {
            if (!Periods.IsMap()) {
                ythrow TParsingError() << "periods is not map";
            }
            if (!Patterns.IsMap()) {
                ythrow TParsingError() << "patterns is not map";
            }
            if (!Signals.IsMap()) {
                ythrow TParsingError() << "signals is not map";
            }
        }
    };

    TTypeConf TypeConfFromJson(const NJson::TJsonValue& value, const TTypeConf* commonTypePtr) {
        if (!value.IsMap()) {
            ythrow TParsingError() << "TypeConf is not map";
        }

        TJsonRefs refs(value);

        TVector<std::pair<TRegExMatch, TAggregationRules>> patterns;
        const size_t patternsSize = refs.Patterns.GetMap().size() + (commonTypePtr ? commonTypePtr->GetPatterns()->GetSize() : 0);
        patterns.reserve(patternsSize);

        TVector<std::pair<TSignalName, TAggregationRules>> signals;
        const size_t signalsSize = refs.Signals.GetMap().size() + (commonTypePtr ? commonTypePtr->GetSignals()->size() : 0);
        signals.reserve(signalsSize);

        TVector<std::pair<TSignalName, TAggregationRules>> periodicSignals;
        TVector<std::pair<TRegExMatch, TAggregationRules>>  periodicPatterns;

        ProcessSignals(refs.Signals, refs.Periods, signals, periodicSignals);
        ProcessSignals(refs.Patterns, refs.Periods, patterns, periodicPatterns);

        if (commonTypePtr != nullptr) {
            signals.insert(signals.end(), commonTypePtr->GetSignals()->cbegin(), commonTypePtr->GetSignals()->cend());

            const auto& commonPatterns = commonTypePtr->GetPatterns()->GetPatterns();
            patterns.insert(patterns.end(), commonPatterns.cbegin(), commonPatterns.cend());

            const auto& commonPeriodicSignals = commonTypePtr->GetPeriodicSignals();
            periodicSignals.insert(periodicSignals.end(), commonPeriodicSignals->cbegin(), commonPeriodicSignals->cend());

            const auto& commonPeriodicPatterns = commonTypePtr->GetPeriodicPatterns()->GetPatterns();
            periodicPatterns.insert(periodicPatterns.end(), commonPeriodicPatterns.cbegin(), commonPeriodicPatterns.cend());
        }
        return TTypeConf(std::move(signals), std::move(patterns), std::move(periodicSignals), std::move(periodicPatterns));
    }

    TYasmConf YasmConfFromJson(const NJson::TJsonValue& value) {
        const NJson::TJsonValue& confList = value["conflist"];
        if (!confList.IsMap()) {
            ythrow TParsingError() << "conflist is not map";
        }
        if (!confList.Has(COMMON)) {
            ythrow TParsingError() << "conflist has no common"; //TODO separate exception type
        }

        TTypeConf commonType = TypeConfFromJson(confList[COMMON], nullptr);
        TMap<TString, std::pair<TTypeConf, TTypeConf>> types;

        for (const auto& p: confList.GetMap()) {
            if (p.first == COMMON) {
                continue;
            }
            types.emplace(p.first,
                std::make_pair(TypeConfFromJson(p.second, &commonType), TypeConfFromJson(p.second, nullptr)));
        }
        return TYasmConf(std::move(types), std::move(commonType));
    }
}

TPatterns::TPatterns(const TVector<std::pair<TRegExMatch, TAggregationRules>>& patterns)
    : Patterns(patterns)
{
}

TPatterns::TPatterns(const TVector<std::pair<TRegExMatch, TAggregationRules>>&& patterns)
    : Patterns(std::move(patterns))
{
}

size_t TPatterns::GetSize() const noexcept {
    return Patterns.size();
}

const TVector<std::pair<TRegExMatch, TAggregationRules>>& TPatterns::GetPatterns() const {
    return Patterns;
}

const TAggregationRules* TPatterns::GetMatchedRule(const TString& signal) const {
    for (auto it = Patterns.cbegin(); it != Patterns.cend(); ++it) {
        if (it->first.Match(signal.data())) {
            return &(it->second);
        }
    }
    return nullptr;
}

TTypeConf::TTypeConf()
    : SignalsMap(MakeAtomicShared<TSignalsMap>())
    , Patterns(new TPatterns(TVector<std::pair<TRegExMatch, NZoom::NAccumulators::TAggregationRules>>()))
    , PeriodicSignalsMap(MakeAtomicShared<TSignalsMap>())
    , PeriodicPatterns(Patterns)
{
}

TTypeConf::TTypeConf(TTypeConf&& other)
    : SignalsMap(std::move(other.SignalsMap))
    , Patterns(std::move(other.Patterns))
    , PeriodicSignalsMap(std::move(other.PeriodicSignalsMap))
    , PeriodicPatterns(std::move(other.PeriodicPatterns))
{
}

TTypeConf::TTypeConf(
    TVector<TSignalPair>&& signals,
    TPatterns&& patterns,
    TVector<TSignalPair>&& periodicSignals,
    TPatterns&& periodicPatterns
)
    : SignalsMap(MakeAtomicShared<TSignalsMap>())
    , Patterns(new TPatterns(std::move(patterns)))
    , PeriodicSignalsMap(MakeAtomicShared<TSignalsMap>())
    , PeriodicPatterns(new TPatterns(std::move(periodicPatterns)))
{
    for (const auto& pair: signals) {
        SignalsMap->emplace(pair);
    }

    for (const auto& pair: periodicSignals) {
        PeriodicSignalsMap->emplace(pair);
    }
}

const TAtomicSharedPtr<TPatterns> TTypeConf::GetPatterns() const {
    return Patterns;
}

const TAtomicSharedPtr<TTypeConf::TSignalsMap> TTypeConf::GetSignals() const {
    return SignalsMap;
}

const TAtomicSharedPtr<TPatterns> TTypeConf::GetPeriodicPatterns() const {
    return PeriodicPatterns;
}

const TAtomicSharedPtr<TTypeConf::TSignalsMap> TTypeConf::GetPeriodicSignals() const {
    return PeriodicSignalsMap;
}

const NZoom::NAccumulators::TAggregationRules* TTypeConf::GetRules(const NZoom::NSignal::TSignalName& name) const {
    const auto it = SignalsMap->find(name);
    if (it == SignalsMap->end()) {
        return nullptr;
    }
    return &it->second;
}

TYasmConf::TYasmConf(TMap<TString, std::pair<TTypeConf, TTypeConf>>&& types, TTypeConf&& commonType)
    : Types(std::move(types))
    , CommonType(std::move(commonType))
{
}

TYasmConf::TYasmConf(TYasmConf&& other)
    : Types(std::move(other.Types))
    , CommonType(std::move(other.CommonType))
{
}

const TTypeConf& TYasmConf::GetTypeConf(TStringBuf itype, const bool includeCommon) const {
    if (itype == COMMON) {
        return CommonType;
    }
    const auto it = Types.find(itype);
    if (it != Types.cend()) {
        if (includeCommon) {
            return it->second.first;
        } else {
            return it->second.second;
        }
    }
    if (includeCommon) {
        return CommonType;
    }
    return EmptyType;
}

TYasmConf TYasmConf::FromString(TStringBuf str) {
    NJson::TJsonValue parsed;
    if (!NJson::ReadJsonTree(str, &parsed)) {
        ythrow TParsingError() << "Failed to parse json";
    }
    return YasmConfFromJson(parsed);
}

TYasmConf* TYasmConf::FromCharString(TStringBuf str) {
    return MakeHolder<TYasmConf>(FromString(str)).Release();
}

TYasmConf TYasmConf::FromFileName(const TString& path) {
    const auto blob(TBlob::FromFile(path));
    return TYasmConf::FromString(TStringBuf(blob.AsCharPtr(), blob.Size()));
}
