# coding: utf-8

from util.datetime.base cimport TDuration, TInstant
from util.system.types cimport i64, ui64, ui16
from util.generic.string cimport TStringBuf, TString
from util.generic.maybe cimport TMaybe
from util.generic.vector cimport TVector
from util.generic.ptr cimport THolder

from libcpp cimport bool as bool_t
from libcpp.utility cimport pair

from collections import namedtuple

from cpython cimport PyObject, Py_DECREF

from infra.yasm.zoom.components.aggregators.preaggregates cimport TCommonRules, TAggregationRules, FillRules
from infra.yasm.zoom.components.aggregators.metrics cimport TTaggedMetricManager
from infra.yasm.zoom.components.accumulators cimport TAccumulator
from infra.yasm.zoom.components.containers.agent cimport TAgentContainer
from infra.yasm.zoom.components.containers.integrate cimport TTagMerger
from infra.yasm.zoom.components.containers.group cimport TGroupContainer
from infra.yasm.zoom.components.containers.rollup cimport TRollupContainer

from infra.yasm.zoom.components.record.record cimport (TRecord, TTaggedRecordPtr, TContinuousRecord, IContinuousSignalCallback,
                                                       TContinuousSignalToRecordCallback, TSignalValueToRecordStorageCallback,
                                                       TWindowRecord, TSingleMultiTaggedRecord)
from infra.yasm.zoom.components.subscription.subscription cimport TSubscriptionWithValueSeries, TSubscriptionValueSeriesMerger

from infra.yasm.zoom.components.value_types cimport TMetricManager, TValueRef, TValue
from infra.yasm.zoom.components.yasmconf.yasmconf cimport TYasmConf

from infra.yasm.zoom.components.serialization.zoom_to_msgpack.zoom_to_msgpack cimport TMsgPackSerializer
from infra.yasm.zoom.components.serialization.python.py_to_zoom cimport TPyDeserializer, PyToValue
from infra.yasm.zoom.components.serialization.python.zoom_to_py cimport (TPyListIterator, TSignalValueToDictStorage,
                                                                         TTagRecordToDictStorage, ValueToPy,
                                                                         TContinuousValueToAccumulatorCallback, TSubscriptionMergedDataPySerializer)

from infra.yasm.zoom.python.responses cimport (TPerInstanceRecordsIterator, TInstanceContent,
                                               TMiddleResponseLoader, TServerResponseLoader,
                                               TAgentProtobufResponseParser, TSubscriptionSplitter,
                                               TGetSubscriptionValuesResponseLoader, THostSubscriptionsIterator,
                                               TNodeAgentResponseLoader)
from infra.yasm.zoom.python.histdb cimport (ESnappyMode, TSimpleSomethingFormat,
                                            TSimpleSomethingIterator, TSimpleSomethingKeyIterator,
                                            TSimpleSomethingRow, TSimpleSomethingIteratorRow, TSimpleSomethingKeyIteratorRow, TReadData)
from infra.yasm.zoom.python.unistat.unistat cimport EDeserializationStatus, TUnistatStats, TUnistatValuesIterator, TUnistatDeserializer
from infra.yasm.zoom.python.pipelines.pipelines cimport (TRequesterPipeline, TMiddlePipeline, TServerPipeline, TServerWorker,
                                                         IMessagePusher, TMessagePusher, TAsyncMessagePusher, TInMemoryMessagePusher,
                                                         TRequesterPipelineStatsAndMessages, TServerStats, TSubscriptionStoreProtobufRequestSerializer,
                                                         THistoryApiRequestSerializer, THistoryApiResponseDeserializer, THistoryApiRequestDeserializer,
                                                         THistoryRequest, THistoryResponse, THistoryAggregatorWrapper, THistoryAggregatorToPythonVisitor)
from infra.yasm.zoom.python.re2 cimport TRe2Validator

from infra.yasm.zoom.components.accumulators_types cimport TAggregationRules as TAccumulatorAggregationRules
from infra.yasm.zoom.components.signal cimport TSignalName

__doc__ = "Yasm speedups"


cdef class ZYasmConf:
    cdef TYasmConf* yasmConf

    def __cinit__(self, json_config):
        cdef TStringBuf config_buf = json_config
        with nogil:
            self.yasmConf = TYasmConf.FromCharString(config_buf)

    def __dealloc__(self):
        del self.yasmConf


cdef class ZRecord:
    cdef TRecord* record
    cdef bool_t skip_empty_ugram_buckets

    def __cinit__(self):
        self.record = NULL
        self.skip_empty_ugram_buckets = True

    def __dealloc__(self):
        if (self.record != NULL):
            del self.record

    @staticmethod
    cdef ZRecord from_ptr(TRecord* ptr):
        record = ZRecord()
        record.record = ptr
        return record

    @classmethod
    def from_dict(cls, data, skip_empty_ugram_buckets=True):
        record = ZRecord()
        record.skip_empty_ugram_buckets = skip_empty_ugram_buckets
        with nogil:
            record.record = TPyDeserializer.DeserializeDict(<PyObject*>data, record.skip_empty_ugram_buckets)
        return record

    @classmethod
    def from_iter(cls, data, skip_empty_ugram_buckets=True):
        record = ZRecord()
        record.skip_empty_ugram_buckets = skip_empty_ugram_buckets
        record.record = TPyDeserializer.DeserializeIterable(<PyObject*>data, record.skip_empty_ugram_buckets)
        return record

    def dumps(self):
        if (self.record == NULL):
            return ""

        cdef TStringBuf value
        cdef TMaybe[TMsgPackSerializer] storage
        storage.ConstructInPlace(self.skip_empty_ugram_buckets)
        with nogil:
            self.record.Process(storage.GetRef())
            value = storage.GetRef().GetValue()
        return value

    def get_values(self, cached_lists=False):
        if (self.record == NULL):
            return {}

        cdef TSignalValueToDictStorage storage
        storage.Init(self.skip_empty_ugram_buckets)
        self.record.Process(storage)
        res = <object>storage.GetValue()
        Py_DECREF(res)
        return res

    def len(self):
        if (self.record == NULL):
            return 0
        return self.record.Len()


cdef class ZTaggedRecord:
    cdef TTaggedRecordPtr tagged_record

    def __cinit__(self):
        self.tagged_record = NULL

    def __dealloc__(self):
        if self.tagged_record != NULL:
            del self.tagged_record

    @staticmethod
    cdef ZTaggedRecord from_ptr(TTaggedRecordPtr ptr):
        tagged_record = ZTaggedRecord()
        tagged_record.tagged_record = ptr
        return tagged_record

    def get_values(self):
        if (self.tagged_record == NULL):
            return {}

        cdef TTagRecordToDictStorage storage
        storage.Init()
        self.tagged_record.Process(storage)
        res = <object>storage.GetValue()
        Py_DECREF(res)
        return res


cdef class ZSingleMultiTaggedRecord:
    cdef THolder[TSingleMultiTaggedRecord] record

    @staticmethod
    cdef ZSingleMultiTaggedRecord from_ptr(TSingleMultiTaggedRecord* ptr):
        record = ZSingleMultiTaggedRecord()
        record.record.Reset(ptr)
        return record

    def len(self):
        if self.record.Get() == NULL:
            return 0
        return self.record.Get().GetRecord().Len()


cdef class ZWindowRecord:
    cdef TMaybe[TWindowRecord] record

    def __cinit__(self, size):
        cdef size_t local_size = size
        self.record.ConstructInPlace(local_size)

    def push(self, timestamp, value):
        cdef TInstant time_instant = TInstant.Seconds(timestamp)
        cdef THolder[TValue] v

        v.Reset(PyToValue(<PyObject*> value, True))
        with nogil:
            self.record.GetRef().Push(time_instant, v.Get()[0])

    def rollup(self, ZAccumulator accumulator):
        cdef TMaybe[TContinuousValueToAccumulatorCallback] callback

        with nogil:
            callback.ConstructInPlace(accumulator.data[0])
            self.record.GetRef().Process(callback.GetRef())

    @property
    def size(self):
        return self.record.GetRef().Len()


cdef class ZMetricManager:
    cdef TMetricManager metrics

    def __cinit__(self):
        pass

    @staticmethod
    cdef _to_dict_impl(TMetricManager metrics):
        res = {}
        res["totalPtsMetric"] = metrics.TotalPoints
        res["pointsNegMetric"] = metrics.NegativePoints
        res["noneSigMetric"] = metrics.NonePoints
        res["hgramCntMetric"] = metrics.TotalHgrams
        res["hgramLenMetric"] = metrics.HgramsLen
        res["ugramCntMetric"] = metrics.TotalUgrams
        res["ugramLenMetric"] = metrics.UgramsLen
        res["listsCntMetric"] = metrics.TotalLists
        res["listsLenMetric"] = metrics.ListsLen
        return res

    def to_dict(self):
        return ZMetricManager._to_dict_impl(self.metrics)


cdef class ZTaggedMetricManager:
    cdef TTaggedMetricManager metrics

    def __cinit__(self):
        pass

    def to_dict(self):
        cdef TVector[pair[TString, TMetricManager]] tagMetrics = self.metrics.Get()

        result = {}
        for pair in tagMetrics:
            result[pair.first] = ZMetricManager._to_dict_impl(pair.second)

        return result


cdef class ZSubscriptionValueSeriesMerger:
    cdef THolder[TSubscriptionValueSeriesMerger] merger
    cdef THolder[TSubscriptionMergedDataPySerializer] merged_data_serializer

    def __cinit__(self, series_length, resolution, start_ts, allow_legacy_types):
        cdef size_t typed_series_length = series_length
        cdef size_t typed_resolution = resolution
        cdef ui64 typed_start_ts = start_ts
        cdef bool_t typed_allow_legacy_types = allow_legacy_types
        self.merger.Reset(new TSubscriptionValueSeriesMerger(typed_series_length, typed_resolution,
            typed_start_ts))
        self.merged_data_serializer.Reset(new TSubscriptionMergedDataPySerializer(typed_allow_legacy_types))

    def mul_subs_value_series(self, ZSubscriptionsWithValueSeries subs_value_series):
        self.merger.Get().MulSubscriptionsValueSeries(subs_value_series.subscriptionsWithValues.Get()[0])

    def merge_subs_value_series(self, ZSubscriptionsWithValueSeries subs_value_series):
        self.merger.Get().MergeSubscriptionsValueSeries(subs_value_series.subscriptionsWithValues.Get()[0])

    def prepare_results(self):
        self.merged_data_serializer.Get().TraverseMergerData(self.merger.Get()[0])

    def get_values(self):
        return <object>self.merged_data_serializer.Get().GetValues()

    def get_borders(self):
        return <object>self.merged_data_serializer.Get().GetBorders()

    def get_converted_signals(self):
        return <object>self.merged_data_serializer.Get().GetConvertedSignals()


cdef class ZAgentContainer:
    cdef TAgentContainer* data

    def __cinit__(self, ZYasmConf conf, itype, host_aggregation):
        self.data = new TAgentContainer(conf.yasmConf.GetTypeConf(itype, False), host_aggregation)

    def __dealloc__(self):
        del self.data

    def len(self):
        return self.data.Len()

    def clean(self):
        self.data.Clean()

    def mul_container(self, ZAgentContainer other):
        with nogil:
            self.data.Mul(other.data[0])

    def mul_record(self, ZRecord record):
        cdef TAgentContainer* localData = self.data
        if (record.record != NULL):
            with nogil:
                self.data.Mul(record.record[0])

    def mul_record_ttl(self, ZRecord record, size_t ttl):
        cdef TAgentContainer* localData = self.data
        if (record.record != NULL):
            with nogil:
                self.data.MulTTL(record.record[0], TDuration.Seconds(ttl))

    def mul_multi_tagged_record(self, ZSingleMultiTaggedRecord record):
        with nogil:
            self.data.Mul(record.record.Get()[0].GetRecord())

    def get_values(self):
        cdef TSignalValueToDictStorage storage
        storage.Init(False)
        self.data.Process(storage)
        res = <object>storage.GetValue()
        Py_DECREF(res)
        with nogil:
            self.data.Clean()
        return res

    def spread_and_forget(self, others_list):
        cdef TAgentContainer* other_data
        cdef ZAgentContainer other_objc
        for other_obj in others_list:
            other_objc = other_obj
            other_data = other_objc.data
            if other_data == self.data:
                raise ValueError("Mul ZAgentContainer with itself")
            with nogil:
                other_data.Mul(self.data[0])
        return self.get_values()

    def trim(self, size_t size):
        cdef size_t res
        with nogil:
            res = self.data.Trim(size)
        return res


cdef class ZAccumulator:
    cdef TAccumulator* data

    def __cinit__(self, yasmConfName):
        self.data = TAccumulator.FromYasmConfName(yasmConfName)
        if self.data == NULL:
            raise ValueError("Invalid accumulator type name")

    def __dealloc__(self):
        if self.data != NULL:
            del self.data

    def clean(self):
        self.data.Clean()

    def get_value(self, cached_lists=False):
        res = <object>ValueToPy(self.data.GetValue(), True)
        Py_DECREF(res)
        return res

    def mul(self, value):
        cdef TValue* v = NULL
        try:
            with nogil:
                v = PyToValue(<PyObject*> value, True)
                self.data.Mul(v[0])
        finally:
            if v != NULL:
                del v


def loads_subagent_protobuf_response(response):
    cdef TStringBuf resp_buf = response
    cdef TMaybe[TAgentProtobufResponseParser] parser
    cdef TPerInstanceRecordsIterator iterator

    with nogil:
        parser.ConstructInPlace(resp_buf)
        iterator = parser.GetRef().LoadPerInstanceRecords()

    status = None
    cdef TMaybe[TString] status_maybe = parser.GetRef().GetStatus()
    if status_maybe.Defined():
        status = status_maybe.GetRef()

    data = {}
    while not iterator.IsEnd():
        instance_name = iterator.GetInstanceName()
        data[instance_name] = ZSingleMultiTaggedRecord.from_ptr(iterator.ExtractRecord())
        iterator.MoveNext()

    return status, data


def loads_middle_message(body):
    cdef TStringBuf body_buf = body
    cdef TMiddleResponseLoader loader
    cdef TTaggedRecordPtr tagged_record
    with nogil:
        tagged_record = loader.LoadTaggedRecord(body_buf)
    return ZTaggedRecord.from_ptr(tagged_record)


def loads_server_message(body):
    cdef TStringBuf body_buf = body
    cdef TServerResponseLoader loader
    cdef TVector[pair[TString, TTaggedRecordPtr]] result
    with nogil:
        result = loader.LoadHostRecords(body_buf)
    return {pair.first: ZTaggedRecord.from_ptr(pair.second) for pair in result}


cdef class ZSomethingFormat:
    cdef TSimpleSomethingFormat* something_format

    def __cinit__(self, header_data, file_name, mode, use_new_tags=True, use_direct=True):
        cdef TStringBuf header_data_buf
        cdef TString file_name_buf = file_name

        cdef ESnappyMode snappy_mode
        if mode in ("r", "rb"):
            snappy_mode = ESnappyMode.READ
        elif mode in ("a", "ab"):
            snappy_mode = ESnappyMode.APPEND
        else:
            raise ValueError("Invalid mode: {!r}".format(mode))

        cdef bool_t use_new_tags_bool = use_new_tags
        cdef bool_t use_direct_bool = use_direct

        if header_data:
            header_data_buf = header_data
            with nogil:
                self.something_format = new TSimpleSomethingFormat(
                    header_data_buf, file_name_buf, snappy_mode, use_new_tags_bool, use_direct_bool
                )
        else:
            with nogil:
                self.something_format = new TSimpleSomethingFormat(
                    file_name_buf, snappy_mode, use_direct_bool
                )

    def __dealloc__(self):
        if self.something_format != NULL:
            del self.something_format

    def dump(self):
        cdef TString buf
        with nogil:
            buf = self.something_format.Dump()
        return buf

    def write_record(self, timestamp, key, ZRecord record):
        cdef ui64 converted_timestamp = timestamp
        cdef TString converted_key = key
        cdef TRecord* converted_record = record.record
        with nogil:
            self.something_format.WriteRecord(converted_timestamp, converted_key, converted_record[0])

    def has_records(self, timestamps, keys):
        cdef TVector[ui64] converted_timestamps = timestamps
        cdef TVector[TString] converted_keys = keys

        cdef TVector[bool_t] result
        with nogil:
            result = self.something_format.HasRecords(converted_timestamps, converted_keys)

        return result

    def read(self, times, tags):
        cdef TVector[ui64] converted_times = times
        cdef TVector[pair[TString, TVector[TString]]] converted_tags

        for rkey, signals in tags.iteritems():
            converted_tags.push_back((rkey.raw, signals))

        cdef TVector[TReadData] result

        with nogil:
            result = self.something_format.Read(converted_times, converted_tags)
        for res in result:
            record = ZRecord.from_ptr(res.ReleaseRecord())
            values = record.get_values()
            yield (res.TimeStamp, res.RequesterName, res.InstanceName, values)


    def read_records(self, timestamps, keys_and_signals):
        cdef TVector[ui64] converted_timestamps = timestamps
        cdef TVector[pair[TString, TVector[TString]]] converted_keys_and_signals = keys_and_signals

        cdef TVector[TSimpleSomethingRow] result
        with nogil:
            result = self.something_format.ReadRecords(converted_timestamps, converted_keys_and_signals)

        return [
            (row.Timestamp, row.RequestedIndex, ZRecord.from_ptr(row.ReleaseRecord()))
            for row in result
        ]

    def iterate_records(self, timestamps):
        cdef TVector[ui64] converted_timestamps = timestamps

        cdef TSimpleSomethingIterator iterator
        with nogil:
            iterator = self.something_format.IterateRecords(converted_timestamps)

        cdef TMaybe[TSimpleSomethingIteratorRow] row
        while True:
            row = iterator.Next()
            if row.Empty():
                return
            else:
                yield (
                    row.GetRef().Timestamp,
                    row.GetRef().GetKey(),
                    ZRecord.from_ptr(row.GetRef().ReleaseRecord())
                )

    def iterate_keys(self):
        cdef TSimpleSomethingKeyIterator iterator
        with nogil:
            iterator = self.something_format.IterateKeys()

        cdef TMaybe[TSimpleSomethingKeyIteratorRow] row
        while True:
            row = iterator.Next()
            if row.Empty():
                return
            else:
                yield (
                    row.GetRef().GetKey(),
                    row.GetRef().GetSignals()
                )

    def first_record_time(self):
        return self.something_format.FirstRecordTime()

    def last_record_time(self):
        return self.something_format.LastRecordTime()

    def finish(self):
        self.something_format.Finish()


cdef class ZSubscriptionSplitter:
    cdef THolder[TSubscriptionSplitter] splitter

    def __cinit__(self, ui64 bucketCount):
        self.splitter.Reset(new TSubscriptionSplitter(bucketCount))

    def get_total(self):
        return self.splitter.Get().GetBucketCount()

    def set_metagroup_for_group(self, group, metagroup):
        cdef TStringBuf group_name = group
        cdef TStringBuf metagroup_name = metagroup
        self.splitter.Get().SetMetagroupForGroup(group_name, metagroup_name)

    def find_shard_id(self, host, tags):
        cdef TStringBuf host_name = host
        cdef TStringBuf request_key = tags
        return self.splitter.Get().GetSubscriptionBucket(host_name, request_key)


cdef class ZSubscriptionsWithValueSeries:
    cdef THolder[TVector[TSubscriptionWithValueSeries]] subscriptionsWithValues

    @staticmethod
    cdef ZSubscriptionsWithValueSeries from_ptr(TVector[TSubscriptionWithValueSeries]* subsWithValuesVector):
        result = ZSubscriptionsWithValueSeries()
        result.subscriptionsWithValues.Reset(subsWithValuesVector)
        return result


cdef class ZGetSubscriptionValuesResponseLoader:
    cdef THolder[TGetSubscriptionValuesResponseLoader] loader

    def __cinit__(self):
        self.loader.Reset(new TGetSubscriptionValuesResponseLoader())

    def load(self, protobuf_response):
        cdef TStringBuf response_to_parse = protobuf_response
        with nogil:
            self.loader.Get().Load(response_to_parse)
        cdef THostSubscriptionsIterator it = self.loader.Get().GetHostSubscriptionsIterator()
        result = {}
        try:
            while not it.IsEnd():
                result[it.GetHostName()] = ZSubscriptionsWithValueSeries.from_ptr(it.ExtractValues())
                it.MoveNext()
        finally:
            self.loader.Get().Clear()
        return result

    @staticmethod
    def split(protobuf_response, ZSubscriptionSplitter splitter):
        cdef TStringBuf response_to_parse = protobuf_response
        cdef TVector[TString] result
        cdef ui64 max_start_ts = 0
        with nogil:
            result = TGetSubscriptionValuesResponseLoader.Split(response_to_parse, splitter.splitter.Get()[0], max_start_ts)
        return result, max_start_ts


cdef class ZSubscriptionStoreProtobufRequestSerializer:

    @staticmethod
    def subscriptions_dict_to_list_request(subscriptions_dict):
        return TSubscriptionStoreProtobufRequestSerializer.SubscriptionsDictToListRequest(<PyObject*>subscriptions_dict)


cdef class ZHistoryRequest:
    cdef THolder[THistoryRequest] request

    @staticmethod
    cdef ZHistoryRequest from_ptr(THistoryRequest* request):
        result = ZHistoryRequest()
        result.request.Reset(request)
        return result

    def get_host_name(self):
        return self.request.Get()[0].GetHostName()


cdef class ZHistoryResponse:
    cdef THolder[THistoryResponse] response

    @staticmethod
    cdef ZHistoryResponse from_ptr(THistoryResponse* response):
        result = ZHistoryResponse()
        result.response.Reset(response)
        return result

    @staticmethod
    def create(response):
        return ZHistoryResponse.from_ptr(THistoryApiResponseDeserializer.FromPython(<PyObject*>response))


cdef class ZHistoryApiRequestSerializer:

    @staticmethod
    def requests_to_proto(requests, deadline=None, request_id=None):
        cdef TInstant c_deadline = TInstant.Seconds(deadline or 0)
        cdef TString c_request_id = request_id or ""
        return THistoryApiRequestSerializer.RequestsToProto(<PyObject*>requests, c_deadline, c_request_id)

    @staticmethod
    def responses_to_proto(responses):
        cdef ZHistoryResponse typed_response
        cdef TVector[THistoryResponse*] typed_responses
        for response in responses:
            typed_response = response
            typed_responses.push_back(typed_response.response.Get())
        return THistoryApiRequestSerializer.ResponsesToProto(typed_responses)

    @staticmethod
    def load_requests(content):
        cdef TStringBuf c_content = content
        cdef THistoryApiRequestDeserializer deserializer
        with nogil:
            deserializer.Load(c_content)

        result = []
        while not deserializer.IsEnd():
            result.append(ZHistoryRequest.from_ptr(deserializer.Next()))
        return result

cdef class ZHistoryApiResponseMerger:
    cdef TMaybe[THistoryApiResponseDeserializer] deserializer

    def __cinit__(self):
        self.deserializer.ConstructInPlace()

    def load_responses(self, content):
        cdef TStringBuf c_content = content
        cdef TVector[int] codes
        with nogil:
            codes = self.deserializer.GetRef().Load(c_content)
        result = []
        for code in codes:
            result.append(code)
        return result

    def merge_loaded_responses(self):
        with nogil:
            self.deserializer.GetRef().Merge()

        result = []
        while not self.deserializer.GetRef().IsEnd():
            result.append(ZHistoryResponse.from_ptr(self.deserializer.GetRef().Next()))
        return result


cdef class ZHistoryAggregator:
    cdef TMaybe[THistoryAggregatorWrapper] aggregator

    def __cinit__(self, ZYasmConf conf):
        self.aggregator.ConstructInPlace(conf.yasmConf[0])

    def add_requests(self, requests):
        self.aggregator.GetRef().AddRequests(<PyObject*>requests)

    def add_metagroups(self, metagroups):
        self.aggregator.GetRef().AddMetagroups(<PyObject*>metagroups)

    def mul_response(self, ZHistoryResponse response):
        with nogil:
            self.aggregator.GetRef().Mul(response.response.Get()[0])

    def overwrite(self, ZHistoryAggregator other):
        with nogil:
            self.aggregator.GetRef().Overwrite(other.aggregator.GetRef())

    def overwrite_since(self, ZHistoryAggregator other, since, until):
        cdef TInstant since_instance = TInstant.Seconds(since)
        cdef TInstant until_instance = TInstant.Seconds(until)
        with nogil:
            self.aggregator.GetRef().OverwriteSince(other.aggregator.GetRef(), since_instance, until_instance)

    def overwrite_continuous(self, ZHistoryAggregator other):
        with nogil:
            self.aggregator.GetRef().OverwriteContinuous(other.aggregator.GetRef())

    def get_responses(self, allow_legacy_types=True):
        cdef THistoryAggregatorToPythonVisitor visitor
        visitor.SetAllowLegacyTypes(allow_legacy_types)
        self.aggregator.GetRef().Visit(visitor)
        res = <object>visitor.GetResponses()
        Py_DECREF(res)
        converted_signals = <object>visitor.GetConvertedSmallSignals()
        Py_DECREF(converted_signals)
        return res, converted_signals


cdef class ZUnistatLoader:
    cdef TMaybe[TUnistatDeserializer] deserializer

    def __cinit__(self, prefix, limit):
        cdef TStringBuf prefixBuf = prefix
        cdef size_t limit_size
        if limit <= 0:
            limit = 1000
        limit_size = limit
        self.deserializer.ConstructInPlace(prefixBuf, limit_size)

    def loads(self, response):
        cdef TUnistatValuesIterator it
        cdef TStringBuf respBuf = response
        cdef pair[TString, TRecord*] item

        with nogil:
            self.deserializer.GetRef().Loads(respBuf, it)

        records_list = []
        while it.IsValid():
            item = it.GetAndMove()
            pyItem = (item.first, ZRecord.from_ptr(item.second))
            records_list.append(pyItem)

        cdef TUnistatStats stats = it.GetStats()

        cdef THolder[TValue] ugramSizeStatsValue
        ugramSizeStatsValue.Reset(stats.GetAndResetUgramSizes())
        ugramSizeStats = <object>ValueToPy(ugramSizeStatsValue.Get().GetValue(), False)
        Py_DECREF(ugramSizeStats)

        py_stats = {"filtered_signals_mmmm": stats.FilteredSignals,
                    "limit_usage_perc_xxxx": stats.LimitUsage,
                    "num_hgrams_hgram": stats.NumHgrams,
                    "num_signals_hgram": stats.NumSignals,
                    "ugrams_changed_buckets_mmmm": stats.NumUgramsChangedBuckets,
                    "ugram_bucket_counts_hgram": ugramSizeStats,
                    "response_bytes_hgram": stats.ResponseBytes,
                    "invalid_signals_mmmm": stats.InvalidSignals,
                    "parse_errors_mmmm": stats.ParseErrors}
        error_message = None
        if stats.Status == EDeserializationStatus.INVALID_UNSPECIFIED:
            error_message = "Unknown error"
        if stats.Status == EDeserializationStatus.EXTRA_DEPTH:
            error_message = "Extra bracket detected while parsing hgram"
        if stats.Status == EDeserializationStatus.TOO_MUCH_ARGUMENTS_IN_BUCKET:
            error_message = "Too much arguments in ugram bucket detected"

        return (records_list, py_stats, error_message)


cdef class ZCommonRules:
    cdef TMaybe[TCommonRules] rules

    def __cinit__(self):
        self.rules.ConstructInPlace()

    def add_rule(self, request_key, selector = None):
        if selector is None:
            self.rules.GetRef().AddRule(request_key)
        else:
            self.rules.GetRef().AddRule(request_key, selector)


cdef class ZAggregationRules:
    cdef TMaybe[TAggregationRules] rules

    def __cinit__(self):
        self.rules.ConstructInPlace()

    def add_custom_rule(self, request_key, tag_names):
        cdef TStringBuf request_key_str = request_key
        cdef TVector[TStringBuf] tag_names_vector = tag_names
        self.rules.GetRef().AddCustomRule(request_key, tag_names_vector)

    def add_default_rule(self, tag_names):
        cdef TVector[TStringBuf] tag_names_vector = tag_names
        self.rules.GetRef().AddDefaultRule(tag_names_vector)


def fillCommonAggregationRules(commonRule, aggregationRules, path):
    cdef ZAggregationRules aggrRule = aggregationRules
    cdef ZCommonRules commRule = commonRule
    cdef TStringBuf pathBuf = path
    FillRules(commRule.rules.GetRef(), aggrRule.rules.GetRef(), pathBuf)


cdef class AbstractMessagePusher:

    cdef IMessagePusher* get_pusher(self):
        raise NotImplementedError("not implemented")


cdef class ZMessagePusher(AbstractMessagePusher):
    cdef THolder[TMessagePusher] pusher

    def __cinit__(self, url):
        cdef TStringBuf url_str = url
        self.pusher.Reset(new TMessagePusher(url_str))

    cdef IMessagePusher* get_pusher(self):
        return self.pusher.Get()


cdef class ZAsyncMessagePusher(AbstractMessagePusher):
    cdef THolder[TAsyncMessagePusher] pusher

    def __cinit__(self, url, duration):
        cdef TStringBuf url_str = url
        cdef TDuration dur = TDuration.Seconds(duration)
        self.pusher.Reset(new TAsyncMessagePusher(url_str, dur))

    cdef IMessagePusher* get_pusher(self):
        return self.pusher.Get()

    def queue_size(self):
        return self.pusher.Get().QueueSize()

    def allocated_in_back(self):
        return self.pusher.Get().AllocatedInBack()


cdef class ZInMemoryMessagePusher(AbstractMessagePusher):
    cdef THolder[TInMemoryMessagePusher] pusher

    def __cinit__(self):
        self.pusher.Reset(new TInMemoryMessagePusher())

    cdef IMessagePusher* get_pusher(self):
        return self.pusher.Get()

    def clean(self):
        self.pusher.Get().Clear()

    def get_messages(self):
        return [msg for msg in self.pusher.Get().GetMessages()]


RequesterPipelineStatsAndMessages = namedtuple("RequesterPipelineStatsAndMessages", [
    "deserialize_errors",
    "aggregate_errors",
    "invalid_response_hosts",
    "serialize_errors",
    "ignored_signals",
    "non_proto_responses",
    "hosts_success"
])


cdef class ZRequesterPipeline:
    cdef TMaybe[TRequesterPipeline] pipeline

    def __cinit__(self, ZYasmConf conf, pushers,
                  ZAggregationRules aggregation_rules, ZCommonRules common_rules,
                  group, middle_count):
        cdef ui64 middle_count_int = middle_count
        cdef TStringBuf group_str = group
        cdef TVector[IMessagePusher*] pusher_vector
        cdef AbstractMessagePusher typed_pusher
        for pusher in pushers:
            typed_pusher = pusher
            pusher_vector.push_back(typed_pusher.get_pusher())
        self.pipeline.ConstructInPlace(
            conf.yasmConf[0],
            pusher_vector,
            aggregation_rules.rules.GetRef(),
            common_rules.rules.GetRef(),
            group_str,
            middle_count_int
        )

    def set_subscriptions(self, subs):
        self.pipeline.GetRef().SetSubscriptions(<PyObject*> subs)

    def set_time(self, timestamp):
        cdef TInstant time_instance = TInstant.Seconds(timestamp)
        self.pipeline.GetRef().SetTime(time_instance)

    def set_metric_manager(self, ZTaggedMetricManager metric_manager):
        self.pipeline.GetRef().SetMetricManager(metric_manager.metrics)

    def add_host_response(self, hostname, response, content_type):
        cdef TStringBuf hostname_str = hostname
        cdef TString response_str = response
        cdef TString content_type_str = content_type
        self.pipeline.GetRef().AddHostResponse(hostname_str, response_str, content_type_str)

    def clean(self):
        with nogil:
            self.pipeline.GetRef().Clean()

    def finish(self):
        cdef TRequesterPipelineStatsAndMessages statsAndMessages
        with nogil:
            self.pipeline.GetRef().Finish(&statsAndMessages)
        return RequesterPipelineStatsAndMessages(
            deserialize_errors = statsAndMessages.DeserializeErrors,
            aggregate_errors = statsAndMessages.AggregateErrors,
            invalid_response_hosts = statsAndMessages.InvalidResponseHosts,
            serialize_errors = statsAndMessages.SerializeErrors,
            ignored_signals = statsAndMessages.IgnoredSignals,
            non_proto_responses = statsAndMessages.NonProtoResponses,
            hosts_success = statsAndMessages.HostsSuccess
        )

    def get_middle_messages(self):
        return self.pipeline.GetRef().GetMiddleMessages()

    def get_server_message(self):
        return self.pipeline.GetRef().GetServerMessage()


cdef class ZMiddlePipeline:
    cdef TMaybe[TMiddlePipeline] pipeline

    def __cinit__(self, ZYasmConf conf, AbstractMessagePusher pusher, group_name, timestamp):
        cdef TString group_name_str = group_name
        cdef TInstant time_instance = TInstant.Seconds(timestamp)
        self.pipeline.ConstructInPlace(conf.yasmConf[0], pusher.get_pusher()[0], group_name_str, time_instance)

    def set_subscriptions(self, subs):
        self.pipeline.GetRef().SetSubscriptions(<PyObject*> subs)

    def send_tsdb_messages(self):
        with nogil:
            self.pipeline.GetRef().SendTSDBMessages()

    def clean(self):
        with nogil:
            self.pipeline.GetRef().Clean()

    def finish(self):
        with nogil:
            self.pipeline.GetRef().Finish()

    def get_server_message(self):
        return self.pipeline.GetRef().GetServerMessage()

    def mul_tagged_record(self, ZTaggedRecord tagged_record, ZTaggedMetricManager metric_manager):
        if tagged_record.tagged_record != NULL:
            with nogil:
                self.pipeline.GetRef().Mul(tagged_record.tagged_record[0], metric_manager.metrics)


cdef class ZServerPipeline:
    cdef TMaybe[TServerPipeline] pipeline

    def __cinit__(self, aggr_name):
        cdef TString aggr_name_str = aggr_name
        self.pipeline.ConstructInPlace(aggr_name_str)

    def set_subscriptions(self, subs):
        self.pipeline.GetRef().SetSubscriptions(<PyObject*> subs)

    def clean(self):
        with nogil:
            self.pipeline.GetRef().Clean()

    def finish(self):
        cdef TServerStats stats;
        with nogil:
            stats = self.pipeline.GetRef().Finish();
        return {
            "key_set_power_max": stats.MaxKeySetPower,
            "key_set_power_sum": stats.KeySetPowerSum,
            "resolved_key_set_power_max": stats.MaxResolvedKeySetPower,
            "resolved_key_set_power_sum": stats.ResolvedKeySetPowerSum
        }

    def get_requested_points_message(self, iteration_ts):
        cdef TInstant iteration_ts_instant = TInstant.Seconds(iteration_ts)
        return self.pipeline.GetRef().GetRequestedPointsMessage(iteration_ts_instant)

    def mul_tagged_record(self, host_name, ZTaggedRecord tagged_record):
        cdef TString host_name_str = host_name
        if tagged_record.tagged_record != NULL:
            with nogil:
                self.pipeline.GetRef().Mul(host_name_str, tagged_record.tagged_record[0])


cdef class ZServerWorker:
    cdef TServerWorker* worker

    def __cinit__(self, aggr_name, worker_count, subscription_port, threads):
        self.worker = new TServerWorker(<TString>aggr_name, <size_t>worker_count, <ui16>subscription_port, <size_t>threads)

    def process(self, iteration_ts, worker, data):
        self.worker.Process(TInstant.Seconds(iteration_ts), <TString>worker, <TString>data)

    def realtime_delays(self):
        return [d.MicroSeconds() / 1000000.0 for d in self.worker.RealtimeDelays()]

    def __dealloc__(self):
        del self.worker


cdef class ZSignalName:
    cdef TSignalName* signal_name

    def __cinit__(self, name):
        self.signal_name = new TSignalName(<TStringBuf>name)

    def get_aggregation_rules(self):
        rules = self.signal_name.GetAggregationRules()
        if rules != NULL:
            return dict(rules.GetFormattedAccumulatorTypes())
        else:
            return None

    def __dealloc__(self):
        del self.signal_name


def debug_value(value):
    cdef TValue* v = PyToValue(<PyObject*> value, True)
    try:
        res = <object>ValueToPy(v.GetValue(), True)
        Py_DECREF(res)
        return res
    finally:
        del v

UnistatEndpoint = namedtuple("UnistatEndpoint", ("port", "path", "prefix", "labels", "container_fqdn"))

SubagentEndpoint = namedtuple("SubagentEndpoint", ("port", "container_fqdn"))

PodInfo = namedtuple("PodInfo", ("container_id", "prefix", "labels", "container_fqdn"))

WorkloadInfo = namedtuple("WorkloadInfo", ("container_id", "prefix", "labels", "container_fqdn"))

ContainerInfo = namedtuple("ContainerInfo", ("container_id", "labels", "container_fqdn"))

cdef class ZNodeAgentResponseLoader:
    cdef THolder[TNodeAgentResponseLoader] loader
    cdef list unistat_endpoints, subagent_endpoints, pods_info, workloads_info, system_containers_info

    def __cinit__(self):
        self.loader.Reset(new TNodeAgentResponseLoader())

    def __init__(self):
        self.unistat_endpoints = []
        self.subagent_endpoints = []
        self.pods_info = []
        self.workloads_info = []
        self.system_containers_info = []

    def load(self, protobuf_response):
        cdef TStringBuf response_to_parse = protobuf_response
        with nogil:
            self.loader.Get().Load(response_to_parse)

        unistat_endpoints = []
        for c_unistat_endpoint in self.loader.Get().ExtractUnistatEndpoints():
            unistat_endpoints.append(
                UnistatEndpoint(
                    c_unistat_endpoint.Port,
                    c_unistat_endpoint.Path,
                    c_unistat_endpoint.Prefix,
                    c_unistat_endpoint.Labels,
                    c_unistat_endpoint.ContainerFqdn,
                )
            )

        subagent_endpoints = []
        for c_subagent_endpoint in self.loader.Get().ExtractSubagentEndpoints():
            subagent_endpoints.append(
                SubagentEndpoint(
                    c_subagent_endpoint.Port,
                    c_subagent_endpoint.ContainerFqdn,
                )
            )

        pods_info = []
        for c_pod_info in self.loader.Get().ExtractPodsInfo():
            pods_info.append(
                PodInfo(
                    c_pod_info.ContainerId,
                    c_pod_info.Prefix,
                    c_pod_info.Labels,
                    c_pod_info.ContainerFqdn
                )
            )

        workloads_info = []
        for c_workload_info in self.loader.Get().ExtractWorkloadsInfo():
            workloads_info.append(
                WorkloadInfo(
                    c_workload_info.ContainerId,
                    c_workload_info.Prefix,
                    c_workload_info.Labels,
                    c_workload_info.ContainerFqdn
                )
            )

        system_containers_info = []
        for c_container_info in self.loader.Get().ExtractSystemContainersInfo():
            system_containers_info.append(
                ContainerInfo(
                    c_container_info.ContainerId,
                    c_container_info.Labels,
                    c_container_info.ContainerFqdn
                )
            )

        self.unistat_endpoints = unistat_endpoints
        self.subagent_endpoints = subagent_endpoints
        self.pods_info = pods_info
        self.workloads_info = workloads_info
        self.system_containers_info = system_containers_info

    def get_unistat_endpoints(self):
        return self.unistat_endpoints

    def get_subagent_endpoints(self):
        return self.subagent_endpoints

    def get_pods_info(self):
        return self.pods_info

    def get_workloads_info(self):
        return self.workloads_info

    def get_system_containers_info(self):
        return self.system_containers_info


cdef class ZRe2Validator:

    cdef TMaybe[TRe2Validator] validator

    def __cinit__(self, pattern):
        cdef TStringBuf c_pattern = pattern
        self.validator.ConstructInPlace(c_pattern)

    def is_ok(self):
        return self.validator.GetRef().IsOk()

    def get_error(self):
        return self.validator.GetRef().GetError()
