#include "history.h"
#include "common.h"

#include <infra/yasm/zoom/components/serialization/history/history.h>
#include <infra/yasm/zoom/components/serialization/python/zoom_to_py.h>
#include <infra/yasm/zoom/components/serialization/python/py_to_zoom.h>

#include <library/cpp/pybind/cast.h>

#include <util/datetime/base.h>
#include <util/stream/mem.h>

using namespace NZoom::NPython;
using namespace NZoom::NHost;
using namespace NZoom::NSubscription;
using namespace NZoom::NSignal;
using namespace NZoom::NProtobuf;
using namespace NZoom::NAggregators;
using namespace NZoom::NYasmConf;
using namespace NZoom::NContainers;
using google::protobuf::Arena;
using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

namespace {
    static constexpr TStringBuf START_TIME_FIELD(TStringBuf("st"));
    static constexpr TStringBuf END_TIME_FIELD(TStringBuf("et"));
    static constexpr TStringBuf SERIES_START_TIME_FIELD(TStringBuf("series_st"));
    static constexpr TStringBuf PERIOD_FIELD(TStringBuf("period"));
    static constexpr TStringBuf HOST_FIELD(TStringBuf("host"));
    static constexpr TStringBuf HOSTS_FIELD(TStringBuf("hosts"));
    static constexpr TStringBuf TAG_FIELD(TStringBuf("tag"));
    static constexpr TStringBuf TAG_SIGNALS_FIELD(TStringBuf("tag_signals"));
    static constexpr TStringBuf SIGNAL_FIELD(TStringBuf("signal"));
    static constexpr TStringBuf VALUES_FIELD(TStringBuf("values"));
    static constexpr TStringBuf STATUS_CODES_FIELD(TStringBuf("status_codes"));
    static constexpr TStringBuf STATUS_CODE_FIELD(TStringBuf("status_code"));
    static constexpr TStringBuf REPLICA_INDEX_FIELD(TStringBuf("single_replica"));
    static constexpr TDuration MINIMAL_RESOLUTION = TDuration::Seconds(5);

    ssize_t CastToInt(PyObject* value) {
        if (PyInt_Check(value)) {
            return PyInt_AsSsize_t(value);
        } else {
            ythrow yexception() << "integer expected";
        }
    }

    TString CastToString(PyObject* value) {
        TString str(ObjectToString(value));
        if (str) {
            return str;
        } else {
            ythrow yexception() << "non-empty string expected";
        }
    }

    template <class F>
    void VisitDict(PyObject* root, F&& cb) {
        if (!IteratePyDict(root, std::forward<F>(cb))) {
            ythrow yexception() << "dict expected";
        }
    }

    template <class F>
    void VisitList(PyObject* root, F&& cb) {
        if (!IteratePyList(root, std::forward<F>(cb))) {
            ythrow yexception() << "list expected";
        }
    }

    template <class F>
    void ExtractRequests(PyObject* requests, F&& cb) {
        VisitList(requests, [&](PyObject* request) {
            TInstant start;
            TInstant end;
            TDuration period;
            TVector<THostName> hostNames;
            TReplicaIndex replicaIndex;
            TVector<std::pair<TInternedRequestKey, TVector<TSignalName>>> tagSignals;
            VisitDict(request, [&](const TString& field, PyObject* value) {
                if (field == START_TIME_FIELD) {
                    start = TInstant::Seconds(CastToInt(value));
                } else if (field == END_TIME_FIELD) {
                    end = TInstant::Seconds(CastToInt(value));
                } else if (field == PERIOD_FIELD) {
                    period = TDuration::Seconds(CastToInt(value));
                } else if (field == HOSTS_FIELD) {
                    VisitList(value, [&hostNames](PyObject* host) {
                        hostNames.emplace_back(CastToString(host));
                    });
                } else if (field == TAG_SIGNALS_FIELD) {
                    VisitDict(value, [&tagSignals](const TString& tags, PyObject* signals) {
                        TInternedRequestKey requestKey(tags);
                        TVector<TSignalName> signalNames;
                        VisitList(signals, [&signalNames](PyObject* signal) {
                            signalNames.emplace_back(CastToString(signal));
                        });
                        tagSignals.emplace_back(requestKey, std::move(signalNames));
                    });
                } else if (field == REPLICA_INDEX_FIELD) {
                    if (value != Py_None) {
                        int replicaIndexValue = CastToInt(value);
                        if (replicaIndexValue != 0 && replicaIndexValue != 1) {
                            ythrow yexception() << "invalid " << field << " given";
                        }
                        replicaIndex.ConstructInPlace(replicaIndexValue);
                    }
                } else {
                    ythrow yexception() << "unknown field " << field << " given";
                }
            });

            if (!start || !end || !period) {
                ythrow yexception() << "some fields not specified";
            } else if (start > end) {
                ythrow yexception() << "start should be less then or equal to end";
            } else if (period.GetValue() % MINIMAL_RESOLUTION.GetValue() || period < MINIMAL_RESOLUTION) {
                ythrow yexception() << "period should be greather then " << MINIMAL_RESOLUTION << " and should multiple of it";
            }

            for (const auto& [requestKey, signals] : tagSignals) {
                for (const auto& signalName : signals) {
                    for (const auto& hostName : hostNames) {
                        cb(THistoryRequest{
                            .Start = start,
                            .End = end,
                            .Period = period,
                            .HostName = hostName,
                            .RequestKey = requestKey,
                            .SignalName = signalName,
                            .ReplicaIndex = replicaIndex
                        });
                    }
                }
            }
        });
    }

    NPyBind::TPyObjectPtr ToPython(TSignalName signalName) {
        const auto& name(signalName.GetName());
        return NPyBind::TPyObjectPtr(PyString_FromStringAndSize(name.data(), name.size()), true);
    }

    NPyBind::TPyObjectPtr ToPython(TInternedRequestKey requestKey) {
        const auto& name(requestKey.GetRequestKey().ToNamed());
        return NPyBind::TPyObjectPtr(PyString_FromStringAndSize(name.data(), name.size()), true);
    }

    NPyBind::TPyObjectPtr ToPython(THostName hostName) {
        const auto& name(hostName.GetName());
        return NPyBind::TPyObjectPtr(PyString_FromStringAndSize(name.data(), name.size()), true);
    }

    NPyBind::TPyObjectPtr ToPython(TString str) {
        return NPyBind::TPyObjectPtr(PyString_FromStringAndSize(str.data(), str.size()), true);
    }

    NPyBind::TPyObjectPtr ToPython(size_t value) {
        return NPyBind::TPyObjectPtr(PyInt_FromSize_t(value), true);
    }
}

TString THistoryApiRequestSerializer::RequestsToProto(PyObject* requests, TInstant deadline, const TString& requestId) {
    Arena arena;
    auto* result = Arena::CreateMessage<NYasm::NInterfaces::NInternal::THistoryReadAggregatedRequest>(&arena);

    if (deadline) {
        result->SetDeadlineTimestamp(deadline.Seconds());
    }
    if (requestId) {
        result->SetRequestId(requestId);
    }

    THistoryRequestWriter writer(*result);
    ExtractRequests(requests, [&](THistoryRequest request) {
        writer.Add(request);
    });

    return result->SerializeAsString();
}

TString THistoryApiRequestSerializer::ResponsesToProto(const TVector<NZoom::NProtobuf::THistoryResponse*>& responses) {
    Arena arena;
    auto* result = Arena::CreateMessage<NYasm::NInterfaces::NInternal::THistoryReadAggregatedResponse>(&arena);

    THistoryResponseWriter writer(*result);
    for (const auto* response : responses) {
        writer.Add(*response);
    }

    return result->SerializeAsString();
}

THistoryApiRequestDeserializer::THistoryApiRequestDeserializer()
    : Iterator(Requests.begin())
{
}

void THistoryApiRequestDeserializer::Load(TStringBuf content) {
    Arena arena;
    auto* message = Arena::CreateMessage<NYasm::NInterfaces::NInternal::THistoryReadAggregatedRequest>(&arena);
    TMemoryInput stream(content);
    message->ParseFromArcadiaStream(&stream);
    Requests = THistoryRequest::FromProto(*message);
    Iterator = Requests.begin();
}

THistoryRequest* THistoryApiRequestDeserializer::Next() {
    auto request(MakeHolder<THistoryRequest>(std::move(*Iterator)));
    ++Iterator;
    return request.Release();
}

bool THistoryApiRequestDeserializer::IsEnd() const {
    return Iterator == Requests.end();
}

THistoryApiResponseDeserializer::THistoryApiResponseDeserializer()
    : Iterator(Responses.begin())
{
}

TVector<int> THistoryApiResponseDeserializer::Load(TStringBuf content) {
    Arena arena;
    auto* message = Arena::CreateMessage<NYasm::NInterfaces::NInternal::THistoryReadAggregatedResponse>(&arena);
    TMemoryInput stream(content);
    message->ParseFromArcadiaStream(&stream);
    auto newResponses = THistoryResponse::FromProto(*message);
    Responses.reserve(Responses.size() + newResponses.size());
    TVector<int> codes(Reserve(newResponses.size()));
    for (const auto& response: newResponses) {
        codes.push_back(response.StatusCode);
    }
    std::move(std::begin(newResponses), std::end(newResponses), std::back_inserter(Responses));
    Iterator = Responses.begin();
    return codes;
}

void THistoryApiResponseDeserializer::Merge() {
    Responses = THistoryResponse::MergeMultiple(Responses);
    Iterator = Responses.begin();
}

THistoryResponse* THistoryApiResponseDeserializer::Next() {
    auto response = MakeHolder<THistoryResponse>(std::move(*Iterator));
    ++Iterator;
    return response.Release();
}

bool THistoryApiResponseDeserializer::IsEnd() const {
    return Iterator == Responses.end();
}

THistoryResponse* THistoryApiResponseDeserializer::FromPython(PyObject* response) {
    TMaybe<TInstant> start;
    TMaybe<TInstant> seriesStart;
    TMaybe<TInstant> end;
    TMaybe<TDuration> period;
    TMaybe<THostName> hostName;
    TMaybe<TInternedRequestKey> requestKey;
    TMaybe<TSignalName> signalName;
    TMaybe<EStatusCode> statusCode;
    TVector<NZoom::NValue::TValue> values;
    TReplicaIndex replicaIndex;

    VisitDict(response, [&](const TString& field, PyObject* value) {
        if (field == START_TIME_FIELD) {
            start.ConstructInPlace(TInstant::Seconds(CastToInt(value)));
        } else if (field == SERIES_START_TIME_FIELD) {
            seriesStart.ConstructInPlace(TInstant::Seconds(CastToInt(value)));
        } else if (field == END_TIME_FIELD) {
            end.ConstructInPlace(TInstant::Seconds(CastToInt(value)));
        } else if (field == PERIOD_FIELD) {
            period.ConstructInPlace(TDuration::Seconds(CastToInt(value)));
        } else if (field == HOST_FIELD) {
            hostName.ConstructInPlace(CastToString(value));
        } else if (field == TAG_FIELD) {
            requestKey.ConstructInPlace(CastToString(value));
        } else if (field == SIGNAL_FIELD) {
            signalName.ConstructInPlace(CastToString(value));
        } else if (field == STATUS_CODE_FIELD) {
            statusCode.ConstructInPlace(static_cast<EStatusCode>(CastToInt(value)));
        } else if (field == VALUES_FIELD) {
            VisitList(value, [&values](PyObject* val) {
                values.emplace_back(PyToOwnedValue(val, true));
            });
        } else if (field == REPLICA_INDEX_FIELD) {
            if (value != Py_None) {
                int replicaIndexValue = CastToInt(value);
                if (replicaIndexValue != 0 && replicaIndexValue != 1) {
                    ythrow yexception() << "invalid " << field << " given";
                }
                replicaIndex.ConstructInPlace(replicaIndexValue);
            }
        } else {
            ythrow yexception() << "unknown field " << field << " given";
        }
    });

    if (seriesStart.Empty()) {
        seriesStart = start.GetRef();
    }
    if (statusCode.Empty()) {
        statusCode.ConstructInPlace(EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
    }
    THistoryRequest request{
        .Start = start.GetRef(),
        .End = end.GetRef(),
        .Period = period.GetRef(),
        .HostName = hostName.GetRef(),
        .RequestKey = requestKey.GetRef(),
        .SignalName = signalName.GetRef(),
        .ReplicaIndex = replicaIndex
    };
    return new THistoryResponse(request, seriesStart.GetRef(), std::move(values), statusCode.GetRef());
}

THistoryAggregatorWrapper::THistoryAggregatorWrapper(const TYasmConf& conf)
    : Aggregator(conf)
{
}

void THistoryAggregatorWrapper::AddRequests(PyObject* requests) {
    ExtractRequests(requests, [this](THistoryRequest request) {
        Aggregator.AddRequest(request);
    });
}

void THistoryAggregatorWrapper::AddMetagroups(PyObject* metagroups) {
    VisitDict(metagroups, [this](const TString& metagroup, PyObject* val) {
        TVector<THostName> groups;
        VisitList(val, [&groups](PyObject* val) {
            groups.emplace_back(CastToString(val));
        });
        Aggregator.AddMetagroup(metagroup, groups);
    });
}

void THistoryAggregatorWrapper::Mul(const THistoryResponse& response) {
    Aggregator.Mul(response);
}

void THistoryAggregatorWrapper::Overwrite(const THistoryAggregatorWrapper& other) {
    Aggregator.Overwrite(other.Aggregator);
}

void THistoryAggregatorWrapper::OverwriteSince(const THistoryAggregatorWrapper& other, TInstant since, TInstant until) {
    Aggregator.OverwriteSince(other.Aggregator, since, until);
}

void THistoryAggregatorWrapper::OverwriteContinuous(const THistoryAggregatorWrapper& other) {
    Aggregator.OverwriteContinuous(other.Aggregator);
}

void THistoryAggregatorWrapper::Visit(IHistoryAggregatorVisitor& visitor) const {
    Aggregator.Visit(visitor);
}

THistoryAggregatorToPythonVisitor::THistoryAggregatorToPythonVisitor()
    : ResultHolder(NPyBind::TPyObjectPtr(PyList_New(0), true))
    , SignalName(NPyBind::BuildPyObject(SIGNAL_FIELD), true)
    , RequestKey(NPyBind::BuildPyObject(TAG_FIELD), true)
    , HostName(NPyBind::BuildPyObject(HOST_FIELD), true)
    , StatusCodes(NPyBind::BuildPyObject(STATUS_CODES_FIELD), true)
    , StartTime(NPyBind::BuildPyObject(START_TIME_FIELD), true)
    , EndTime(NPyBind::BuildPyObject(END_TIME_FIELD), true)
    , SeriesStartTime(NPyBind::BuildPyObject(SERIES_START_TIME_FIELD), true)
    , Period(NPyBind::BuildPyObject(PERIOD_FIELD), true)
    , Values(NPyBind::BuildPyObject(VALUES_FIELD), true)
    , ConvertedSignalsHolder(NPyBind::TPyObjectPtr(PyList_New(0), true))
    , AllowLegacyTypes(true)
{
}

void THistoryAggregatorToPythonVisitor::OnRequest(const THistoryRequest& request) {
    SeriesHolder = NPyBind::TPyObjectPtr(PyDict_New(), true);
    StatusCodesHolder = NPyBind::TPyObjectPtr(PyDict_New(), true);
    CurrentRequestSmallHgramConverted = false;

    PyDict_SetItem(SeriesHolder.Get(), SignalName.Get(), ToPython(request.SignalName).Get());
    PyDict_SetItem(SeriesHolder.Get(), RequestKey.Get(), ToPython(request.RequestKey).Get());
    PyDict_SetItem(SeriesHolder.Get(), HostName.Get(), ToPython(request.HostName).Get());
    PyDict_SetItem(SeriesHolder.Get(), StartTime.Get(), ToPython(request.Start.Seconds()).Get());
    PyDict_SetItem(SeriesHolder.Get(), EndTime.Get(), ToPython(request.End.Seconds()).Get());
    PyDict_SetItem(SeriesHolder.Get(), Period.Get(), ToPython(request.Period.Seconds()).Get());

    CurrentRequestSignalNameHolder = NPyBind::TPyObjectPtr(PyList_New(0), true);
    PyList_Append(CurrentRequestSignalNameHolder.Get(), ToPython(request.RequestKey.GetName()).Get());
    PyList_Append(CurrentRequestSignalNameHolder.Get(), ToPython(request.SignalName).Get());

}

void THistoryAggregatorToPythonVisitor::OnStatusCode(THostName hostName, EStatusCode statusCode) {
    if (statusCode != EStatusCode::THistoryAggregatedSeries_EStatusCode_OK && statusCode != EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND) {
        PyDict_SetItem(StatusCodesHolder.Get(), ToPython(hostName).Get(), ToPython(statusCode).Get());
    }
}

void THistoryAggregatorToPythonVisitor::OnContainer(const TTimestampedSeriesContainer& container) {
    container.Visit(*this);
}

void THistoryAggregatorToPythonVisitor::OnStartTime(TInstant startTime) {
    PyDict_SetItem(SeriesHolder.Get(), SeriesStartTime.Get(), ToPython(startTime.Seconds()).Get());
}

void THistoryAggregatorToPythonVisitor::OnValueCount(size_t count) {
    ValuesHolder = NPyBind::TPyObjectPtr(PyList_New(count), true);
}

void THistoryAggregatorToPythonVisitor::OnValue(NZoom::NValue::TValueRef value) {
    PyObject* pyObject;
    TNormalHgramSizeGetter normalHgramSizeGetter;
    if (!AllowLegacyTypes && value.GetType() == NValue::EValueType::SMALL_HGRAM) {
        NZoom::NAccumulators::TAccumulator convertingAccumulator(NAccumulators::EAccumulatorType::Hgram, false);
        convertingAccumulator.Mul(value);
        NZoom::NValue::TValueRef convertedValue = convertingAccumulator.GetValue();
        convertedValue.Update(normalHgramSizeGetter);
        pyObject = ValueToPy(convertedValue, true);
    } else {
        value.Update(normalHgramSizeGetter);
        pyObject = ValueToPy(value, true);
    }
    TMaybe<size_t> normalHgramSize = normalHgramSizeGetter.GetNormalHgramNonZeroValuesCount();
    if (!AllowLegacyTypes && normalHgramSize.Defined()) {
        CurrentRequestSmallHgramConverted |= normalHgramSize > 0 && normalHgramSize <= NHgram::MAX_SMALL_HGRAM_SIZE;
    }
    PyList_SET_ITEM(ValuesHolder.Get(), ValuePosition, pyObject);
    ++ValuePosition;
}

void THistoryAggregatorToPythonVisitor::OnFinish() {
    PyDict_SetItem(SeriesHolder.Get(), StatusCodes.Get(), StatusCodesHolder.Get());
    StatusCodesHolder.Reset();

    PyDict_SetItem(SeriesHolder.Get(), Values.Get(), ValuesHolder.Get());
    ValuesHolder.Reset();
    ValuePosition = 0;

    PyList_Append(ResultHolder.Get(), SeriesHolder.Get());
    SeriesHolder.Reset();

    if (CurrentRequestSmallHgramConverted) {
        PyList_Append(ConvertedSignalsHolder.Get(), CurrentRequestSignalNameHolder.Get());
    }
    CurrentRequestSignalNameHolder.Reset();
}

PyObject* THistoryAggregatorToPythonVisitor::GetConvertedSmallSignals() {
    return ConvertedSignalsHolder.RefGet();
}

PyObject* THistoryAggregatorToPythonVisitor::GetResponses() {
    return ResultHolder.RefGet();
}
