#pragma once

#include <infra/yasm/zoom/components/serialization/history/history.h>
#include <infra/yasm/zoom/components/aggregators/history.h>

#include <library/cpp/pybind/ptr.h>

#include <util/generic/string.h>

#include <Python.h>

namespace NZoom::NPython {
    class THistoryApiRequestSerializer {
    public:
        /* Builds a history api requests from list of dictionaries like this:
         * {"st": 1549527445,
         *  "et": 1549528935,
         *  "period": 5,
         *  "tag_signals": {"itype=upper;geo=sas": ["accesslog-2xx_summ"]},
         *  "hosts": ["IVA.005", "VLA.027.0"]}
         **/
        static TString RequestsToProto(PyObject* requests, TInstant deadline, const TString& requestId);

        static TString ResponsesToProto(const TVector<NZoom::NProtobuf::THistoryResponse*>& responses);
    };

    class THistoryApiRequestDeserializer {
    public:
        THistoryApiRequestDeserializer();

        void Load(TStringBuf content);

        NZoom::NProtobuf::THistoryRequest* Next();

        bool IsEnd() const;

    private:
        TVector<NZoom::NProtobuf::THistoryRequest> Requests;
        TVector<NZoom::NProtobuf::THistoryRequest>::iterator Iterator;
    };

    class THistoryApiResponseDeserializer {
    public:
        THistoryApiResponseDeserializer();

        TVector<int> Load(TStringBuf content);
        void Merge();

        NZoom::NProtobuf::THistoryResponse* Next();

        bool IsEnd() const;

        static NZoom::NProtobuf::THistoryResponse* FromPython(PyObject* response);

    private:
        TVector<NZoom::NProtobuf::THistoryResponse> Responses;
        TVector<NZoom::NProtobuf::THistoryResponse>::iterator Iterator;
    };

    class THistoryAggregatorWrapper {
    public:
        THistoryAggregatorWrapper(const NZoom::NYasmConf::TYasmConf& conf);

        void AddRequests(PyObject* requests);
        void AddMetagroups(PyObject* metagroups);

        void Mul(const NZoom::NProtobuf::THistoryResponse& response);
        void Overwrite(const THistoryAggregatorWrapper& other);
        void OverwriteSince(const THistoryAggregatorWrapper& other, TInstant since, TInstant until);
        void OverwriteContinuous(const THistoryAggregatorWrapper& other);
        void Visit(NZoom::NAggregators::IHistoryAggregatorVisitor& visitor) const;

    private:
        NZoom::NAggregators::THistoryAggregator Aggregator;
    };

    class TNormalHgramSizeGetter: public NZoom::NValue::IUpdatable, public NHgram::IHgramStorageCallback {
    public:
        TNormalHgramSizeGetter() {
        }

        void MulNone() override final {}
        void MulFloat(const double) override final {}
        void MulVec(const TVector<double>&) override final {}
        void MulCountedSum(const double, const ui64) override final {}
        void MulHgram(const NHgram::THgram& value) override final {
            value.Store(*this);
        }

        void OnStoreSmall(const TVector<double>&, const size_t) override final {}
        void OnStoreUgram(const NHgram::TUgramBuckets&) override final {}
        void OnStoreNormal(const TVector<double>& values, const size_t, const i16) override final {
            double count = 0;
            for (double value : values) {
                count += value;
            }
            NormalHgramNonZeroValuesCount = count;
        }

        TMaybe<size_t> GetNormalHgramNonZeroValuesCount() const {
            return NormalHgramNonZeroValuesCount;
        }
    private:
        TMaybe<size_t> NormalHgramNonZeroValuesCount;
    };

    class THistoryAggregatorToPythonVisitor final : public NZoom::NAggregators::IHistoryAggregatorVisitor, public NZoom::NContainers::ITimestampedSeriesContainerVisitor {
    public:
        THistoryAggregatorToPythonVisitor();

        void OnRequest(const NZoom::NProtobuf::THistoryRequest& request) override;
        void OnStatusCode(NZoom::NHost::THostName hostName, NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode statusCode) override;
        void OnContainer(const NZoom::NContainers::TTimestampedSeriesContainer& container) override;
        void OnStartTime(TInstant startTime) override;
        void OnValueCount(size_t count) override;
        void OnValue(NZoom::NValue::TValueRef value) override;
        void OnFinish() override;

        void SetAllowLegacyTypes(bool allowLegacyTypes) {
            AllowLegacyTypes = allowLegacyTypes;
        }
        PyObject* GetConvertedSmallSignals();
        PyObject* GetResponses();

    private:
        NPyBind::TPyObjectPtr ResultHolder;
        NPyBind::TPyObjectPtr SeriesHolder;
        NPyBind::TPyObjectPtr ValuesHolder;
        NPyBind::TPyObjectPtr StatusCodesHolder;

        NPyBind::TPyObjectPtr SignalName;
        NPyBind::TPyObjectPtr RequestKey;
        NPyBind::TPyObjectPtr HostName;
        NPyBind::TPyObjectPtr StatusCodes;
        NPyBind::TPyObjectPtr StartTime;
        NPyBind::TPyObjectPtr EndTime;
        NPyBind::TPyObjectPtr SeriesStartTime;
        NPyBind::TPyObjectPtr Period;
        NPyBind::TPyObjectPtr Values;

        NPyBind::TPyObjectPtr ConvertedSignalsHolder;
        NPyBind::TPyObjectPtr CurrentRequestSignalNameHolder;

        bool AllowLegacyTypes;
        bool CurrentRequestSmallHgramConverted;

        size_t ValuePosition = 0;
    };
}
