#include "message_pusher.h"

using namespace NZoom::NPython;

TMessagePusher::TMessagePusher(TStringBuf url)
    : Url(url)
{
}

void TMessagePusher::AddMessage(TInstant, TStringBuf message) {
    Queue.emplace_back(message);
}

void TMessagePusher::Finish() {
    while (Queue) {
        auto options = NHttp::TFetchOptions()
                           .SetTimeout(TDuration::Seconds(5))
                           .SetPostData(Queue.back())
                           .SetContentType("application/x-protobuf");
        Queue.pop_back();
        NHttp::TFetchQuery query(Url, options);

        auto reply = NHttp::Fetch(query);
        if (!reply->Success()) {
            ythrow yexception() << "Could not send to " << reply->RequestUrl << ". Result " << reply->Code << " " << reply->Data;
        }
    }
};

TAsyncMessagePusher::TAsyncMessagePusher(TStringBuf url, TDuration period)
    : Url(url)
    , Period(period)
{
}

void TAsyncMessagePusher::RemoveOutdated() {
    auto maxTime = Messages.back().Time;
    while (!Messages.empty() && (maxTime - Messages.front().Time > Period)) {
        Messages.pop_front();
    }
}

void TAsyncMessagePusher::SendMessagesImpl() {
    auto guard = Guard(Lock);
    if (CurrentRequest.Defined() || Messages.empty()) {
        return;
    }

    RemoveOutdated();

    if (Messages.empty()) {
        return;
    }

    auto options = NHttp::TFetchOptions()
                       .SetTimeout(TDuration::Seconds(5))
                       .SetPostData(TString{Messages.front().Message})
                       .SetContentType("application/x-protobuf");
    NHttp::TFetchQuery query(Url, options);

    auto cb = [this](auto result) {
        SendMessagesCallBack(result);
    };

    CurrentRequest = NHttp::FetchAsync(query, cb);
}

bool TAsyncMessagePusher::IsRequestProcessed(NHttpFetcher::TResultRef reply) noexcept {
    auto guard = Guard(Lock);

    Y_ASSERT(CurrentRequest.Defined());
    CurrentRequest.Clear();

    if (reply->Success()) {
        Y_ASSERT(!Messages.empty());
        Messages.pop_front();
        return true;
    } else {
        Cerr << "Could not send to " << reply->RequestUrl << ". Result " << reply->Code << " " << reply->Data
             << "\n Queue size " << Messages.size() << Endl;
        return false;
    }
}

void TAsyncMessagePusher::SendMessagesCallBack(NHttpFetcher::TResultRef reply) noexcept {
    if (IsRequestProcessed(reply)) {
        SendMessagesImpl();
    }
}

void TAsyncMessagePusher::AddMessage(TInstant time, TStringBuf message) {
    auto guard = Guard(Lock);
    Messages.emplace_back(time, message);
}

void TAsyncMessagePusher::Finish() {
    SendMessagesImpl();
}

TAsyncMessagePusher::TMessage::TMessage(TInstant time, TStringBuf message)
    : Time(time)
    , Block(MmapAllocator()->Allocate(message.size()))
    , Message(reinterpret_cast<char*>(Block.Data), message.size())
{
    MemCopy(reinterpret_cast<char*>(Block.Data), message.data(), message.size());
}

TAsyncMessagePusher::TMessage::~TMessage() {
    MmapAllocator()->Release(Block);
}

size_t TAsyncMessagePusher::QueueSize() const {
    auto guard = Guard(Lock);
    return Messages.size();
}

size_t TAsyncMessagePusher::AllocatedInBack() const {
    auto guard = Guard(Lock);
    if (Messages.empty()) {
        return 0;
    }
    return Messages.back().Block.Len;
}

TAsyncMessagePusher::~TAsyncMessagePusher() {
    auto guard = Guard(Lock);
    if (CurrentRequest.Defined()) {
        CurrentRequest->Cancel();
    }
}

void TInMemoryMessagePusher::AddMessage(TInstant, TStringBuf message) {
    Messages.emplace_back(TString{message});
}

void TInMemoryMessagePusher::Finish() {
}

void TInMemoryMessagePusher::Clear() {
    Messages.clear();
}

TVector<TString> TInMemoryMessagePusher::GetMessages() const {
    return Messages;
}
