#pragma once

#include <library/cpp/http/client/client.h>
#include <util/generic/queue.h>
#include <util/generic/vector.h>

#include <util/memory/mmapalloc.h>
#include <util/memory/pool.h>

namespace NZoom {
    namespace NPython {
        class IMessagePusher {
        public:
            virtual void AddMessage(TInstant time, TStringBuf message) = 0;
            virtual void Finish() = 0;

            virtual ~IMessagePusher() = default;
        };

        class TMessagePusher: public IMessagePusher {
        public:
            TMessagePusher(TStringBuf url);
            void AddMessage(TInstant time, TStringBuf message) override;
            void Finish() override;

        private:
            TVector<TString> Queue;
            const TString Url;
        };

        class TAsyncMessagePusher: public IMessagePusher {
        public:
            TAsyncMessagePusher(TStringBuf url, TDuration period);
            void AddMessage(TInstant time, TStringBuf message) override;
            void Finish() override;
            size_t AllocatedInBack() const;
            size_t QueueSize() const;

            ~TAsyncMessagePusher();

        private:
            struct TMessage {
                TMessage(TInstant time, TStringBuf message);
                ~TMessage();

                const TInstant Time;
                const IAllocator::TBlock Block;
                const TStringBuf Message;
            };

            void RemoveOutdated();
            void SendMessagesImpl();

            bool IsRequestProcessed(NHttpFetcher::TResultRef reply) noexcept;
            void SendMessagesCallBack(NHttpFetcher::TResultRef reply) noexcept;

            const TString Url;
            const TDuration Period;

            TDeque<TMessage> Messages;
            TMaybe<NHttp::TFetchState> CurrentRequest;

            TMutex Lock;
        };

        class TInMemoryMessagePusher: public IMessagePusher {
        public:
            void AddMessage(TInstant time, TStringBuf message) override;
            void Finish() override;
            void Clear();
            TVector<TString> GetMessages() const;

        private:
            TVector<TString> Messages;
        };
    }
}
