#include "middle.h"
#include "serializers.h"

using namespace NTags;
using namespace NZoom::NAggregators;
using namespace NZoom::NContainers;
using namespace NZoom::NHost;
using namespace NZoom::NPython;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NSubscription;
using namespace NZoom::NValue;
using namespace NMonitoring;

namespace {
    struct TMiddleVisitor final: public ITagGroupContainerCallback {
        TMiddleVisitor(const THostName& groupName,
                       const TSubscriptionFilter& subscriptions,
                       TTsdbRequestState& tsdbRequestState,
                       msgpack::sbuffer& serverMessage,
                       TInstant time)
            : GroupName(groupName)
            , Subscriptions(subscriptions)
            , ServerMessage(serverMessage)
            , ServerPacker(ServerMessage)
            , TsdbRequestPacker(tsdbRequestState.CreatePacker(time, GroupName))
        {
        }

        void SetObjectsCount(const size_t count) override {
            TsdbRequestPacker.SetCount(count);
        }

        void OnTagContainer(TInstanceKey key, const TGroupContainer& container) override {
            TsdbRequestPacker.AddContainer(key, container);
            WriteServerMessage(key, container);
        }

        void WriteServerMessage(TInstanceKey key, const TGroupContainer& container) {
            TFilteringSignalValueVisitor visitor(Subscriptions, key);
            container.Process(visitor);
            auto signals(visitor.Finish());
            if (!signals.empty()) {
                TTaggedFoundSignals::insert_ctx ctx;
                auto it(Found.find(key, ctx));
                if (it != Found.end()) {
                    for (auto& signal : signals) {
                        it->second.emplace(std::move(signal));
                    }
                } else {
                    Found.emplace_direct(ctx, key, std::move(signals));
                }
            }
        }

        void Finish() {
            PackServerMessage(ServerPacker, GroupName, Found);
        }

        const THostName GroupName;
        const TSubscriptionFilter& Subscriptions;
        msgpack::sbuffer& ServerMessage;
        msgpack::packer<msgpack::sbuffer> ServerPacker;
        TTaggedFoundSignals Found;
        TTsdbRequestPacker TsdbRequestPacker;
    };
}

TMiddlePipeline::TMiddlePipeline(const NZoom::NYasmConf::TYasmConf& conf, IMessagePusher& tsdbPusher, const TString& groupName, TInstant timestamp)
    : GroupName(groupName)
    , Timestamp(timestamp)
    , TsdbPusher(tsdbPusher)
    , GroupAggregator(conf)
{
}

void TMiddlePipeline::Mul(const NZoom::NRecord::TTaggedRecord& taggedRecord,
                          NZoom::NAggregators::TTaggedMetricManager& metricManager) {
    GroupAggregator.Mul(taggedRecord, metricManager);
}

void TMiddlePipeline::SetSubscriptions(const TTagSignals& subscriptions) {
    Subscriptions.Clear();
    for (const auto& tagSignals : subscriptions) {
        Subscriptions.Add(tagSignals.first.GetRequestKey(), tagSignals.second);
    }
}

void TMiddlePipeline::SetSubscriptions(PyObject* value) {
    TVector<TSubscription> subscriptions = DeserializeSubscriptions(value);
    SetSubscriptions(SubscriptionsToTagSignals(subscriptions));
}

void TMiddlePipeline::Clean() {
    Clean(TInstant::Now());
}

void TMiddlePipeline::Clean(const TInstant explicitNow) {
    GroupAggregator.Clean(explicitNow);
    ServerMessage.clear();
    TsdbRequestState.Clear();
    Finished = false;
}

void TMiddlePipeline::Finish() {
    if (Finished) {
        ythrow yexception() << "middle pipeline already finished";
    }
    PrepareMessages();
    Finished = true;
}

TStringBuf TMiddlePipeline::GetServerMessage() const {
    if (!Finished) {
        ythrow yexception() << "middle pipeline not finished yet";
    }
    return {ServerMessage.data(), ServerMessage.size()};
}

void TMiddlePipeline::PrepareMessages() {
    TsdbRequestState.ConstructInPlace();
    TMiddleVisitor visitor(GroupName, Subscriptions, *TsdbRequestState, ServerMessage, Timestamp);
    GroupAggregator.Process(visitor);
    visitor.Finish();
}

void TMiddlePipeline::SendTSDBMessages() {
    if (!TsdbRequestState.Defined()) {
        return;
    }
    auto message(TsdbRequestState->Serialize());
    TsdbPusher.AddMessage(Timestamp, message);
    TsdbPusher.Finish();
}
