#include "requester.h"
#include "serializers.h"

#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>
#include <infra/yasm/zoom/python/responses.h>

#include <solomon/libs/cpp/selectors/matchers.h>

#include <util/generic/hash.h>

using namespace NTags;
using namespace NZoom::NYasmConf;
using namespace NZoom::NAggregators;
using namespace NZoom::NContainers;
using namespace NZoom::NPython;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;
using namespace NZoom::NSubscription;
using namespace NZoom::NHost;
using namespace NMonitoring;

namespace {
    static constexpr TStringBuf COMMON_ITYPE(TStringBuf("common"));

    static const TVector<TStringBuf> TIER_ONLY_ITYPES({
        TStringBuf("disk"),
        TStringBuf("mailmulca"),
        TStringBuf("metrics"),
        TStringBuf("yalite"),
        TStringBuf("zen")
    });

    static const NTags::TInternedTagNameSet HOST_TAG_SET(TVector<TStringBuf>({"host"}));

    inline bool IsTierOnlyItype(TStringBuf itype) {
        return Find(TIER_ONLY_ITYPES.begin(), TIER_ONLY_ITYPES.end(), itype) != TIER_ONLY_ITYPES.end();
    }

    inline bool IsValidInstanceKey(TInstanceKey key) {
        return !key.IsSmallAggregate() || IsTierOnlyItype(key.GetItype());
    }

    inline TTaggedRecord AggregateRecordsByHost(TTaggedRecord& taggedRecord) {
        TVector<std::pair<NTags::TInstanceKey, NZoom::NRecord::TRecord>> values(Reserve(taggedRecord.Len()));
        for (auto& [key, record] : taggedRecord.GetValues()) {
            values.emplace_back(key.AggregateBy(HOST_TAG_SET), std::move(record));
        }
        return TTaggedRecord(std::move(values));
    }

     inline TTaggedRecord AddGroupAndHostAndSort(THostName group, THostName host, TTaggedRecord &taggedRecord) {
        TVector<std::pair<NTags::TInstanceKey, NZoom::NRecord::TRecord>> values(Reserve(taggedRecord.Len()));
        for (auto& [key, record] : taggedRecord.GetValues()) {
            values.emplace_back(key.AddGroupAndHost(group, host), std::move(record));
        }

        Sort(values, [](const auto& lhs, const auto& rhs) {
            return lhs.first.GetHostName() < rhs.first.GetHostName();
        });

        return TTaggedRecord(std::move(values));
    }

    class TTaggedRecordMerger {
    public:
        void Add(NZoom::NRecord::TTaggedRecord& taggedRecord) {
            Values.reserve(taggedRecord.Len() + Values.size());
            for (auto& [key, record] : taggedRecord.GetValues()) {
                MoveValues(record, Values[key]);
            }
        }

        void AddOnlyRaw(NZoom::NRecord::TTaggedRecord& taggedRecord) {
            Values.reserve(taggedRecord.Len() + Values.size());
            for (auto& [key, record] : taggedRecord.GetValues()) {
                if (IsValidInstanceKey(key)) {
                    MoveValues(record, Values[key]);
                }
            }
        }

        TTaggedRecord Create() {
            TVector<std::pair<TInstanceKey, TRecord>> taggedRecord;
            taggedRecord.reserve(Values.size());
            for (auto& [key, values] : Values) {
                TVector<std::pair<TSignalName, TValue>> record;
                record.reserve(values.size());
                for (auto& [signal, value] : values) {
                    record.emplace_back(std::move(signal), std::move(value));
                }
                taggedRecord.emplace_back(std::move(key), std::move(record));
            }
            return TTaggedRecord(std::move(taggedRecord));
        }

    private:
        void MoveValues(NZoom::NRecord::TRecord& source, THashMap<TSignalName, TValue>& dest) {
            dest.reserve(source.Len() + dest.size());
            for (auto& [signal, value] : source.GetValues()) {
                dest.emplace(std::move(signal), std::move(value));
            }
        }

        THashMap<TInstanceKey, THashMap<TSignalName, TValue>> Values;
    };

    struct TTagRecordVisitor final : public ITagRecordCallback {
        TTagRecordVisitor(const TSubscriptionFilter* subscriptions,
                          THostName hostName,
                          THostName groupName,
                          TTsdbRequestState& tsdbRequestState,
                          TInstant time)
            : Subscriptions(subscriptions)
            , HostName(hostName)
            , TsdbRequestPacker(tsdbRequestState.CreatePacker(time, hostName, groupName))
        {
        }

        void SetObjectsCount(const size_t count) override {
            TsdbRequestPacker.SetCount(count);
        }

        void OnTagRecord(TInstanceKey key, const TRecord& record) override {
            TsdbRequestPacker.AddRecord(key, record);
            PrepareServerMessage(key, record);
        }

        void PrepareServerMessage(TInstanceKey key, const TRecord& record) {
            if (Subscriptions) {
                TFilteringSignalValueVisitor visitor(*Subscriptions, key);
                record.Process(visitor);
                auto signals = visitor.Finish();
                if (!signals.empty()) {
                    TTaggedFoundSignals::insert_ctx ctx;
                    auto it = Found.find(key, ctx);
                    if (it != Found.end()) {
                        for (auto& signal : signals) {
                            it->second.emplace(std::move(signal));
                        }
                    } else {
                        Found.emplace_direct(ctx, key, std::move(signals));
                    }
                }
            }
        }

        void Finish(msgpack::sbuffer& serverMessage) {
            if (!Found.empty()) {
                msgpack::packer<msgpack::sbuffer> packer(serverMessage);
                PackServerMessage(packer, HostName.GetName(), Found);
            }
        }

        const TSubscriptionFilter* Subscriptions;
        const THostName HostName;

        TTaggedFoundSignals Found;

        TTsdbRequestPacker TsdbRequestPacker;
    };

    struct TTagContainerVisitor final: public ITagGroupContainerCallback {
        TTagContainerVisitor(TVector<msgpack::sbuffer>& messages) {
            for (auto& message : messages) {
                Packers.emplace_back(MakeHolder<msgpack::packer<msgpack::sbuffer>>(message));
            }
        }

        void OnTagContainer(TInstanceKey key, const TGroupContainer& container) override {
            auto& packer(*Packers[ComputeHash(key.ToNamed()) % Packers.size()]);
            packer.pack_array(2);
            PackString(packer, key.ToNamed());
            TSignalValueSerializer<NZoom::NHgram::TUgramCompressor> visitor(packer);
            container.Process(visitor);
        }

        TVector<THolder<msgpack::packer<msgpack::sbuffer>>> Packers;
    };

    struct TRecordFactory : public ISignalValueCallback {
        void SetObjectsCount(const size_t count) override {
            Values.reserve(count);
        }

        void OnSignalValue(const TSignalName& name, const TValueRef& value) override {
            Values.emplace_back(name, value);
        }

        NZoom::NRecord::TRecord Create() {
            return TRecord(std::move(Values));
        }

        TVector<std::pair<TSignalName, TValue>> Values;
    };

    struct TCommonRecordFactory : public ISignalValueCallback {
        TCommonRecordFactory(const THashSet<TSignalName>& seenSignals,
                             TSelectorPtr commonSignalSelector)
            : SeenSignals(seenSignals)
            , CommonSignalSelector(commonSignalSelector)
        {
        }

        void SetObjectsCount(const size_t count) override {
            Values.reserve(count);
        }

        void OnSignalValue(const TSignalName& signal, const TValueRef& value) override {
            if (SeenSignals.contains(signal)) {
                return;
            }

            const auto& name(signal.GetName());
            if (name.StartsWith(TStringBuf("cpu"))
                    || name.StartsWith(TStringBuf("cgroup"))
                    || name.StartsWith(TStringBuf("disk"))
                    || name.StartsWith(TStringBuf("fdstat"))
                    || name.StartsWith(TStringBuf("instances"))
                    || name.StartsWith(TStringBuf("iostat"))
                    || name.StartsWith(TStringBuf("loadavg"))
                    || name.StartsWith(TStringBuf("mem"))
                    || name.StartsWith(TStringBuf("netstat"))
                    || name.StartsWith(TStringBuf("sockstat"))
                    || name.StartsWith(TStringBuf("hbf4"))
                    || name.StartsWith(TStringBuf("hbf6"))
            ) {
                if (CommonSignalSelector->Match(name)) {
                    Values.emplace_back(signal, value);
                }
            }
        }

        NZoom::NRecord::TRecord Create() {
            return TRecord(std::move(Values));
        }

        const THashSet<TSignalName>& SeenSignals;
        TSelectorPtr CommonSignalSelector;
        TVector<std::pair<TSignalName, TValue>> Values;
    };

    struct TTaggedRecordFactory final : public ITagAgentContainerCallback {
        void SetObjectsCount(const size_t count) override {
            Values.reserve(count);
        }

        void OnTagContainer(TInstanceKey key, const TGroupContainer& container) override {
            TRecordFactory recordFactory;
            container.Process(recordFactory);
            Values.emplace_back(key, recordFactory.Create());
        }

        TTaggedRecord Create() {
            return TTaggedRecord(std::move(Values));
        }

        TVector<std::pair<TInstanceKey, TRecord>> Values;
    };

    struct TCommonMerger final : public ITagRecordCallback {
        TCommonMerger(const TCommonRules& commonRules, const TRecord* commonRecord)
            : CommonRules(commonRules)
            , CommonRecord(commonRecord)
        {
        }

        void SetObjectsCount(const size_t count) override {
            AffectedKeys.reserve(count);
        }

        void OnTagRecord(TInstanceKey key, const TRecord& record) override {
            if (key.GetItype() != COMMON_ITYPE && IsValidInstanceKey(key)) {
                auto selector(CommonRules.Apply(key));
                if (selector) {
                    auto& [commonSignalSelector, seenSignals] = AffectedKeys[key];

                    commonSignalSelector = selector;
                    for (const auto& [signal, _] : record.GetValues()) {
                        seenSignals.emplace(signal);
                    }
                }
            }
        }

        TTaggedRecord Create() {
            if (CommonRecord == nullptr || AffectedKeys.empty()) {
                return {};
            }

            TVector<std::pair<TInstanceKey, TRecord>> result;
            result.reserve(AffectedKeys.size());
            for (const auto& [key, affected] : AffectedKeys) {
                auto& [commonSignalSelector, seenSignals] = affected;

                TCommonRecordFactory factory(seenSignals, commonSignalSelector);
                CommonRecord->Process(factory);
                result.emplace_back(key, factory.Create());
            }

            return TTaggedRecord(std::move(result));
        }

        const TCommonRules& CommonRules;
        const TRecord* CommonRecord;
        THashMap<TInstanceKey, std::pair<TSelectorPtr, THashSet<TSignalName>>> AffectedKeys;
    };
}

TRequesterPipeline::TThreadedExecutor::TThreadedExecutor(const TRequesterPipeline& parentPipeline,
                                                         TSharedData& sharedData,
                                                         IMessagePusher& tsdbPusher)
    : ParentPipeline(parentPipeline)
    , SharedData(sharedData)
    , TsdbPusher(tsdbPusher)
    , Thread(ExecutorMain, this)
{
    ResetAfterIterationEnd();
    AtomicSet(Exit, 0);

    Thread.Start();
}

void TRequesterPipeline::TThreadedExecutor::ResetAfterIterationEnd() {
    AtomicSet(FinalizeOnEmptyQueue, 0);
    IterationFinalized = NThreading::NewPromise();
    JobsProcessed = NThreading::NewPromise();
    TsdbRequestState.Clear();
}

TRequesterPipeline::TThreadedExecutor::~TThreadedExecutor() {
    AtomicSet(Exit, 1);
}

void* TRequesterPipeline::TThreadedExecutor::ExecutorMain(void* untypedSelf) noexcept {
    auto* self = static_cast<TThreadedExecutor*>(untypedSelf);
    self->ExecutorMain();
    return nullptr;
}

void TRequesterPipeline::TThreadedExecutor::ExecutorMain() noexcept {
    while (!AtomicGet(Exit)) {
        // processing
        while (!AtomicGet(Exit)) {
            THostResponseJob job;
            auto finalizeOnEmptyQueue = AtomicGet(FinalizeOnEmptyQueue);
            while (SharedData.HostResponseQueue.Dequeue(&job)) {
                ProcessHostResponse(job.HostName, job.Response, job.IsProtoResponse);
            }
            if (!finalizeOnEmptyQueue) {
                Sleep(TDuration::MilliSeconds(25));
            } else {
                // finalization flag was set BEFORE we checked for elements in the queue, it means that it's time to finalize
                break;
            }
        }
        JobsProcessed.SetValue();

        // finalizing
        SendTSDBMessage();
        auto prevIterationPromise = IterationFinalized;
        // we expect that the owner is waiting on prevIterationPromise's future
        ResetAfterIterationEnd();
        prevIterationPromise.SetValue();
    }
}

std::pair<TRequesterPipeline::TThreadedExecutor::TFuture, TRequesterPipeline::TThreadedExecutor::TFuture>
TRequesterPipeline::TThreadedExecutor::FinalizeIterationJobs()
{
    auto result = std::make_pair(JobsProcessed.GetFuture(), IterationFinalized.GetFuture());
    AtomicSet(FinalizeOnEmptyQueue, 1);
    return result;
}

void TRequesterPipeline::TThreadedExecutor::ProcessHostResponse(const THostName& host, const TString& response,
                                                                bool isProtoResponse) {
    auto taggedRecord = ParseAgentResponse(host, response, isProtoResponse);
    if (taggedRecord) {
        // empty response can mean a bad response or a normal response of an agent that is starting
        try {
            AtomicAdd(SharedData.HostsSuccess, 1);
            Mul(host, *taggedRecord);
        } catch (...) {
            TGuard lock(SharedData.SharedMessagesLock);
            SharedData.AggregateErrors.emplace(host.GetName(), CurrentExceptionMessage());
        }
    }
}

THolder<TTaggedRecord> TRequesterPipeline::TThreadedExecutor::ParseAgentResponse(const THostName& host, const TString& response,
                                                                                 bool isProtoResponse)
{
    THolder<TTaggedRecord> result;
    bool invalidResponseStatus = false;
    try {
        if (isProtoResponse) {
            TAgentProtobufResponseParser responseParser(response);
            auto parsedResult = responseParser.TaggedRecord();
            AtomicAdd(SharedData.IgnoredSignals, parsedResult.second);

            using NYasm::NInterfaces::NInternal::EAgentStatus;
            switch (responseParser.GetStatusCode()) {
                case EAgentStatus::STATUS_OK: {
                    result = std::move(parsedResult.first);
                    break;
                }
                case EAgentStatus::STATUS_NO_DATA: {
                    // weird logic ported from python
                    result = std::move(parsedResult.first);
                    invalidResponseStatus = true;
                    break;
                }
                default: {
                    // everything else is an empty response
                    break;
                }
            }
        } else {
            TAgentResponseParser responseParser(response);
            auto parsedResult = responseParser.GetRecordIterator().TaggedRecord();
            auto status = responseParser.GetStatus();
            if (status.Defined() && status.GetRef() != "starting") {
                invalidResponseStatus = status.GetRef() != "ok";
                result = std::move(parsedResult);
            }
            AtomicAdd(SharedData.NonProtoResponses, 1);
        }
    } catch (...) {
        TGuard lock(SharedData.SharedMessagesLock);
        SharedData.DeserializeErrors.emplace(host.GetName(), CurrentExceptionMessage());
    }
    if (invalidResponseStatus) {
        TGuard lock(SharedData.SharedMessagesLock);
        SharedData.InvalidResponseHosts.insert(host.GetName());
    }
    return result;
}

void TRequesterPipeline::TThreadedExecutor::Mul(const THostName& hostName, TTaggedRecord& taggedRecord) {
    auto hostifiedRecords = AddGroupAndHostAndSort(ParentPipeline.GroupName, hostName, taggedRecord);
    THolder<TRecord> commonRecord;

    for (auto& [key, record] : hostifiedRecords.GetValues()) {
        if (key.GetItype() == COMMON_ITYPE && !commonRecord) {
            // simply take first one record
            TVector<std::pair<TSignalName, TValue>> commonRecordValues;
            for (auto& [signal, value] : record.GetValues()) {
                commonRecordValues.emplace_back(signal, value.GetValue());
            }
            commonRecord = MakeHolder<TRecord>(std::move(commonRecordValues));
            break;
        }
    }

    TVector<std::pair<TInstanceKey, TRecord>> currentValues;
    THostName currentHostName;

    for (auto& [key, record] : hostifiedRecords.GetValues()) {
        auto keyHostName = key.GetHostName();
        if (keyHostName != currentHostName) {
            if (!currentHostName.Empty()) {
                TVector<std::pair<TInstanceKey, TRecord>> tmp;
                tmp.swap(currentValues);
                TTaggedRecord hostifiedRecord(std::move(tmp));
                ProcessHostifiedRecord(currentHostName, hostifiedRecord, commonRecord.Get());

            }
            currentHostName = keyHostName;
            currentValues.clear();
        }
        currentValues.emplace_back(key, std::move(record));
    }

    if (!currentValues.empty()) {
        TTaggedRecord hostifiedRecord = std::move(currentValues);
        ProcessHostifiedRecord(currentHostName, hostifiedRecord, commonRecord.Get());
    }
}

void TRequesterPipeline::TThreadedExecutor::ProcessHostifiedRecord(const THostName& hostName, TTaggedRecord& hostifiedRecord,
                                                                   const TRecord* commonRecord) {
    auto commons = PrepareCommons(hostifiedRecord, commonRecord);
    auto preaggregates = PreparePreaggregates(hostifiedRecord, commons);

    TTaggedRecordMerger merger;
    merger.AddOnlyRaw(hostifiedRecord);
    merger.Add(commons);
    merger.Add(preaggregates);
    auto mergedRecords = merger.Create();
    auto it = ParentPipeline.Subscriptions.find(hostName);
    auto subscriptionFilter = (it != ParentPipeline.Subscriptions.end()) ? &it->second : nullptr;
    TTagRecordVisitor visitor(subscriptionFilter, hostName, ParentPipeline.GroupName, GetTsdbRequestState(),
        ParentPipeline.Timestamp);
    mergedRecords.Process(visitor);
    auto aggregatedRecord = AggregateRecordsByHost(mergedRecords);

    {
        // write host's result to shared data
        TGuard guard(SharedData.SharedDataLock);
        visitor.Finish(SharedData.ServerMessage);
        SharedData.GroupAggregator.Mul(aggregatedRecord, *SharedData.MetricManager);
    }
}

TTaggedRecord TRequesterPipeline::TThreadedExecutor::PrepareCommons(const TTaggedRecord& taggedRecord,
                                                                    const TRecord* commonRecord) {
    TCommonMerger visitor(ParentPipeline.CommonRules, commonRecord);
    taggedRecord.Process(visitor);
    return visitor.Create();
}

TTaggedRecord TRequesterPipeline::TThreadedExecutor::PreparePreaggregates(const TTaggedRecord& taggedRecord,
                                                                          const TTaggedRecord& commonRecord) {
    TConfigurableAggregator preaggregator(ParentPipeline.Conf, ParentPipeline.AggregationRules);
    preaggregator.Mul(taggedRecord);
    preaggregator.Mul(commonRecord);

    TTaggedRecordFactory factory;
    preaggregator.Process(factory);

    return factory.Create();
}

TTsdbRequestState& TRequesterPipeline::TThreadedExecutor::GetTsdbRequestState() {
    if (!TsdbRequestState.Defined()) {
        TsdbRequestState.ConstructInPlace();
    }
    return TsdbRequestState.GetRef();
}

void TRequesterPipeline::TThreadedExecutor::SendTSDBMessage() {
    try {
        if (TsdbRequestState.Defined()) {
            auto message = TsdbRequestState->Serialize();
            TsdbPusher.AddMessage(ParentPipeline.Timestamp, message);
            TsdbPusher.Finish();
        }
    } catch (...) {
        TGuard lock(SharedData.SharedMessagesLock);
        SharedData.SerializeErrors.emplace(CurrentExceptionMessage());
    }
}

TRequesterPipeline::TRequesterPipeline(const TYasmConf& conf,
                                       const TVector<IMessagePusher*>& perThreadTsdbPushers,
                                       const TAggregationRules& aggregationRules,
                                       const TCommonRules& commonRules,
                                       TStringBuf group,
                                       size_t middleCount)
    : Conf(conf)
    , AggregationRules(aggregationRules)
    , CommonRules(commonRules)
    , GroupName(group)
    , MiddleMessages(middleCount)
    , SharedData(conf)
{
    for (auto messagePusher: perThreadTsdbPushers) {
        Executors.push_back(MakeHolder<TThreadedExecutor>(*this, SharedData, *messagePusher));
    }
}

TRequesterPipeline::TSharedData::TSharedData(const NZoom::NYasmConf::TYasmConf& conf)
    : HostResponseQueue()
    , GroupAggregator(conf)
    , MetricManager(nullptr)
    , IgnoredSignals(0)
    , NonProtoResponses(0)
    , HostsSuccess(0)
{
}

void TRequesterPipeline::TSharedData::ExtractStatsAndMessages(TRequesterPipelineStatsAndMessages& statsAndMessages) {
    statsAndMessages.DeserializeErrors.swap(DeserializeErrors);
    statsAndMessages.AggregateErrors.swap(AggregateErrors);

    statsAndMessages.InvalidResponseHosts.reserve(InvalidResponseHosts.size());
    std::copy(InvalidResponseHosts.begin(), InvalidResponseHosts.end(),
        std::back_inserter(statsAndMessages.InvalidResponseHosts));

    statsAndMessages.SerializeErrors.reserve(SerializeErrors.size());
    std::copy(SerializeErrors.begin(), SerializeErrors.end(),
        std::back_inserter(statsAndMessages.SerializeErrors));

    statsAndMessages.IgnoredSignals = AtomicGet(IgnoredSignals);
    statsAndMessages.NonProtoResponses = AtomicGet(NonProtoResponses);
    statsAndMessages.HostsSuccess = AtomicGet(HostsSuccess);
}

void TRequesterPipeline::TSharedData::Clean(TInstant now) {
    GroupAggregator.Clean(now);
    ServerMessage.clear();
    MetricManager = nullptr;

    DeserializeErrors.clear();
    AggregateErrors.clear();
    InvalidResponseHosts.clear();
    SerializeErrors.clear();
    AtomicSet(IgnoredSignals, 0);
    AtomicSet(NonProtoResponses, 0);
    AtomicSet(HostsSuccess, 0);
}

void TRequesterPipeline::SetSubscriptions(const THostTagSignals& subscriptions) {
    Subscriptions.clear();
    for (const auto& hostTagSignals : subscriptions) {
        auto& filter(Subscriptions[hostTagSignals.first]);
        for (const auto& tagSignals : hostTagSignals.second) {
            filter.Add(tagSignals.first.GetRequestKey(), tagSignals.second);
        }
    }
}

void TRequesterPipeline::SetSubscriptions(PyObject* value) {
    TVector<TSubscription> subscriptions = DeserializeSubscriptions(value);
    SetSubscriptions(SubscriptionsToHostTagSignals(subscriptions));
}

void TRequesterPipeline::SetMetricManager(NZoom::NAggregators::TTaggedMetricManager& metricManager) {
    SharedData.MetricManager = &metricManager;
}

void TRequesterPipeline::SetTime(TInstant timestamp) {
    Timestamp = timestamp;
}

void TRequesterPipeline::AddHostResponse(TStringBuf host, const TString& response, const TString& responseContentType) {
    if (!SharedData.MetricManager) {
        ythrow yexception() << "Metric manager is not set";
    }
    THostName hostName(host);
    bool isProtoResponse = responseContentType == PROTOBUF_CONTENT_TYPE;
    SharedData.HostResponseQueue.Enqueue(THostResponseJob{
        .HostName = hostName,
        .Response = response,
        .IsProtoResponse = isProtoResponse
    });
}

void TRequesterPipeline::PrepareMiddleMessages() {
    TTagContainerVisitor visitor(MiddleMessages);
    SharedData.GroupAggregator.Process(visitor);
}

void TRequesterPipeline::Finish(TRequesterPipelineStatsAndMessages* statsAndMessages) {
    if (Finished) {
        ythrow yexception() << "requester pipeline already finished";
    }
    TVector<TThreadedExecutor::TFuture> jobsProcessedFutures(Reserve(Executors.size()));
    TVector<TThreadedExecutor::TFuture> finalizedFutures(Reserve(Executors.size()));
    for (const auto& executor: Executors) {
        auto [jobsProcessed, iterationFinalized] = executor->FinalizeIterationJobs();
        jobsProcessedFutures.push_back(jobsProcessed);
        finalizedFutures.push_back(iterationFinalized);
    }
    for (auto& future: jobsProcessedFutures) {
        future.Wait();
    }

    PrepareMiddleMessages();

    for (auto& future: finalizedFutures) {
        future.Wait();
    }
    Finished = true;

    if (statsAndMessages) {
        SharedData.ExtractStatsAndMessages(*statsAndMessages);
    }
}

TStringBuf TRequesterPipeline::GetServerMessage() const {
    if (!Finished) {
        ythrow yexception() << "requester pipeline not finished yet";
    }
    return {SharedData.ServerMessage.data(), SharedData.ServerMessage.size()};
}

TVector<TStringBuf> TRequesterPipeline::GetMiddleMessages() const {
    if (!Finished) {
        ythrow yexception() << "requester pipeline not finished yet";
    }
    TVector<TStringBuf> messages;
    for (const auto& message : MiddleMessages) {
        messages.emplace_back(message.data(), message.size());
    }
    return messages;
}

void TRequesterPipeline::Clean() {
    Clean(TInstant::Now());
}

void TRequesterPipeline::Clean(const TInstant explicitNow) {
    SharedData.Clean(explicitNow);
    for (auto& message : MiddleMessages) {
        message.clear();
    }
    Finished = false;
}
