#pragma once

#include "subscriptions.h"
#include "message_pusher.h"
#include "tsdb.h"

#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/zoom/components/aggregators/subscription_filter.h>
#include <infra/yasm/zoom/components/aggregators/preaggregates.h>
#include <infra/yasm/zoom/components/aggregators/group.h>
#include <infra/yasm/zoom/components/aggregators/metrics.h>

#include <util/generic/queue.h>
#include <util/system/thread.h>
#include <util/thread/lfqueue.h>
#include <library/cpp/threading/future/future.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

namespace NZoom {
    namespace NPython {
        struct TRequesterPipelineStatsAndMessages {
            THashMap<TString, TString> DeserializeErrors;
            THashMap<TString, TString> AggregateErrors;
            TVector<TString> InvalidResponseHosts;
            TVector<TString> SerializeErrors;
            long IgnoredSignals = 0;
            long NonProtoResponses = 0;
            long HostsSuccess = 0;
        };

        class TRequesterPipeline {
        public:
            TRequesterPipeline(const NZoom::NYasmConf::TYasmConf& conf,
                               const TVector<IMessagePusher*>& perThreadTsdbPushers,
                               const NZoom::NAggregators::TAggregationRules& aggregationRules,
                               const NZoom::NAggregators::TCommonRules& commonRules,
                               TStringBuf groupName,
                               size_t middleCount);

            void SetSubscriptions(const NZoom::NSubscription::THostTagSignals& subscriptions);
            void SetSubscriptions(PyObject* value);
            void SetTime(TInstant timestamp);
            void SetMetricManager(NZoom::NAggregators::TTaggedMetricManager& metricManager);

            /** Start adding host responses only when all iteration properties were set by the setters above */
            void AddHostResponse(TStringBuf host, const TString& response, const TString& responseContentType);

            /** Finish should be called only after all host responses were added. */
            void Finish(TRequesterPipelineStatsAndMessages* statsAndMessages);

            /** Extracting of iteration results should be performed only after a call to Finish() */
            TStringBuf GetServerMessage() const;
            TVector<TStringBuf> GetMiddleMessages() const;

            /** Methods to clean pipeline before the next iteration */
            void Clean();
            void Clean(TInstant explicitNow); // Debugging hook

        private:
            struct THostResponseJob {
                NZoom::NHost::THostName HostName;
                TString Response;
                bool IsProtoResponse;
            };

            struct TSharedData {
                TLockFreeQueue<THostResponseJob> HostResponseQueue;

                /** Mutex protecting non-thread safe data fields */
                TMutex SharedDataLock;

                /** Data filled by concurrent workers */
                NZoom::NAggregators::TGroupAggregator GroupAggregator;
                msgpack::sbuffer ServerMessage;
                NZoom::NAggregators::TTaggedMetricManager* MetricManager = nullptr; // non-optional

                /** Mutex protecting non-thread safe message and stats fields */
                TMutex SharedMessagesLock;

                /** Stats and errors */
                THashMap<TString, TString> DeserializeErrors;
                THashMap<TString, TString> AggregateErrors;
                THashSet<TString> InvalidResponseHosts;
                THashSet<TString> SerializeErrors;
                TAtomic IgnoredSignals = 0;
                TAtomic NonProtoResponses = 0;
                TAtomic HostsSuccess = 0;

                explicit TSharedData(const NZoom::NYasmConf::TYasmConf& conf);
                void ExtractStatsAndMessages(TRequesterPipelineStatsAndMessages& statsAndMessages);
                void Clean(TInstant now);
            };

            class TThreadedExecutor {
            public:
                using TFuture = NThreading::TFuture<void>;

                TThreadedExecutor(const TRequesterPipeline& parentPipeline, TSharedData& sharedData, IMessagePusher& tsdbPusher);
                ~TThreadedExecutor();

                /** Call it to finalize iteration when all jobs were added to the queue.
                 *  Returns two futures:
                 *    - First for the moment when all jobs are processed
                 *    - Second for the moment when all jobs are finalized */
                std::pair<TFuture, TFuture> FinalizeIterationJobs();

                static void* ExecutorMain(void* self) noexcept;

            private:
                using TPromise = NThreading::TPromise<void>;

                void ExecutorMain() noexcept;
                void ResetAfterIterationEnd();
                /** Methods called from ExecutorMain */
                void ProcessHostResponse(const NZoom::NHost::THostName& host, const TString& response, bool isProtoResponse);
                THolder<NZoom::NRecord::TTaggedRecord> ParseAgentResponse(const NZoom::NHost::THostName& host,
                                                                          const TString& response,
                                                                          bool isProtoResponse);
                void SendTSDBMessage();
                void Mul(const NZoom::NHost::THostName& hostName, NZoom::NRecord::TTaggedRecord& taggedRecord);
                void ProcessHostifiedRecord(const NZoom::NHost::THostName& hostName,
                                            NZoom::NRecord::TTaggedRecord& hostifiedRecord,
                                            const NZoom::NRecord::TRecord* commonRecord);
                NZoom::NRecord::TTaggedRecord PrepareCommons(const NZoom::NRecord::TTaggedRecord& taggedRecord,
                                                             const NZoom::NRecord::TRecord* commonRecord);
                NZoom::NRecord::TTaggedRecord PreparePreaggregates(const NZoom::NRecord::TTaggedRecord& taggedRecord,
                                                                   const NZoom::NRecord::TTaggedRecord& commonRecord);
                TTsdbRequestState& GetTsdbRequestState();

                /** Fields accessed from ExecutorMain */
                const TRequesterPipeline& ParentPipeline;
                TSharedData& SharedData;
                IMessagePusher& TsdbPusher;
                TMaybe<TTsdbRequestState> TsdbRequestState;

                /** Stuff for synchronization with a single-threaded owner */
                TPromise JobsProcessed;
                TPromise IterationFinalized;
                TAtomic FinalizeOnEmptyQueue;
                TThread Thread;
                TAtomic Exit;
            };

            static constexpr TStringBuf PROTOBUF_CONTENT_TYPE = "application/x-protobuf";

            void PrepareMiddleMessages();

            const NZoom::NYasmConf::TYasmConf& Conf;
            const NZoom::NAggregators::TAggregationRules& AggregationRules;
            const NZoom::NAggregators::TCommonRules& CommonRules;
            const NZoom::NHost::THostName GroupName;

            /** fields that are readonly when processing host responses*/
            THashMap<NZoom::NHost::THostName, NZoom::NAggregators::TSubscriptionFilter> Subscriptions;
            TInstant Timestamp;
            bool Finished = false;
            TVector<msgpack::sbuffer> MiddleMessages;

            /** Data that is accessed from multiple worker threads */
            TSharedData SharedData;

            /** threaded executor */
            TVector<THolder<TThreadedExecutor>> Executors;
        };
    }
}
