#pragma once

#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/common/labels/host/host.h>
#include <infra/yasm/zoom/components/aggregators/subscription_filter.h>
#include <infra/yasm/zoom/components/serialization/common/msgpack_utils.h>
#include <infra/monitoring/common/msgpack.h>

#include <util/generic/xrange.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

namespace NZoom {
    namespace NPython {
        using TFoundSignals = THashMap<NZoom::NSignal::TSignalName, NZoom::NValue::TValueRef>;
        using TTaggedFoundSignals = THashMap<NTags::TInstanceKey, TFoundSignals>;

        template <class TUgramCompressorSingleton>
        struct TSignalValueSerializer final : public NZoom::NRecord::ISignalValueCallback {
            TSignalValueSerializer(msgpack::packer<msgpack::sbuffer>& packer)
                : Packer(packer)
                , ValueVisitor(Packer)
            {
            }

            void SetObjectsCount(const size_t count) override {
                Packer.pack_map(count);
            }

            void OnSignalValue(const NZoom::NSignal::TSignalName& name, const NZoom::NValue::TValueRef& value) override {
                NMonitoring::PackString(Packer, name.GetName());
                value.Update(ValueVisitor);
            }

            void VisitFoundSignals(const TFoundSignals& foundSignals) {
                SetObjectsCount(foundSignals.size());
                for (const auto& signalValue : foundSignals) {
                    OnSignalValue(signalValue.first, signalValue.second);
                }
            }

            msgpack::packer<msgpack::sbuffer>& Packer;
            TValueRefSerializer<msgpack::sbuffer, TUgramCompressorSingleton> ValueVisitor;
        };

        struct TFilteringSignalValueVisitor final : public NZoom::NRecord::ISignalValueCallback {

            TFilteringSignalValueVisitor(const NZoom::NAggregators::TSubscriptionFilter& filter, NTags::TInstanceKey instanceKey)
                : Filter(filter)
                , InstanceKey(instanceKey)
            {
            }

            void SetObjectsCount(const size_t count) override {
                Signals.reserve(count);
                Values.reserve(count);
            }

            void OnSignalValue(const NZoom::NSignal::TSignalName& name, const NZoom::NValue::TValueRef& value) override {
                Signals.emplace_back(name);
                Values.emplace_back(value);
            }

            TFoundSignals Finish() {
                TFoundSignals result;
                const auto matches(Filter.Match(InstanceKey, Signals));
                const auto elements(matches.Count());
                if (elements) {
                    for (const auto& idx : xrange(Signals.size())) {
                        if (matches.Get(idx)) {
                            result.emplace(Signals[idx], Values[idx]);
                        }
                    }

                }
                return result;
            }

            const NZoom::NAggregators::TSubscriptionFilter& Filter;
            const NTags::TInstanceKey InstanceKey;

            TVector<NZoom::NSignal::TSignalName> Signals;
            TVector<NZoom::NValue::TValueRef> Values;
        };

        void PackServerMessage(msgpack::packer<msgpack::sbuffer>& packer, const NZoom::NHost::THostName& hostName, const TTaggedFoundSignals& found);
    }
}
