#include "server.h"

#include <infra/yasm/zoom/components/serialization/common/msgpack_utils.h>
#include <infra/yasm/common/threaded_executor.h>
#include <infra/monitoring/common/msgpack.h>

#include <util/generic/xrange.h>
#include <util/digest/numeric.h>

using namespace NTags;
using namespace NZoom::NAccumulators;
using namespace NZoom::NContainers;
using namespace NZoom::NPython;
using namespace NZoom::NRecord;
using namespace NZoom::NValue;
using namespace NZoom::NHost;
using namespace NZoom::NSignal;
using namespace NZoom::NSubscription;
using namespace NYasm::NCommon;

namespace {
    class TWorkerServerExecutor: public TBaseThreadedExecutor {
        Y_DECLARE_SINGLETON_FRIEND()
    public:
        static constexpr size_t WORKER_THREAD_COUNT = 6;
        static TWorkerServerExecutor& Get() {
            return *SingletonWithPriority<TWorkerServerExecutor, 100001>();
        }

    protected:
        TWorkerServerExecutor()
            : TBaseThreadedExecutor("TWorkerServerExecutor", WORKER_THREAD_COUNT) {
        }
    };

    class TAsyncRequestResultAccumulator: public TNonCopyable {
    public:
        TAsyncRequestResultAccumulator() = default;
        ~TAsyncRequestResultAccumulator() {
            // wait for all in-progress jobs to complete but do not re-throw their exceptions to avoid termination
            Wait(false);
        }

        /**
         * Add tag group to the task.
         * @param tagGroup should be unique. Do not add the same tag group more than once
         */
        void AddGroupToAccumulate(TTagGroup& tagGroup) {
            if (tagGroup.Tags.empty())
                return;

            for (auto& requestKeyGroup: tagGroup.RequestKeyGroups) {
                TagGroupsTasks.emplace_back(&requestKeyGroup, &tagGroup);
            }
        }
        /**
         * Asynchronously start accumulating added groups.
         */
        void Start() {
            for (auto it = TagGroupsTasks.begin(); it != TagGroupsTasks.end();) {
                auto from = it;
                size_t lastTaskSize = 0;
                while (it != TagGroupsTasks.end() && lastTaskSize < TASK_SIZE_THRESHOLD) {
                    lastTaskSize += it->second->Tags.size();
                    ++it;
                }
                Futures.emplace_back(TWorkerServerExecutor::Get().Add([from, to = it]() {
                    for (auto it = from; it != to; ++it) {
                        if (PrepareGroup(*it->first, *it->second)) {
                            AccumulateGroup(*it->first, *it->second);
                        }
                    }
                }));
            }
        }
        /**
         * Wait for groups to accumulate
         * @param rethrowException pass true to rethrow exception from a failed group processing
         */
        void Wait(bool rethrowException) {
            TMaybe<TWorkerServerExecutor::TFuture> futureWithException;
            // wait for all futures even if some fail, so do not use WaitAll here
            for (auto& future: Futures) {
                future.Wait();
                if (!futureWithException.Defined() && future.HasException()) {
                    futureWithException = future;
                }
            }
            Futures.clear();
            if (rethrowException && futureWithException.Defined()) {
                futureWithException->GetValue(); // will rethrow an exception
            }
        }

    private:
        static bool PrepareGroup(TRequestKeyGroup& requestKeyGroup, const TTagGroup& tagGroup) {
            TDynamicFilter filter(requestKeyGroup.RequestKey.GetRequestKey());
            filter.FeedMany(tagGroup.Tags);

            const auto variants = filter.Resolve();

            if (!variants.empty()) {
                requestKeyGroup.ResolvedTags.insert(variants.begin(), variants.end());
            }
            return !requestKeyGroup.ResolvedTags.empty();
        }

        static void AccumulateGroup(TRequestKeyGroup& requestKeyGroup, const TTagGroup& tagGroup) {
            for (const auto instanceIndex : xrange(tagGroup.Tags.size())) {
                const auto tag = tagGroup.Tags[instanceIndex];
                const auto value = tagGroup.Values[instanceIndex].GetValue();

                if (requestKeyGroup.ResolvedTags.contains(tag)) {
                    requestKeyGroup.Accumulator.Mul(value);
                }
            }
        }

        TVector<TWorkerServerExecutor::TFuture> Futures;
        TVector<std::pair<TRequestKeyGroup*, TTagGroup*>> TagGroupsTasks;
        const size_t TASK_SIZE_THRESHOLD = 1000;
    };
}

TRequestKeyGroup::TRequestKeyGroup(const TInternedRequestKey& requestKey, EAccumulatorType accumulatorType)
    : RequestKey(requestKey)
    , ResolvedTags()
    , Accumulator(accumulatorType) {
}

void TTagGroup::Clear() {
    Tags.clear();
    Values.clear();
    RequestKeyGroups.clear();
}

TServerPipeline::TServerPipeline(const TString& aggrName)
    : AggrName(aggrName)
{
}

void TServerPipeline::Mul(THostName hostName, const TTaggedRecord& taggedRecord) {
    auto& signalTagValues(Points[hostName]);
    for (const auto& tagRecord : taggedRecord.GetValues()) {
        for (const auto& signalValue : tagRecord.second.GetValues()) {
            auto& tagGroup(signalTagValues[signalValue.first]);
            tagGroup.Tags.emplace_back(tagRecord.first);
            tagGroup.Values.emplace_back(signalValue.second.GetValue());
        }
    }
}

void TServerPipeline::Mul(const TString& hostName, const TTaggedRecord& taggedRecord) {
    Mul(THostName(hostName), taggedRecord);
}

void TServerPipeline::SetSubscriptions(const TVector<TSubscription>& subscriptions) {
    for (auto& subscription : subscriptions) {
        TSignalName signalName = subscription.GetSignalExpression();
        auto aggregationRules = signalName.GetAggregationRules();
        const auto& requestKey = subscription.GetRequestKey();
        if (aggregationRules) {
            auto& tagGroup = Points[subscription.GetHostName()][signalName];
            tagGroup.RequestKeyGroups.emplace_back(requestKey,
                aggregationRules->GetAccumulatorType(EAggregationMethod::Group));
        } else {
            // write n/a immediately
            RequestedPoints[subscription.GetHostName()][requestKey.GetRequestKey().ToNamed()].emplace_back(signalName, TValue());
        }
    }
}

void TServerPipeline::SetSubscriptions(PyObject* value) {
    SetSubscriptions(DeserializeSubscriptions(value));
}

TServerStats TServerPipeline::Finish() {
    TAsyncRequestResultAccumulator accumulator;
    for (auto& hostSignals : Points) {
        for (auto& signalTags : hostSignals.second) {
            accumulator.AddGroupToAccumulate(signalTags.second);
        }
    }
    accumulator.Start();
    accumulator.Wait(true);
    TServerStats stats;
    for (auto& hostSignals : Points) {
        for (auto& signalTags : hostSignals.second) {
            WriteGroup(hostSignals.first, signalTags.first, signalTags.second);

            stats.KeySetPowerSum += signalTags.second.Tags.size();
            stats.MaxKeySetPower = Max(stats.MaxKeySetPower, signalTags.second.Tags.size());
            for (const auto& requestKeyGroup : signalTags.second.RequestKeyGroups) {
                stats.ResolvedKeySetPowerSum += requestKeyGroup.ResolvedTags.size();
                stats.MaxResolvedKeySetPower = Max(stats.MaxResolvedKeySetPower, requestKeyGroup.ResolvedTags.size());
            }
            signalTags.second.Clear();
        }
    }
    Finished = true;
    return stats;
}

void TServerPipeline::WriteGroup(THostName hostName, TSignalName signal, TTagGroup& tagGroup) {
    auto& hostKeys = RequestedPoints[hostName];
    for (const auto& requestKeyGroup : tagGroup.RequestKeyGroups) {
        const auto& requestKey = requestKeyGroup.RequestKey.GetRequestKey();
        hostKeys[requestKey.ToNamed()].emplace_back(signal, requestKeyGroup.Accumulator.GetValue());
    }
}

void TServerPipeline::Clean() {
    RequestedPoints.clear();
    Points.clear();
    Finished = false;
}

TString TServerPipeline::GetRequestedPointsMessage(TInstant iterationTs) const {
    using namespace NMonitoring;
    if (!Finished) {
        ythrow yexception() << "server pipeline not finished yet";
    }
    msgpack::sbuffer requestedPointsMessage;
    msgpack::packer<msgpack::sbuffer> packer(requestedPointsMessage);
    TValueRefSerializer<msgpack::sbuffer, NZoom::NHgram::TUgramCompressor> serializer(packer);

    packer.pack_map(2);
    PackString(packer, ITERATION_TIMESTAMP_FIELD_NAME);
    packer.pack_int64(iterationTs.Seconds());

    PackString(packer, SUBSCRIPTION_LIST_FIELD_NAME);
    ui32 requestedPointsCount = 0;
    for (const auto& hostTags : RequestedPoints) {
        for (const auto& tagSignals : hostTags.second) {
            requestedPointsCount += tagSignals.second.size();
        }
    }
    packer.pack_array(requestedPointsCount);
    for (const auto& hostTags : RequestedPoints) {
        for (const auto& tagSignals : hostTags.second) {
            for (const auto& signalValue : tagSignals.second) {
                packer.pack_map(4);
                PackString(packer, HOST_FIELD_NAME);
                PackString(packer, hostTags.first.GetName());
                PackString(packer, TAGS_FIELD_NAME);
                PackString(packer, tagSignals.first);
                PackString(packer, SIGNAL_FIELD_NAME);
                PackString(packer, signalValue.first.GetName());
                PackString(packer, VALUE_FIELD_NAME);
                signalValue.second.Update(serializer);
            }
        }
    }
    return {requestedPointsMessage.data(), requestedPointsMessage.size()};
}
