#include "server_worker.h"
#include "server.h"

#include <infra/yasm/common/threaded_executor.h>
#include <infra/yasm/zoom/components/aggregators/subscription_filter.h>
#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/zoom/components/serialization/common/msgpack_utils.h>
#include <infra/yasm/zoom/components/yasmconf/yasmconf.h>
#include <infra/yasm/zoom/python/responses.h>

#include <library/cpp/http/client/client.h>
#include <library/cpp/http/fetch/exthttpcodes.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/threading/cron/cron.h>
#include <library/cpp/threading/hot_swap/hot_swap.h>

#include <util/datetime/base.h>
#include <util/generic/queue.h>
#include <util/string/printf.h>

using namespace NZoom::NPython;

namespace {
    class TLocalSubscriptions: public TAtomicRefCount<TLocalSubscriptions> {
    public:
        explicit TLocalSubscriptions(TStringBuf storeResponse)
            : Subscriptions_{DeserializeSubscriptions(storeResponse)}
        {
        }

        const auto& GetSubscriptions() const noexcept {
            return Subscriptions_;
        }

    private:
        TVector<NZoom::NSubscription::TSubscription> Subscriptions_;
    };

    class TSubscriptionsClient {
        static constexpr TDuration DefaultTimeout = TDuration::Seconds(15);
    public:
        explicit TSubscriptionsClient(ui16 port) noexcept
            : Url_{Sprintf("http://localhost:%d", port)}
        {
        }

        NThreading::TFuture<TString> ListAll() {
            auto startTime = TInstant::Now();

            NHttp::TFetchOptions options;
            options.Timeout = DefaultTimeout;
            NHttp::TFetchQuery query(Url_ + "/list_all", options);

            auto promise = NThreading::NewPromise<TString>();
            NHttp::FetchAsync(query, [promise, startTime](NHttpFetcher::TResultRef result) mutable {
                LogResponseStatus(startTime, "TSubscriptionsClient::ListAll", result->Code);
                if (result->Code == HttpCodes::HTTP_OK) {
                    promise.SetValue(result->Data);
                } else {
                    promise.SetException(TStringBuilder{} << "server response: " << result->Code << ' ' << result->Data);
                }
            });
            return promise.GetFuture();
        }

        NThreading::TFuture<void> PushValues(TString data) {
            auto startTime = TInstant::Now();

            NHttp::TFetchOptions options;
            options.Timeout = DefaultTimeout;
            options.PostData = std::move(data);
            NHttp::TFetchQuery query(Url_ + "/push_values", options);

            auto promise = NThreading::NewPromise();
            NHttp::FetchAsync(query, [promise, startTime](NHttpFetcher::TResultRef result) mutable {
                LogResponseStatus(startTime, "TSubscriptionsClient::PushValues", result->Code);
                if (result->Code == HttpCodes::HTTP_OK) {
                    promise.SetValue();
                } else {
                    promise.SetException(TStringBuilder{} << "server response: " << result->Code << ' ' << result->Data);
                }
            });
            return promise.GetFuture();
        }

        static void LogResponseStatus(TInstant startTime, TStringBuf endpoint, int resultCode) {
            TDuration elapsedTime = TInstant::Now() - startTime;
            Cerr << TInstant::Now() << ' ' << endpoint  << ": "
                 << resultCode << ' ' << ExtHttpCodeStr(resultCode)
                 << ", time=" << elapsedTime << Endl;
        }


    private:
        const TString Url_;
    };

    class TPipeline: private TStreamingResponseLoader, public TAtomicRefCount<TPipeline> {
    public:
        explicit TPipeline(const TString& aggrName)
            : Impl_{aggrName}
        {
        }

        virtual ~TPipeline() = default;

        /**
         * Parse and process data from worker.
         *
         * @param data  metrics from worker
         * @return number of workers uploaded data to this pipeline
         */
        size_t LoadData(TStringBuf data) {
            with_lock(Mutex_) {
                Unpack(data);
                return ++DoneWorkers_;
            }
        }

        TString Finalize(TInstant iterationTime, TIntrusivePtr<TLocalSubscriptions> subscriptions) {
            with_lock(Mutex_) {
                Impl_.SetSubscriptions(subscriptions->GetSubscriptions());
                Impl_.Finish();
                return Impl_.GetRequestedPointsMessage(iterationTime);
            }
        }

    private:
        void OnMessage(const msgpack::object& obj) override {
            Y_ENSURE(obj.type == msgpack::type::ARRAY, "root is not array");

            const msgpack::object_array root = obj.via.array;
            Y_ENSURE(root.size == 2, "root has wrong size=" << root.size);

            Y_ENSURE(root.ptr[0].type == msgpack::type::STR);
            TStringBuf hostName = AsStrBuf(root.ptr[0]);

            TTagRecordVector tags;
            Y_ENSURE(root.ptr[1].type == msgpack::type::MAP, "tag signals is not map");
            const msgpack::object_map source = root.ptr[1].via.map;
            tags.reserve(source.size);

            for (size_t idx = 0; idx < source.size; ++idx) {
                const msgpack::object_kv& kv = source.ptr[idx];
                tags.emplace_back(
                        NTags::TInstanceKey::FromNamed(AsStrBuf(kv.key)),
                        UnpackRecord(kv.val));
            }

            Impl_.Mul(NZoom::NHost::THostName{hostName}, NZoom::NRecord::TTaggedRecord{std::move(tags)});
        }

    private:
        TMutex Mutex_;
        TServerPipeline Impl_;
        size_t DoneWorkers_{0};
    };

    class TPipelinesMap {
    public:
        explicit TPipelinesMap(TString aggrName) noexcept
            : AggrName_{std::move(aggrName)}
        {
        }

        TIntrusivePtr<TPipeline> GetOrCreate(TInstant time) {
            {
                TLightReadGuard read{Lock_};
                if (auto it = Pipelines_.find(time); it != Pipelines_.end()) {
                    return it->second;
                }
            }

            {
                TLightWriteGuard write{Lock_};
                if (Pipelines_.empty() || time >= Pipelines_.begin()->first) {
                    auto it = Pipelines_.emplace(time, MakeIntrusive<TPipeline>(AggrName_)).first;
                    return it->second;
                }
            }

            // received too old data
            return nullptr;
        }

        TIntrusivePtr<TPipeline> Pop(TInstant iterationTime) {
            TLightWriteGuard write{Lock_};
            auto node = Pipelines_.extract(iterationTime);
            return node ? node.mapped() : nullptr;
        }

        TMaybe<TInstant> GetMin() const {
            TLightReadGuard read{Lock_};
            if (Pipelines_.empty()) {
                return Nothing();
            }
            return Pipelines_.begin()->first;
        }

        size_t Size() const {
            TLightReadGuard read{Lock_};
            return Pipelines_.size();
        }

    private:
        const TString AggrName_;
        TLightRWLock Lock_;
        TMap<TInstant, TIntrusivePtr<TPipeline>> Pipelines_;
    };
}

class TRealtimeDelays {
public:
    void Add(TDuration value) {
        with_lock(Lock_) {
            Values_.push_back(value);
        }
    }

    TVector<TDuration> Take() {
        TVector<TDuration> result;
        with_lock(Lock_) {
            result = std::move(Values_);
        }
        return result;
    }

private:
    TAdaptiveLock Lock_;
    TVector<TDuration> Values_;
};

class TServerWorker::TImpl {
    static constexpr size_t MaxPipelines = 4;
public:
    TImpl(TString aggrName, size_t workersCount, ui16 subscriptionsPort, size_t threads)
        : Pipelines_{std::move(aggrName)}
        , WorkersCount_{workersCount}
        , Executor_{"Wrk/Process", threads}
        , SubscriptionsClient_{subscriptionsPort}
    {
        // use separate thread to parse and update subscriptions
        UpdaterHandler_ = NCron::StartPeriodicJob([this]{
            UpdateSubscriptions();
        }, TDuration::Seconds(10), "Wrk/Subscriptions");
    }

    ~TImpl() {
        UpdaterHandler_.Destroy();
        Executor_.Stop();
    }

    void UpdateSubscriptions() {
        try {
            auto startTime = TInstant::Now();

            auto resp = SubscriptionsClient_.ListAll().GetValueSync();
            auto* subscriptions = new TLocalSubscriptions{resp};
            Subscriptions_.AtomicStore(subscriptions);

            TDuration elapsedTime = TInstant::Now() - startTime;
            Cerr << TInstant::Now() << " loaded " << subscriptions->GetSubscriptions().size() << " subscriptions, time=" << elapsedTime << Endl;
        } catch (...) {
            Cerr << TInstant::Now() << " cannot update subscriptions: " << CurrentExceptionMessage() << Endl;
        }
    }

    void Process(TInstant iterationTime, TString worker, TString data) {
        Y_UNUSED(worker);

        Executor_.AddAndForget([this, iterationTime, data = std::move(data)]() {
            auto pipeline = Pipelines_.GetOrCreate(iterationTime);
            if (!pipeline) {
                // received too old data, skip it completely
                return;
            }


            size_t workersDone = pipeline->LoadData(data);
            if (workersDone < WorkersCount_) {
                // not all workers sent data in current iteration
                return;
            }

            auto subscriptions = Subscriptions_.AtomicLoad();
            if (!subscriptions) {
                // subscriptions not yet loaded
                return;
            }

            // first process outdated data
            while (true) {
                auto expiredTime = Pipelines_.GetMin();
                if (expiredTime.Empty() || *expiredTime >= iterationTime) {
                    break;
                }

                if (auto expiredPipeline = Pipelines_.Pop(*expiredTime)) {
                    auto values = expiredPipeline->Finalize(*expiredTime, subscriptions);
                    PushValues(std::move(values));
                }
            }

            // then process current iteration
            pipeline = Pipelines_.Pop(iterationTime);
            if (pipeline) {
                auto values = pipeline->Finalize(iterationTime, subscriptions);
                PushValues(std::move(values))
                        .Subscribe([iterationTime, this](const NThreading::TFuture<void>&) {
                            TDuration delay = TInstant::Now() - iterationTime;
                            RealtimeDelays_.Add(delay);
                            Cerr << TInstant::Now() << " realtime delay " << delay << Endl;
                        });
            }

            // ensure that we keep only fixed amount of active pipelines
            while (Pipelines_.Size() > MaxPipelines) {
                if (auto expiredTime = Pipelines_.GetMin()) {
                    if (auto expiredPipeline = Pipelines_.Pop(*expiredTime)) {
                        auto values = expiredPipeline->Finalize(*expiredTime, subscriptions);
                        PushValues(std::move(values));
                    }
                }
            }
        });
    }

    TVector<TDuration> RealtimeDelays() {
        return RealtimeDelays_.Take();
    }

    NThreading::TFuture<void> PushValues(TString values) {
        return SubscriptionsClient_.PushValues(std::move(values));
    }

private:
    TPipelinesMap Pipelines_;
    size_t WorkersCount_;
    NYasm::NCommon::TBaseThreadedExecutor Executor_;
    TSubscriptionsClient SubscriptionsClient_;
    NCron::IHandlePtr UpdaterHandler_;
    THotSwap<TLocalSubscriptions> Subscriptions_;
    TRealtimeDelays RealtimeDelays_;
};

TServerWorker::TServerWorker(TString aggrName, size_t workersCount, ui16 subscriptionsPort, size_t threads)
    : Impl{new TImpl{std::move(aggrName), workersCount, subscriptionsPort, threads}}
{
}

TServerWorker::~TServerWorker() {
    // needed for pimpl
}

void TServerWorker::Process(TInstant iterationTime, TString worker, TString data) {
    Impl->Process(iterationTime, std::move(worker), std::move(data));
}

TVector<TDuration> TServerWorker::RealtimeDelays() {
    return Impl->RealtimeDelays();
}
