#include "subscriptions.h"
#include "common.h"

#include <infra/monitoring/common/msgpack.h>
#include <infra/yasm/interfaces/internal/subscription_store.pb.h>
#include <infra/yasm/zoom/components/serialization/common/msgpack_utils.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/subscription.h>

#include <library/cpp/pybind/ptr.h>

namespace {
    TVector<NZoom::NSubscription::TSubscription> UnpackSubscriptions(const msgpack::object_array& arr) {
        TVector<NZoom::NSubscription::TSubscription> result;
        size_t failedCount = 0;

        for (const auto& subscriptionDef: NMonitoring::TArrayIterator(arr)) {
            try {
                Y_ENSURE(subscriptionDef.type == msgpack::type::MAP);

                TStringBuf hostName;
                TStringBuf requestKey;
                TStringBuf signalExpression;
                bool allowLegacyTypes = NZoom::NSubscription::TSubscription::ALLOW_LEGACY_TYPES_DEFAULT;

                for (const auto& subscriptionField: NMonitoring::TMapIterator(subscriptionDef.via.map)) {
                    TStringBuf fieldName = NZoom::NPython::AsStrBuf(subscriptionField.key);
                    if (fieldName == NZoom::NSubscription::HOST_FIELD_NAME) {
                        hostName = NZoom::NPython::AsStrBuf(subscriptionField.val);
                    } else if (fieldName == NZoom::NSubscription::TAGS_FIELD_NAME) {
                        requestKey = NZoom::NPython::AsStrBuf(subscriptionField.val);
                    } else if (fieldName == NZoom::NSubscription::SIGNAL_FIELD_NAME) {
                        signalExpression = NZoom::NPython::AsStrBuf(subscriptionField.val);
                    } else if (fieldName == NZoom::NSubscription::ALLOW_LEGACY_TYPES_FIELD_NAME) {
                        Y_ENSURE(subscriptionField.val.type == msgpack::type::BOOLEAN);
                        allowLegacyTypes = subscriptionField.val.as<bool>();
                    }
                }

                if (!hostName.empty() && !requestKey.empty() && !signalExpression.empty()) {
                    result.emplace_back(
                            NZoom::NHost::THostName{hostName},
                            NZoom::NSubscription::TInternedRequestKey{requestKey},
                            TString{signalExpression},
                            allowLegacyTypes);
                } else {
                    ++failedCount;
                }
            }  catch (...) {
                Cerr << TInstant::Now() << " ignore wrong subscription: " << CurrentExceptionMessage() << Endl;
                ++failedCount;
            }
        }

        if (failedCount > 0) {
            Cerr << TInstant::Now() << " failed to parse " << failedCount << " of " << arr.size << " subscriptions";
        }

        return result;
    }
}

namespace NZoom {
    namespace NPython {
        TVector<NSubscription::TSubscription> DeserializeSubscriptions(PyObject* subscriptions) {
            TVector<NSubscription::TSubscription> result;

            NPyBind::TPyObjectPtr iterator(PyObject_GetIter(subscriptions), true);
            if (iterator.Get() == nullptr) {
                ythrow yexception() << "Value is not iterable";
            }

            PyObject* subscriptionEntry;
            while (subscriptionEntry = PyIter_Next(iterator.Get())) {
                NPyBind::TPyObjectPtr subscriptionEntryPtr(subscriptionEntry, true);
                TString hostName;
                TString requestKey;
                TString signalExpression;

                IteratePyDict(subscriptionEntryPtr.Get(), [&](const TString& fieldName, PyObject* value) {
                    if (fieldName == NSubscription::HOST_FIELD_NAME) {
                        hostName = ObjectToString(value);
                    } else if (fieldName == NSubscription::TAGS_FIELD_NAME) {
                        requestKey = ObjectToString(value);
                    } else if (fieldName == NSubscription::SIGNAL_FIELD_NAME) {
                        signalExpression = ObjectToString(value);
                    }
                });

                try {
                    // ignore wrong subscriptions
                    if (!hostName.empty() && !requestKey.empty() && !signalExpression.empty()) {
                        result.emplace_back(hostName, requestKey, signalExpression);
                    }
                } catch (...) {
                    // ignore wrong subscriptions
                    Cerr << TInstant::Now() << " ignore wrong subscription: " << CurrentExceptionMessage() << Endl;
                }
            }
            return result;
        }

        TVector<NSubscription::TSubscription> DeserializeSubscriptions(TStringBuf msgPack) {
            auto request = msgpack::unpack(msgPack.data(), msgPack.size());
            Y_ENSURE(request->type == msgpack::type::MAP, "wrong type of object given");

            for (const auto& requestField: NMonitoring::TMapIterator(request->via.map)) {
                TStringBuf fieldName = AsStrBuf(requestField.key);
                if (fieldName != NSubscription::SUBSCRIPTION_LIST_FIELD_NAME) {
                    // skip not interested fields
                    continue;
                }

                Y_ENSURE(requestField.val.type == msgpack::type::ARRAY, "wrong type of subscription field");
                return UnpackSubscriptions(requestField.val.via.array);
            }

            return {};
        }

        TString TSubscriptionStoreProtobufRequestSerializer::SubscriptionsDictToListRequest(PyObject* subscriptionsDict) {
            using google::protobuf::Arena;
            Arena arena;
            auto request = Arena::CreateMessage<NYasm::NInterfaces::NInternal::TSubscriptionListRequest>(&arena);

            NZoom::NProtobuf::THostNameSerializer hostNameSerializer(request->MutableHostNameTable());
            NZoom::NProtobuf::TInternedRequestKeySerializer requestKeySerializer(request->MutableRequestKeyTable());
            NZoom::NProtobuf::TSignalExpressionSerializer signalNameSerializer(request->MutableSignalNameTable());
            NZoom::NProtobuf::TSubscriptionSerializer serializer(hostNameSerializer, requestKeySerializer, signalNameSerializer);

            THashSet<NSubscription::TSubscription> subSet;

            IteratePyDict(subscriptionsDict, [request, &serializer, &subSet](const TString& hostName, PyObject* tagsToSignals) {
                if (!hostName.empty()) {
                    NHost::THostName host(hostName);
                    IteratePyDict(tagsToSignals,
                        [request, &serializer, &host, &subSet](const TString& tags, PyObject* signals) {
                            if (!tags.empty()) {
                                if (PySet_Check(signals)) {
                                    try {
                                        NSubscription::TInternedRequestKey requestKey(tags);
                                        NPyBind::TPyObjectPtr iterator(PyObject_GetIter(signals), true);
                                        if (iterator.Get()) {
                                            PyObject* signalObject = nullptr;
                                            while (signalObject = PyIter_Next(iterator.Get())) {
                                                NPyBind::TPyObjectPtr signalObjectPtr(signalObject, true);
                                                TString signal = ObjectToString(signalObjectPtr.Get());
                                                if (!signal.empty()) {
                                                    try {
                                                        NSubscription::TSubscription subscription(host, requestKey, signal);
                                                        if (subSet.insert(subscription).second) {
                                                            serializer.Intern(subscription,
                                                                request->MutableSubscriptions()->Add());
                                                        }
                                                    } catch (...) {
                                                        // ignore wrong subscriptions
                                                    }
                                                }
                                            }
                                        }
                                    } catch (...) {
                                        // Ignore subscriptions with incorrect tags
                                    }
                                }
                            }
                        });
                }
            });
            return request->SerializeAsString();
        }
    }
}
