#include "responses.h"

#include <infra/yasm/zoom/components/serialization/common/msgpack_utils.h>
#include <infra/yasm/zoom/components/serialization/common/codec_utils.h>
#include <infra/yasm/zoom/components/serialization/deserializers/msgpack.h>
#include <infra/yasm/zoom/components/serialization/python/zoom_to_py.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/record.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/instance_key.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/signal_name.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/subscription.h>
#include <infra/yasm/zoom/components/subscription/subscription.h>
#include <infra/yasm/common/labels/tags/verification.h>
#include <infra/yasm/interfaces/internal/subscription_store.pb.h>
#include <infra/node_agent/api/proto/agent_rest.pb.h>

#include <library/cpp/json/json_reader.h>

#include <util/string/cast.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

using namespace NZoom::NPython;

namespace {
    static NJson::TJsonValue EMPTY_ARRAY(NJson::JSON_ARRAY);
    static NJson::TJsonValue EMPTY_MAP(NJson::JSON_MAP);

    bool IsCorrectInstanceKey(const NTags::TInstanceKey& key) {
        if (!key.Empty() && NTags::IsCorrectItype(key.GetItype())) {
            for (const auto& tag: key.GetTags()) {
                if (!NTags::IsCorrectTagName(tag.Name) || !NTags::IsCorrectInstanceTagValue(tag.Value)) {
                    return false;
                }
            }
            return true;
        } else {
            return false;
        }
    }
}

TAgentResponseParser::TAgentResponseParser(TStringBuf value) {
    NJson::ReadJsonFastTree(value, &ParsedValue, true);
    if (!ParsedValue.IsMap()) {
        ythrow yexception() << "Value is not map";
    }
}

TMaybe<TString> TAgentResponseParser::GetStatus() const {
    const auto it = ParsedValue.GetMap().find("status");
    if (it == ParsedValue.GetMap().end()) {
        return Nothing();
    }
    return it->second.GetString();
}

TAgentResponseParser::TAgentResponseParserIterator TAgentResponseParser::GetRecordIterator() const {
    const auto it = ParsedValue.GetMap().find("aggr");
    if (it == ParsedValue.GetMap().end()) {
        return TAgentResponseParserIterator(EMPTY_ARRAY);
    }
    return TAgentResponseParserIterator(it->second);
}

TAgentResponseParser::TAgentResponseParserIterator::TAgentResponseParserIterator()
    : InnerValue(&EMPTY_MAP)
{
}

TAgentResponseParser::TAgentResponseParserIterator::TAgentResponseParserIterator(const NJson::TJsonValue& innerValue)
    : InnerValue(&innerValue)
{
    if (!InnerValue->IsArray()) {
        ythrow yexception() << "Value is not array";
    }
}

bool TAgentResponseParser::TAgentResponseParserIterator::IsValid() const {
    return Position != InnerValue->GetArray().size();
}

void TAgentResponseParser::TAgentResponseParserIterator::Next() {
    ++Position;
}

const TString& TAgentResponseParser::TAgentResponseParserIterator::GetIType() const {
    return (*InnerValue)[Position][0].GetString();
}

const TString& TAgentResponseParser::TAgentResponseParserIterator::GetTail() const {
    return (*InnerValue)[Position][1].GetString();
}

NZoom::NRecord::TRecord TAgentResponseParser::TAgentResponseParserIterator::CurrentRecords() const {
    const NJson::TJsonValue& value = (*InnerValue)[Position][2];
    if (!value.IsMap()) {
        ythrow yexception() << "Record entry is not map";
    }
    return TJsonPyDeserializer::Deserialize(value, true);
}

NTags::TInstanceKey TAgentResponseParser::TAgentResponseParserIterator::GetInstanceKey() const {
    return NTags::TInstanceKey::FromAgent(GetIType(), GetTail());
}

THolder<NZoom::NRecord::TTaggedRecord> TAgentResponseParser::TAgentResponseParserIterator::TaggedRecord() {
    TVector<std::pair<NTags::TInstanceKey, NZoom::NRecord::TRecord>> items;
    while (IsValid()) {
        const auto key(GetInstanceKey());
        if (IsCorrectInstanceKey(key)) {
            // TODO: metrics?
            items.emplace_back(key, CurrentRecords());
        }
        Next();
    }
    return MakeHolder<NZoom::NRecord::TTaggedRecord>(std::move(items));
}

TAgentProtobufResponseParser::TAgentProtobufResponseParser(TStringBuf valueStr)
    : Impl(MakeHolder<TImpl>(valueStr))
    , PerInstanceRecords()
{
}

TAgentProtobufResponseParser::~TAgentProtobufResponseParser() {
}

class TAgentProtobufResponseParser::TImpl {
public:
    TImpl(TStringBuf valueStr)
        : Response(google::protobuf::Arena::CreateMessage<NYasm::NInterfaces::NInternal::TAgentResponse>(&Arena))
    {
        TMemoryInput stream(valueStr);
        Response->ParseFromArcadiaStream(&stream);
    }

    std::pair<THolder<NZoom::NRecord::TTaggedRecord>, size_t> TaggedRecord() const {
        TVector<std::pair<NTags::TInstanceKey, NZoom::NRecord::TRecord>> items;
        NZoom::NProtobuf::TInstanceKeyDeserializer instanceKeyDeserializer(
            Response->GetAggregatedRecords().GetInstanceKeyTable()
        );
        NZoom::NProtobuf::TSignalNameDeserializer signalNameDeserializer(
            Response->GetAggregatedRecords().GetSignalNameTable()
        );
        size_t ignoredRecords = 0;
        for (const auto& row : Response->GetAggregatedRecords().GetRecords()) {
            auto instanceKey(instanceKeyDeserializer.Deserialize(row.GetInstanceKey()));
            if (IsCorrectInstanceKey(instanceKey)) {
                items.emplace_back(instanceKey, NZoom::NProtobuf::DeserializeProtobufRecord(
                    row.GetRecord(), signalNameDeserializer,
                    true /* drop zero ugram buckets in per instance records (used in server code) */
                ));
            } else {
               ++ignoredRecords;
            }
        }
        return std::make_pair(MakeHolder<NZoom::NRecord::TTaggedRecord>(std::move(items)), ignoredRecords);
    }

    TVector<std::pair<TString, NZoom::NRecord::TSingleMultiTaggedRecord>> PerInstanceRecords() const {
        const auto& perInstanceRecordsProto = Response->GetPerInstanceRecords();
        NZoom::NProtobuf::TInstanceKeyDeserializer instanceKeyDeserializer(perInstanceRecordsProto.GetInstanceKeyTable());
        NZoom::NProtobuf::TSignalNameDeserializer signalNameDeserializer(perInstanceRecordsProto.GetSignalNameTable());

        TVector<std::pair<TString, NZoom::NRecord::TSingleMultiTaggedRecord>> result(
            Reserve(perInstanceRecordsProto.RecordsSize())
        );
        for (const auto& recordProto : perInstanceRecordsProto.GetRecords()) {
            NZoom::NRecord::TRecord values = NZoom::NProtobuf::DeserializeProtobufRecord(
                recordProto.GetRecord(), signalNameDeserializer,
                false /* keep zero ugram buckets in per instance records (used in agent code) */
            );
            TVector<NTags::TInstanceKey> keys(Reserve(recordProto.InstanceKeysSize()));
            for (const auto& instanceKeyProto: recordProto.GetInstanceKeys()) {
                auto instanceKey = instanceKeyDeserializer.Deserialize(instanceKeyProto);
                if (!instanceKey.Empty()) {
                    keys.push_back(std::move(instanceKey));
                }
            }
            result.emplace_back(
                recordProto.GetInstanceName(),
                NZoom::NRecord::TSingleMultiTaggedRecord(std::move(keys), std::move(values))
            );
        }
        return result;
    }

    NYasm::NInterfaces::NInternal::EAgentStatus GetStatusCode() const {
        return Response->GetStatus().GetStatus();
    }

    TMaybe<TString> GetStatus() const {
        using NYasm::NInterfaces::NInternal::EAgentStatus;
        switch (GetStatusCode()) {
            case EAgentStatus::STATUS_STARTING: {
                return "starting";
            }
            case EAgentStatus::STATUS_NO_DATA: {
                return "no data";
            }
            case EAgentStatus::STATUS_OK: {
                return "ok";
            }
            default: {
                return Nothing();
            }
        }
    }

private:
    google::protobuf::Arena Arena;
    NYasm::NInterfaces::NInternal::TAgentResponse* Response;
};

std::pair<THolder<NZoom::NRecord::TTaggedRecord>, size_t> TAgentProtobufResponseParser::TaggedRecord() const {
    return Impl->TaggedRecord();
}

TMaybe<TString> TAgentProtobufResponseParser::GetStatus() const {
    return Impl->GetStatus();
}

NYasm::NInterfaces::NInternal::EAgentStatus TAgentProtobufResponseParser::GetStatusCode() const {
    return Impl->GetStatusCode();
}

TPerInstanceRecordsIterator NZoom::NPython::TAgentProtobufResponseParser::LoadPerInstanceRecords() {
    PerInstanceRecords = Impl->PerInstanceRecords();
    return TPerInstanceRecordsIterator(&PerInstanceRecords);
}

void TStreamingResponseLoader::Unpack(TStringBuf strResponse) {
    const size_t chunkSize = 8192;
    TMemoryInput stream(strResponse);
    msgpack::unpacker unpacker;
    msgpack::unpacked message;
    while (true) {
        unpacker.reserve_buffer(chunkSize);
        const size_t readSize(stream.Read(unpacker.buffer(), chunkSize));
        if (!readSize) {
            // TODO: check that buffer is completely processed
            return;
        }
        unpacker.buffer_consumed(readSize);
        while (unpacker.next(message)) {
            OnMessage(message.get());
        }
    }
}

NZoom::NRecord::TRecord TStreamingResponseLoader::UnpackRecord(const msgpack::object& obj) {
    if (obj.type != msgpack::type::MAP) {
        ythrow yexception() << "signals is not map";
    }
    const msgpack::object_map signals = obj.via.map;
    TVector<std::pair<NZoom::NSignal::TSignalName, NZoom::NValue::TValue>> items;
    items.reserve(signals.size);
    for (size_t idx = 0; idx < signals.size; ++idx) {
        const auto& kv = signals.ptr[idx];
        items.emplace_back(
            kv.key.as<TStringBuf>(),
            TMsgPackValueHierarchy::DeserializeValue(kv.val, true)
        );
    }
    return {std::move(items)};
}

void TMiddleResponseLoader::OnMessage(const msgpack::object& obj) {
    if (obj.type != msgpack::type::ARRAY) {
        ythrow yexception() << "root is not array";
    }

    const msgpack::object_array root(obj.via.array);
    if (root.size != 2) {
        ythrow yexception() << "root has wrong size";
    }
    if (root.ptr[1].type != msgpack::type::MAP) {
        ythrow yexception() << "signals is not map";
    }

    Records.emplace_back(
        NTags::TInstanceKey::FromNamed(root.ptr[0].as<TStringBuf>()),
        UnpackRecord(root.ptr[1])
    );
}

NZoom::NRecord::TTaggedRecord* TMiddleResponseLoader::LoadTaggedRecord(TStringBuf strResponse) {
    Unpack(strResponse);
    return new NZoom::NRecord::TTaggedRecord(std::move(Records));
}

void TServerResponseLoader::OnMessage(const msgpack::object& obj) {
    if (obj.type != msgpack::type::ARRAY) {
        ythrow yexception() << "root is not array";
    }

    const msgpack::object_array root(obj.via.array);
    if (root.size != 2) {
        ythrow yexception() << "root has wrong size";
    }
    if (root.ptr[1].type != msgpack::type::MAP) {
        ythrow yexception() << "tag signals is not map";
    }

    auto& dest(Hosts.emplace_back(root.ptr[0].as<TString>(), TTagRecordVector()).second);
    const msgpack::object_map source = root.ptr[1].via.map;
    dest.reserve(source.size);
    for (size_t idx = 0; idx < source.size; ++idx) {
        const auto& kv = source.ptr[idx];
        dest.emplace_back(
            NTags::TInstanceKey::FromNamed(kv.key.as<TStringBuf>()),
            UnpackRecord(kv.val)
        );
    }
}

TVector<std::pair<TString, NZoom::NRecord::TTaggedRecord*>> TServerResponseLoader::LoadHostRecords(TStringBuf strResponse) {
    Unpack(strResponse);
    TVector<std::pair<TString, NZoom::NRecord::TTaggedRecord*>> result;
    result.reserve(Hosts.size());
    for (auto& record : Hosts) {
        result.emplace_back(std::move(record.first), new NZoom::NRecord::TTaggedRecord(std::move(record.second)));
    }
    return result;
}

void TGetSubscriptionValuesResponseLoader::Load(const TStringBuf& protobufMessage) {
    Clear();

    THashMap<NZoom::NHost::THostName, TVector<NZoom::NSubscription::TSubscriptionWithValueSeries>> groupedValues;

    {
        using google::protobuf::Arena;
        Arena arena;
        auto response = Arena::CreateMessage<NYasm::NInterfaces::NInternal::TGetValuesResponse>(&arena);
        Y_PROTOBUF_SUPPRESS_NODISCARD response->ParseFromArray(protobufMessage.data(), protobufMessage.size());
        if (!response->GetRawRequestKeys().empty() &&
            response->GetRawRequestKeys().size() != response->GetSubscriptions().size()) {
            ythrow yexception() << "Unexpected /get_values response format";
        }

        NZoom::NProtobuf::THostNameDeserializer hostNameDeserializer(response->GetHostNameTable());
        NZoom::NProtobuf::TInternedRequestKeyDeserializer requestKeyDeserializer(response->GetRequestKeyTable());
        NZoom::NProtobuf::TSignalExpressionDeserializer signalNameDeserializer(response->GetSignalNameTable());
        NZoom::NProtobuf::TSubscriptionWithValueSeriesDeserializer deserializer(hostNameDeserializer, requestKeyDeserializer,
            signalNameDeserializer);

        for (const auto& subscriptionProto: response->GetSubscriptions()) {
            NZoom::NSubscription::TSubscriptionWithValueSeries s = deserializer.Deserialize(subscriptionProto);
            auto& hostSubscriptions = groupedValues[s.Subscription.GetHostName()];
            hostSubscriptions.reserve(response->GetSubscriptions().size());
            hostSubscriptions.push_back(std::move(s));
        }
    }

    Items.reserve(groupedValues.size());
    for (auto& [hostName, hostValues]: groupedValues) {
        Items.emplace_back(hostName, std::move(hostValues));
    }
}

namespace {
    struct TResponseState {
        TResponseState()
            : Response(nullptr)
            , HostNameSerializer()
            , RequestKeySerializer()
            , SignalNameSerializer() {
        }
        NYasm::NInterfaces::NInternal::TGetValuesResponse* Response;
        THolder<NZoom::NProtobuf::THostNameSerializer> HostNameSerializer;
        THolder<NZoom::NProtobuf::TInternedRequestKeySerializer> RequestKeySerializer;
        THolder<NZoom::NProtobuf::TSignalExpressionSerializer> SignalNameSerializer;

        bool Empty() const { return !Response; }

        void Init(NYasm::NInterfaces::NInternal::TGetValuesResponse* response) {
            Response = response;
            HostNameSerializer = MakeHolder<NZoom::NProtobuf::THostNameSerializer>(Response->MutableHostNameTable());
            RequestKeySerializer = MakeHolder<NZoom::NProtobuf::TInternedRequestKeySerializer>(Response->MutableRequestKeyTable());
            SignalNameSerializer = MakeHolder<NZoom::NProtobuf::TSignalExpressionSerializer>(Response->MutableSignalNameTable());
        }
    };
}

TVector<TString> TGetSubscriptionValuesResponseLoader::Split(const TStringBuf& protobufResponse,
                                                             const TSubscriptionSplitter& splitter,
                                                             ui64& maxStartTime) {
    using google::protobuf::Arena;
    Arena arena;

    TVector<TString> result(splitter.GetBucketCount());
    maxStartTime = 0;

    auto response = Arena::CreateMessage<NYasm::NInterfaces::NInternal::TGetValuesResponse>(&arena);
    Y_PROTOBUF_SUPPRESS_NODISCARD response->ParseFromArray(protobufResponse.data(), protobufResponse.size());
    const bool haveRawRequestKeys = !response->GetRawRequestKeys().empty();
    if (haveRawRequestKeys && response->GetRawRequestKeys().size() != response->GetSubscriptions().size()) {
        ythrow yexception() << "Unexpected /get_values response format";
    }
    if (response->SubscriptionsSize()) {
        const auto idealInitialCapacity = Max(static_cast<int>(response->SubscriptionsSize() / splitter.GetBucketCount()), 4);

        TVector<TResponseState> responses(splitter.GetBucketCount());
        while (response->SubscriptionsSize() > 0) {
            // Since we use the single arena to avoid copying we call UnsafeArena*() group of functions
            auto* movedSubscription = response->MutableSubscriptions()->UnsafeArenaReleaseLast();

            maxStartTime = Max(maxStartTime, movedSubscription->GetStartTimestamp());
            const TString& hostName = response->GetHostNameTable().GetName(
                movedSubscription->GetSubscription().GetHostName().GetIndex());
            const TString& requestKey = response->GetRequestKeyTable().GetName(
                movedSubscription->GetSubscription().GetRequestKey().GetIndex());
            const TString& signal = response->GetSignalNameTable().GetName(
                movedSubscription->GetSubscription().GetSignalExpression().GetIndex());
            auto workerId = splitter.GetSubscriptionBucket(hostName, requestKey);

            // Find or initialize message where to move the current message
            TResponseState& responseChunkState = responses[workerId];
            if (responseChunkState.Empty()) {
                responseChunkState.Init(Arena::CreateMessage<NYasm::NInterfaces::NInternal::TGetValuesResponse>(&arena));
                responseChunkState.Response->MutableSubscriptions()->Reserve(idealInitialCapacity);
                if (haveRawRequestKeys) {
                    responseChunkState.Response->MutableRawRequestKeys()->Reserve(idealInitialCapacity);
                }
            }

            auto* responseChunk = responseChunkState.Response;
            responseChunk->MutableSubscriptions()->UnsafeArenaAddAllocated(movedSubscription);
            if (haveRawRequestKeys) {
                responseChunk->MutableRawRequestKeys()->UnsafeArenaAddAllocated(
                    response->MutableRawRequestKeys()->UnsafeArenaReleaseLast());
            }

            // Update host name and request key indices and update their tables in response chunk
            responseChunkState.HostNameSerializer->Intern(hostName,
                movedSubscription->MutableSubscription()->MutableHostName());
            responseChunkState.RequestKeySerializer->Intern(requestKey,
                movedSubscription->MutableSubscription()->MutableRequestKey());
            responseChunkState.SignalNameSerializer->Intern(signal,
                movedSubscription->MutableSubscription()->MutableSignalExpression());
        }

        for (size_t i = 0; i < responses.size(); ++i) {
            const auto& responseState = responses[i];
            if (!responseState.Empty()) {
                result[i] = responseState.Response->SerializeAsString();
            }
        }
    }
    return result;
}

void TGetSubscriptionValuesResponseLoader::Clear() {
    Items.clear();
}

THostSubscriptionsIterator TGetSubscriptionValuesResponseLoader::GetHostSubscriptionsIterator() noexcept {
    return THostSubscriptionsIterator(Items);
}

TSubscriptionSplitter::TSubscriptionSplitter(ui64 bucketCount)
    : BucketCount(bucketCount)
    , HostToMetagroup() {
    if (0 == bucketCount) {
        ythrow yexception() << "Splitter can not split into 0 elements";
    }
}

void TSubscriptionSplitter::SetMetagroupForGroup(const NZoom::NHost::THostName& groupName,
                                                 const NZoom::NHost::THostName& metagroupName) {
    HostToMetagroup.emplace(groupName, metagroupName);
}

void TSubscriptionSplitter::SetMetagroupForGroup(const TStringBuf& groupName, const TStringBuf& metagroupName) {
    SetMetagroupForGroup(NZoom::NHost::THostName(groupName), NZoom::NHost::THostName(metagroupName));
}

ui64 TSubscriptionSplitter::GetSubscriptionBucket(const NZoom::NHost::THostName& host,
                                                  const NZoom::NSubscription::TInternedRequestKey& requestKey) const {
    const auto it = HostToMetagroup.find(host);
    if (it != HostToMetagroup.end()) {
        return NSubscription::GetSubscriptionBucket(it->second, requestKey, BucketCount);
    } else {
        return NSubscription::GetSubscriptionBucket(host, requestKey, BucketCount);
    }
}

ui64 TSubscriptionSplitter::GetSubscriptionBucket(const TStringBuf& host, const TStringBuf& tags) const {
    return GetSubscriptionBucket(NZoom::NHost::THostName(host), NZoom::NSubscription::TInternedRequestKey(tags));
}

ui64 TSubscriptionSplitter::GetSubscriptionBucket(const TString& host, const TString& tags) const {
    return GetSubscriptionBucket(NZoom::NHost::THostName(host), NZoom::NSubscription::TInternedRequestKey(tags));
}

ui64 TSubscriptionSplitter::GetBucketCount() const {
    return BucketCount;
}

const TString& THostSubscriptionsIterator::GetHostName() const {
    return Items->at(Index).first.GetName();
}

TVector<NZoom::NSubscription::TSubscriptionWithValueSeries>* THostSubscriptionsIterator::ExtractValues() {
    return new TVector<NZoom::NSubscription::TSubscriptionWithValueSeries>(std::move(Items->at(Index).second));
}

TPerInstanceRecordsIterator::TPerInstanceRecordsIterator()
    : RecordsVector(nullptr)
    , Position(0) {
}

TPerInstanceRecordsIterator::TPerInstanceRecordsIterator(TPerInstanceRecordsIterator::TRecordsVector* recordsVector)
    : RecordsVector(recordsVector)
    , Position(0) {
}

NZoom::NRecord::TSingleMultiTaggedRecord* TPerInstanceRecordsIterator::ExtractRecord() {
    return new NZoom::NRecord::TSingleMultiTaggedRecord(std::move(RecordsVector->at(Position).second));
}

const TString& TPerInstanceRecordsIterator::GetInstanceName() const {
    return RecordsVector->at(Position).first;
}

void TPerInstanceRecordsIterator::MoveNext() noexcept {
    ++Position;
}

bool TPerInstanceRecordsIterator::IsEnd() const noexcept {
    return !RecordsVector || Position >= RecordsVector->size();
}

TNodeAgentResponseLoader::TNodeAgentResponseLoader()
    : UnistatEndpoints()
    , SubagentEndpoints()
    , PodsInfo()
{
}

void TNodeAgentResponseLoader::Clear() {
    UnistatEndpoints.clear();
    SubagentEndpoints.clear();
    PodsInfo.clear();
}

void TNodeAgentResponseLoader::Load(TStringBuf valueStr) {
    Clear();
    google::protobuf::Arena arena;
    auto response = google::protobuf::Arena::CreateMessage<NInfra::NNodeAgent::NApi::NProto::TMonitoringInfo>(&arena);
    TMemoryInput stream(valueStr);
    response->ParseFromArcadiaStream(&stream);

    for (const auto& unistatOptions: response->Getunistats()) {
        THashMap<TString, TString> labels;

        for (const auto& [tag, value]: unistatOptions.Getlabels()) {
            labels[tag] = value;
        }

        UnistatEndpoints.push_back({
            unistatOptions.Getport(),
            unistatOptions.Getpath(),
            unistatOptions.Getprefix(),
            std::move(labels),
            unistatOptions.Getcontainer_fqdn()
        });
    }

    for (const auto& subagentOptions: response->Getyasm_subagents()) {
        SubagentEndpoints.push_back({
           subagentOptions.Getport(),
           subagentOptions.Getcontainer_fqdn()
        });
    }

    for (const auto& podOptions: response->Getpods()) {
        THashMap<TString, TString> labels;

        for (const auto& [tag, value]: podOptions.Getlabels()) {
            labels[tag] = value;
        }

        PodsInfo.push_back({
            podOptions.Getcontainer_id(),
            podOptions.Getprefix(),
            std::move(labels),
            podOptions.Getcontainer_fqdn()
        });
    }

    for (const auto& workloadOptions: response->Getworkloads()) {
        THashMap<TString, TString> labels;

        for (const auto& [tag, value]: workloadOptions.Getlabels()) {
            labels[tag] = value;
        }

        WorkloadsInfo.push_back({
            workloadOptions.Getcontainer_id(),
            workloadOptions.Getprefix(),
            std::move(labels),
            TString() // will be added later
        });
    }

    for (const auto& containerOptions: response->Getsystem_containers()) {
        THashMap<TString, TString> labels;

        for (const auto& [tag, value]: containerOptions.Getlabels()) {
            labels[tag] = value;
        }

        SystemContainersInfo.push_back({
            containerOptions.Getcontainer_id(),
            std::move(labels),
            TString() // will be added later
        });
    }
}

TVector<TNodeAgentUnistatEndpoint> TNodeAgentResponseLoader::ExtractUnistatEndpoints() {
    return std::move(UnistatEndpoints);
}

TVector<TNodeAgentSubagentEndpoint> TNodeAgentResponseLoader::ExtractSubagentEndpoints() {
    return std::move(SubagentEndpoints);
}

TVector<TNodeAgentPodInfo> TNodeAgentResponseLoader::ExtractPodsInfo() {
    return std::move(PodsInfo);
}

TVector<TNodeAgentWorkloadInfo> TNodeAgentResponseLoader::ExtractWorkloadsInfo() {
    return std::move(WorkloadsInfo);
}

TVector<TNodeAgentContainerInfo> TNodeAgentResponseLoader::ExtractSystemContainersInfo() {
    return std::move(SystemContainersInfo);
}
