#pragma once

#include <infra/yasm/interfaces/internal/agent.pb.h>
#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/zoom/components/serialization/python/py_to_zoom.h>
#include <infra/yasm/zoom/components/subscription/subscription.h>

#include <library/cpp/json/json_value.h>

#include <util/stream/mem.h>

#include <contrib/libs/msgpack/include/msgpack.hpp>

namespace NZoom {
    namespace NPython {

        class TAgentResponseParser {
        private:
            NJson::TJsonValue ParsedValue;

        public:
            class TAgentResponseParserIterator {
            private:
                const NJson::TJsonValue* InnerValue = nullptr;
                size_t Position = 0;

                bool IsValid() const;
                void Next();

                const TString& GetIType() const;
                const TString& GetTail() const;
                NTags::TInstanceKey GetInstanceKey() const;
                NZoom::NRecord::TRecord CurrentRecords() const;

            public:
                TAgentResponseParserIterator(const NJson::TJsonValue& innerValue);
                TAgentResponseParserIterator();

                THolder<NZoom::NRecord::TTaggedRecord> TaggedRecord();
            };

            TAgentResponseParser(TStringBuf valueStr);

            TAgentResponseParserIterator GetRecordIterator() const;
            TMaybe<TString> GetStatus() const;
        };

        class TPerInstanceRecordsIterator {
        public:
            using TRecordsVector = TVector<std::pair<TString, NZoom::NRecord::TSingleMultiTaggedRecord>>;

            TPerInstanceRecordsIterator();
            explicit TPerInstanceRecordsIterator(TRecordsVector* recordsVector);

            NZoom::NRecord::TSingleMultiTaggedRecord* ExtractRecord();
            const TString& GetInstanceName() const;

            void MoveNext() noexcept;
            bool IsEnd() const noexcept;
        private:
            TRecordsVector* RecordsVector;
            size_t Position;
        };

        class TAgentProtobufResponseParser {
        public:
            TAgentProtobufResponseParser(TStringBuf valueStr);
            ~TAgentProtobufResponseParser();

            std::pair<THolder<NZoom::NRecord::TTaggedRecord>, size_t> TaggedRecord() const;
            TPerInstanceRecordsIterator LoadPerInstanceRecords();
            TMaybe<TString> GetStatus() const;
            NYasm::NInterfaces::NInternal::EAgentStatus GetStatusCode() const;

        private:
            class TImpl;
            THolder<TImpl> Impl;
            TVector<std::pair<TString, NZoom::NRecord::TSingleMultiTaggedRecord>> PerInstanceRecords;
        };

        struct TNodeAgentUnistatEndpoint {
            int Port;
            TString Path;
            TString Prefix;
            THashMap<TString, TString> Labels;
            TString ContainerFqdn;
        };

        struct TNodeAgentSubagentEndpoint {
            int Port;
            TString ContainerFqdn;
        };

        struct TNodeAgentPodInfo {
            TString ContainerId;
            TString Prefix;
            THashMap<TString, TString> Labels;
            TString ContainerFqdn;
        };

        struct TNodeAgentWorkloadInfo {
            TString ContainerId;
            TString Prefix;
            THashMap<TString, TString> Labels;
            TString ContainerFqdn;
        };

        struct TNodeAgentContainerInfo {
            TString ContainerId;
            THashMap<TString, TString> Labels;
            TString ContainerFqdn;
        };

        class TNodeAgentResponseLoader {
        public:
            TNodeAgentResponseLoader();

            void Load(TStringBuf valueStr);
            TVector<TNodeAgentUnistatEndpoint> ExtractUnistatEndpoints();
            TVector<TNodeAgentSubagentEndpoint> ExtractSubagentEndpoints();
            TVector<TNodeAgentPodInfo> ExtractPodsInfo();
            TVector<TNodeAgentWorkloadInfo> ExtractWorkloadsInfo();
            TVector<TNodeAgentContainerInfo> ExtractSystemContainersInfo();
        private:
            void Clear();
            TVector<TNodeAgentUnistatEndpoint> UnistatEndpoints;
            TVector<TNodeAgentSubagentEndpoint> SubagentEndpoints;
            TVector<TNodeAgentPodInfo> PodsInfo;
            TVector<TNodeAgentWorkloadInfo> WorkloadsInfo;
            TVector<TNodeAgentContainerInfo> SystemContainersInfo;
        };

        struct TInstanceContent {
            TStringBuf Key = nullptr;
            NZoom::NRecord::TContinuousRecord* Record = nullptr;
        };

        class TStreamingResponseLoader {
        public:
            using TTagRecordVector = TVector<std::pair<NTags::TInstanceKey, NZoom::NRecord::TRecord>>;

        protected:
            void Unpack(TStringBuf strResponse);

            NZoom::NRecord::TRecord UnpackRecord(const msgpack::object& obj);

            virtual void OnMessage(const msgpack::object& obj) = 0;
        };

        class TMiddleResponseLoader final : public TStreamingResponseLoader {
        private:
            TTagRecordVector Records;

            void OnMessage(const msgpack::object& obj) override;

        public:
            NZoom::NRecord::TTaggedRecord* LoadTaggedRecord(TStringBuf strResponse);
        };

        class TServerResponseLoader final : public TStreamingResponseLoader {
        private:
            TVector<std::pair<TString, TTagRecordVector>> Hosts;

            void OnMessage(const msgpack::object& obj) override;

        public:
            TVector<std::pair<TString, NZoom::NRecord::TTaggedRecord*>> LoadHostRecords(TStringBuf strResponse);
        };

        class TSubscriptionSplitter {
        public:
            explicit TSubscriptionSplitter(ui64 bucketCount);

            void SetMetagroupForGroup(const TStringBuf& groupName, const TStringBuf& metagroupName);
            void SetMetagroupForGroup(const NHost::THostName& groupName, const NHost::THostName& metagroupName);

            ui64 GetBucketCount() const;
            ui64 GetSubscriptionBucket(const TString& hostName, const TString& requestKey) const;
            ui64 GetSubscriptionBucket(const TStringBuf& hostName, const TStringBuf& requestKey) const;
            ui64 GetSubscriptionBucket(const NHost::THostName& host, const NSubscription::TInternedRequestKey& requestKey) const;
        private:
            const ui64 BucketCount;
            THashMap<NHost::THostName, NHost::THostName> HostToMetagroup;
        };

        using TSubscriptionsHostRecord = std::pair<NZoom::NHost::THostName,
            TVector<NZoom::NSubscription::TSubscriptionWithValueSeries>>;

        class THostSubscriptionsIterator {
        public:
            THostSubscriptionsIterator()
                : Items(nullptr)
                , Index(0) {
            };
            explicit THostSubscriptionsIterator(TVector<TSubscriptionsHostRecord>& items)
                : Items(&items)
                , Index(0) {
            }

            const TString& GetHostName() const;
            TVector<NZoom::NSubscription::TSubscriptionWithValueSeries>* ExtractValues();

            bool IsEnd() const {
                return !Items || Index >= Items->size();
            };

            void MoveNext() {
                ++Index;
            }
        private:
            TVector<TSubscriptionsHostRecord>* Items;
            size_t Index;
        };

        class TGetSubscriptionValuesResponseLoader {
        public:
            // Loads /get_values response and returns max start timestamp of all loaded value series
            void Load(const TStringBuf& protobufResponse);
            // Splits subscriptions in result using splitter and returns protobuf-encoded pieces of original response
            static TVector<TString> Split(const TStringBuf& protobufResponse, const TSubscriptionSplitter& splitter,
                                          ui64& maxStartTime);

            THostSubscriptionsIterator GetHostSubscriptionsIterator() noexcept;
            void Clear();
        private:
            TVector<TSubscriptionsHostRecord> Items;
        };
    }
}
