#include "unistat.h"

#include <infra/yasm/zoom/components/serialization/deserializers/json_callbacks/callbacks.h>

#include <library/cpp/json/fast_sax/parser.h>

#include <util/generic/strbuf.h>

using namespace NZoom::NAccumulators;
using namespace NZoom::NHgram;
using namespace NZoom::NPython;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;

namespace NZoom::NPrivate {
    static constexpr char SIGNAL_MODULE_DELIMITER = '-';
    static constexpr char SIGNAL_TAGS_DELIMITER = ';';

    static constexpr ui32 SIGNAL_NAME_LIMIT = 128;

    bool IsErroredStatus(EDeserializationStatus status) {
        return status == EDeserializationStatus::INVALID_UNSPECIFIED ||
               status == EDeserializationStatus::EXTRA_DEPTH ||
               status == EDeserializationStatus::TOO_MUCH_ARGUMENTS_IN_BUCKET;
    }

    inline bool IsHgram(const TSignalName& name) noexcept {
        const auto* aggRules = name.GetAggregationRules();
        return aggRules && aggRules->GetAccumulatorType(EAggregationMethod::Agent) == EAccumulatorType::Hgram;
    }

    struct TUnistatValueCallback;
    class THgramModeSelectingCallback;

    class THgramCallback: public TAbstactJsonCallback {
        TZoomJsonDeserializingCallback& Callback;
        THgramModeSelectingCallback* Parent = nullptr;

        NZoom::NHgram::THgram Hgram = NZoom::NHgram::THgram::Default();

    public:
        THgramCallback(TZoomJsonDeserializingCallback& callback, THgramModeSelectingCallback* parent)
            : Callback(callback)
            , Parent(parent)
        {
        }

        void Reset() {
            // NOTE(rocco66): array value for dhhh case
            Hgram = NZoom::NHgram::THgram::Default();
        }

        bool OnInteger(long long v) final {
            return OnDouble(double(v));
        }

        bool OnUInteger(unsigned long long v) final {
            return OnDouble(double(v));
        }

        bool OnDouble(double v) final {
            Hgram.MulFloat(v);
            return true;
        }

        bool OnCloseArray() final;
    };

    class TUgramHgramCallback: public TAbstactJsonCallback {
        TZoomJsonDeserializingCallback& Callback;
        TSkippingJsonCallback Skipping;
        THgramModeSelectingCallback* Parent = nullptr;

        NZoom::NHgram::TUgramBuckets Buckets;

        double First = -1.0;
        double Second = -1.0;
        bool InsideValue = false;
        ui8 ItemsCount = 0;
        EDeserializationStatus Status = EDeserializationStatus::UNKNOWN;

        TMaybe<std::pair<double, double>> PrevValues;

    public:
        TUgramHgramCallback(TZoomJsonDeserializingCallback& callback, THgramModeSelectingCallback* parent)
            : Callback(callback)
            , Skipping(callback, *this)
            , Parent(parent)
        {
        }

        void Reset() {
            Buckets.clear();
            PrevValues.Clear();
            Status = EDeserializationStatus::UNKNOWN;
        }

        bool OnInteger(long long v) final {
            return OnDouble(double(v));
        }

        bool OnUInteger(unsigned long long v) final {
            return OnDouble(double(v));
        }

        bool OnDouble(double v) final;

        bool OnOpenArray() final;

        bool OnCloseArray() final;

        EDeserializationStatus GetStatus() const {
            return Status;
        }

    private:
        void ProcessBucket(const double lowerBound, double weight);
    };

    class TValueHolderCallback;

    class THgramModeSelectingCallback: public TAbstactJsonCallback {
    private:
        TZoomJsonDeserializingCallback& Callback;
        THgramCallback Hgram;
        TUgramHgramCallback Ugram;
        TValueHolderCallback* Parent = nullptr;

    public:
        THgramModeSelectingCallback(TZoomJsonDeserializingCallback& callback, TValueHolderCallback* parent)
            : Callback(callback)
            , Hgram(callback, this)
            , Ugram(callback, this)
            , Parent(parent)
        {
        }

        TValue& GetValue();
        void TryToSetInvalidStatus(EDeserializationStatus status);

        bool OnOpenArray() final;

        bool OnCloseArray() final;

        bool OnInteger(long long v) final {
            return OnDouble(double(v));
        }

        bool OnUInteger(unsigned long long v) final {
            return OnDouble(double(v));
        }

        bool OnDouble(double v) final {
            Callback.SetConsumer(&Hgram);
            Hgram.Reset();
            return Hgram.OnDouble(v);
        }
    };

    class TValueHolderCallback: public TAbstactJsonCallback {
    protected:
        TZoomJsonDeserializingCallback& Callback;
        THgramModeSelectingCallback HgramCallback;
        EDeserializationStatus Status = EDeserializationStatus::UNKNOWN;
        TValue Value;

    public:
        TValueHolderCallback(TZoomJsonDeserializingCallback& callback)
            : Callback(callback)
            , HgramCallback(callback, this)
        {
        }

        TValue& GetValue() {
            return Value;
        }

        void TryToSetInvalidStatus(EDeserializationStatus status) {
            if (Status == EDeserializationStatus::UNKNOWN) {
                Status = status;
            }
        }

        EDeserializationStatus GetStatus() const noexcept {
            return Status;
        }
    };

    class TTagsAgentValueCallback: public TValueHolderCallback {
    private:
        TSkippingJsonCallback Skipping;
        TUnistatValueCallback* Parent = nullptr;
        ui8 ItemCount = 0;
        bool InsideValue = false;
        bool InsideNested = false;
        TString Prefix;

        TString PrefixBuf;
        TString Tags;
        TMaybe<TSignalName> NameMaybe;

    public:
        TTagsAgentValueCallback(TZoomJsonDeserializingCallback& callback, TUnistatValueCallback* parent,
            const TStringBuf prefix
        )
            : TValueHolderCallback(callback)
            , Skipping(callback, *this)
            , Parent(parent)
            , Prefix(prefix)
        {
            PrefixBuf.reserve(SIGNAL_NAME_LIMIT);
            PrefixBuf = prefix;
        }

        bool OnOpenArray() final {
            if (!InsideValue) {
                InsideValue = true;
                return true;
            }
            if (ItemCount == 1) {
                ItemCount = 2;
                InsideNested = true;
                if (NameMaybe.Empty() || !IsHgram(NameMaybe.GetRef())) {
                    MarkInvalidAndSkip();
                } else {
                    Callback.SetConsumer(&HgramCallback);
                }
                return true;
            }
            MarkInvalidAndSkip();
            return true;
        }

        bool OnString(const TStringBuf& s) final {
            return OnStringNoCopy(s);
        }

        bool OnStringNoCopy(const TStringBuf& tagsSignal) final;
        bool OnCloseArray() final;
        bool OnNull() final;
        bool OnDouble(double v) final;

        void Reset() {
            InsideValue = false;
            ItemCount = 0;
            Status = EDeserializationStatus::UNKNOWN;
        }

        bool OnBoolean(bool) final {
            MarkInvalidAndSkip();
            return true;
        }

        bool OnInteger(long long v) final {
            return OnDouble(double(v));
        }

        bool OnUInteger(unsigned long long v) final {
            return OnDouble(double(v));
        }

        bool OnOpenMap() final {
            MarkInvalidAndSkip();
            return true;
        }

        bool OnMapKey(const TStringBuf&) final {
            return false; //WTF, should be skipped in OnOpenMap
        }

        bool OnCloseMap() final {
            return true;
        }

        bool OnEnd() final {
            return true;
        }

    private:
        inline void MarkInvalid() {
            if (!IsErroredStatus(Status)) {
                Status = EDeserializationStatus::INVALID_UNSPECIFIED;
            }
        }

        inline void MarkInvalidAndSkip() {
            MarkInvalid();
            Callback.SetConsumer(&Skipping);
        }

    };

    bool THgramCallback::OnCloseArray() {
        Parent->GetValue() = TValue(std::move(Hgram));
        Parent->TryToSetInvalidStatus(EDeserializationStatus::OK);
        Callback.SetConsumer(Parent);
        return Parent->OnCloseArray();
    }

    bool TUgramHgramCallback::OnCloseArray() {
        if (!InsideValue) {
            if (Status == EDeserializationStatus::UNKNOWN) {
                Status = EDeserializationStatus::OK;
                if (PrevValues.Defined() && PrevValues.GetRef().second > 0.0) {
                    Buckets.push_back(NZoom::NHgram::TUgramBucket::Point(PrevValues.GetRef().first, PrevValues.GetRef().second));
                }
                if (!Buckets.empty()) {
                    Parent->GetValue() = TValue(NZoom::NHgram::THgram::Ugram(std::move(Buckets)));
                } else {
                    Parent->GetValue() = TValue(NZoom::NHgram::THgram::EmptyUgram());
                }
            }
            Parent->TryToSetInvalidStatus(Status);
            Callback.SetConsumer(Parent);
            return Parent->OnCloseArray();
        }
        if (ItemsCount != 2) {
            Status = EDeserializationStatus::INVALID_UNSPECIFIED;
            InsideValue = false;
            Callback.SetConsumer(&Skipping);
            return true;
        }
        InsideValue = false;
        ProcessBucket(First, Second);
        ItemsCount = 0;
        return true;
    }

    void TUgramHgramCallback::ProcessBucket(const double lowerBound, double weight) {
        if (weight < 0.0) {
            weight = 0.0;
        }
        if (!PrevValues.Defined()) {
            PrevValues.ConstructInPlace(lowerBound, weight);
        } else {
            std::pair<double, double>& prev = PrevValues.GetRef();
            if (lowerBound == prev.first) {
                Buckets.push_back(NZoom::NHgram::TUgramBucket::Point(prev.first, prev.second));
            } else if (prev.second >= 0.0) {
                Buckets.push_back(NZoom::NHgram::TUgramBucket(prev.first, lowerBound, prev.second));
            }
            prev.first = lowerBound;
            prev.second = weight;
        }
    }

    bool TUgramHgramCallback::OnDouble(double v) {
        if (!InsideValue || ItemsCount >= 2) {
            Status = EDeserializationStatus::TOO_MUCH_ARGUMENTS_IN_BUCKET;
            Parent->TryToSetInvalidStatus(Status);
            Callback.SetConsumer(&Skipping);
            return true;
        }
        if (ItemsCount == 0) {
            First = v;
            ItemsCount = 1;
        } else if (ItemsCount == 1) {
            Second = v;
            ItemsCount = 2;
        }
        return true;
    }

    bool TUgramHgramCallback::OnOpenArray(){
        if (!InsideValue) {
            InsideValue = true;
            ItemsCount = 0;
            return true;
        }
        Status = EDeserializationStatus::EXTRA_DEPTH;
        Parent->TryToSetInvalidStatus(Status);
        return true;
    }

    bool THgramModeSelectingCallback::OnOpenArray() {
        Callback.SetConsumer(&Ugram);
        Ugram.Reset();
        return Ugram.OnOpenArray();
    }

    TValue& THgramModeSelectingCallback::GetValue() {
        return Parent->GetValue();
    }

    void THgramModeSelectingCallback::TryToSetInvalidStatus(EDeserializationStatus status) {
        Parent->TryToSetInvalidStatus(status);
    }

    bool THgramModeSelectingCallback::OnCloseArray() {
        Callback.SetConsumer(Parent);
        return Parent->OnCloseArray();
    }

    class TCountingSkippingJsonCallback: public TSkippingJsonCallback {
    private:
        ui32 Count = 0;

    public:
        TCountingSkippingJsonCallback(TZoomJsonDeserializingCallback& callback, TAbstactJsonCallback& parent)
            : TSkippingJsonCallback(callback, parent)
        {
        }

        bool OnCloseArray() final {
            if (Depth == 1) {
                ++Count;
            }
            return TSkippingJsonCallback::OnCloseArray();
        }

        void Reset() noexcept {
            Count = 0;
        }

        ui32 GetCount() const noexcept {
            return Count;
        }
    };

    struct TUnistatValueCallback: public TAbstactJsonCallback {
        TUnistatTagsMap TagsToSignals;

        ui32 Limit;

        TZoomJsonDeserializingCallback& Callback;
        TTagsAgentValueCallback ValueDeserializingCallback;
        TSkippingJsonCallback Skipping;
        TCountingSkippingJsonCallback CountingSkipping;

        size_t InvalidCounter = 0;
        size_t Count = 0;
        size_t ParseErrorsCount = 0;

        EDeserializationStatus GlobalStatus = EDeserializationStatus::UNKNOWN;

        TUnistatValueCallback(TZoomJsonDeserializingCallback& callback, ui32 limit, const TString& prefix)
            : Limit(limit)
            , Callback(callback)
            , ValueDeserializingCallback(callback, this, prefix)
            , Skipping(Callback, *this)
            , CountingSkipping(Callback, *this)
        {
        }

        bool OnOpenArray() final {
            Callback.SetConsumer(&ValueDeserializingCallback);
            return true;
        }

        bool OnCloseArray() final {
            return true;
        }

        bool OnEnd() final {
            return true;
        }

        void OnValueReady(const TString& tags, TMaybe<TSignalName>& signalNameMaybe, EDeserializationStatus status,
            TValue&& value)
        {
            if (status == EDeserializationStatus::OK) {
                TSignalName& signalName = signalNameMaybe.GetRef();

                THashMap<TSignalName, TValue>::insert_ctx ctx;

                auto& tagToSignals = TagsToSignals[tags];
                const auto it = tagToSignals.find(signalName, ctx);

                if (it.IsEnd()) {
                    tagToSignals.emplace_direct(ctx, std::move(signalName), std::move(value));
                } else {
                    ++InvalidCounter;
                }
            }
            ++Count;
            ValueDeserializingCallback.Reset();
            if (!IsErroredStatus(GlobalStatus)) {
                GlobalStatus = status;
            }
            if (IsErroredStatus(status)) {
                ++ParseErrorsCount;
            }
            if (Count >= Limit) {
                CountingSkipping.Reset();
                Callback.SetConsumer(&CountingSkipping);
            }
        }

        inline bool OnNull() final {
            ++InvalidCounter;
            return true;
        }

        inline bool OnBoolean(bool) final {
            ++InvalidCounter;
            return true;
        }

        inline bool OnInteger(long long) final {
            ++InvalidCounter;
            return true;
        }

        inline bool OnUInteger(unsigned long long) final {
            ++InvalidCounter;
            return true;
        }

        inline bool OnDouble(double) final {
            ++InvalidCounter;
            return true;
        }

        inline bool OnString(const TStringBuf& v) final {
            return OnStringNoCopy(v);
        }

        inline bool OnOpenMap() final {
            ++InvalidCounter;
            Callback.SetConsumer(&Skipping);
            return true;
        }

        inline bool OnMapKey(const TStringBuf&) final {
            return false; // WTF?
        }

        bool OnCloseMap() final {
            return true;
        }

        bool OnStringNoCopy(const TStringBuf&) final {
            ++InvalidCounter;
            return true;
        }

        bool OnMapKeyNoCopy(const TStringBuf&) final {
            return false; //WTF?
        }

        ui32 GetFilteredCount() const noexcept {
            return CountingSkipping.GetCount();
        }

    };

    bool TTagsAgentValueCallback::OnStringNoCopy(const TStringBuf& tagsSignal) {
        if (Y_UNLIKELY(!InsideValue)) {
            MarkInvalidAndSkip();
            return true;
        }
        if (ItemCount != 0) {
            MarkInvalidAndSkip();
            return true;
        }
        ItemCount = 1;
        TStringBuf tags, signalNameStr;
        tagsSignal.RSplit(SIGNAL_TAGS_DELIMITER, tags, signalNameStr);

        PrefixBuf.replace(Prefix.size(), (PrefixBuf.size() - Prefix.size()), signalNameStr);
        NameMaybe = TSignalName::TryNew(PrefixBuf);
        if (NameMaybe.Empty()) {
            MarkInvalidAndSkip();
            return true;
        }
        if (!NSignal::IsValidSignalName(PrefixBuf)) {
            MarkInvalidAndSkip();
            return true;
        }
        Tags = tags;
        return true;
    }

    bool TTagsAgentValueCallback::OnCloseArray() {
        if (!InsideValue) {
            Callback.SetConsumer(Parent);
            return Parent->OnCloseArray();
        }
        if (InsideNested) {
            InsideNested = false;
            return true;
        }
        InsideValue = false;
        Parent->OnValueReady(Tags, NameMaybe, Status, std::move(Value));
        return true;
    }

    bool TTagsAgentValueCallback::OnNull() {
        if (Y_UNLIKELY(!InsideValue)) {
            ++Parent->InvalidCounter;
            return true;
        }
        if (ItemCount != 1) {
            MarkInvalidAndSkip();
            return true;
        }
        ItemCount = 2;
        Value = TValue();
        Status = EDeserializationStatus::NONE;
        return true;
    }

    bool TTagsAgentValueCallback::OnDouble(double v) {
        if (Y_UNLIKELY(!InsideValue)) {
            ++Parent->InvalidCounter;
            return true;
        }
        if (ItemCount != 1) {
            MarkInvalidAndSkip();
            return true;
        }
        ItemCount = 2;
        Value = TValue(v);
        Status = EDeserializationStatus::OK;
        return true;
    }

    inline bool IsDiffUnistat(const TSignalName& name) noexcept {
        const auto* aggRules = name.GetAggregationRules();
        return aggRules && aggRules->IsDiffAggregation();
    }

    inline void IncStats(const TSignalName& name, TUnistatStats& stats) noexcept {
        if (IsHgram(name)) {
            ++stats.NumHgrams;
        } else {
            ++stats.NumSignals;
        }
    }

    TUgramBuckets::const_iterator FindFirstNonEmptyBucket(TUgramBuckets::const_iterator first, TUgramBuckets::const_iterator last) {
        return std::find_if(first, last, [](const auto& bucket) {
            return bucket.Weight > 0.0;
        });
    }

    struct TExtractedValue : public IUpdatable, public IHgramStorageCallback {
        TMaybe<double> FloatMaybe;
        const TUgramBuckets* BucketsPtr = nullptr;

        void MulNone() final {
        }

        void MulFloat(const double value) final {
            FloatMaybe = value;
        }

        void MulVec(const TVector<double>& /*value*/) final {
        }

        void MulCountedSum(const double /*sum*/, const ui64 /*count*/) final {
        }

        void MulHgram(const NZoom::NHgram::THgram& value) final {
            value.Store(*this);
        }

        void OnStoreSmall(const TVector<double>& /*values*/, const size_t /*zeros*/) final {
        }

        void OnStoreNormal(const TVector<double>& /*values*/, const size_t /*zeros*/,
                           const i16 /*startPower*/) final
        {
        }

        void OnStoreUgram(const TUgramBuckets& buckets) final {
            BucketsPtr = &buckets;
        }
    };

    class TDiffBuilder {
    public:
        /*
         * Calculates difference in buckets. Allows new buckets to appear.
         * Returns false if some of the buckets:
         *   - were removed
         *   - changed their bounds
         *   - decreased their weight
         */
        static bool CalculateDiffBuckets(const TUgramBuckets& newBuckets, const TUgramBuckets& oldBuckets,
                                         TVector<TUgramBucket>& result)
        {
            // iterate over old buckets to find in new buckets, skipping empty ones
            auto oldBucketIt = FindFirstNonEmptyBucket(oldBuckets.begin(), oldBuckets.end());
            for (const auto& newBucket: newBuckets) {
                if (oldBucketIt != oldBuckets.end()) {
                    if (HaveSameBounds(newBucket, *oldBucketIt)) {
                        const double bucketDiff = newBucket.Weight - oldBucketIt->Weight;
                        if (bucketDiff >= 0.0) {
                            result.push_back(TUgramBucket(newBucket.LowerBound, newBucket.UpperBound, bucketDiff));
                        } else if (bucketDiff < 0.0) {
                            return false; // negative diff
                        }
                        oldBucketIt = FindFirstNonEmptyBucket(++oldBucketIt, oldBuckets.end());
                    } else {
                        if (newBucket.LowerBound <= oldBucketIt->LowerBound && newBucket.UpperBound <= oldBucketIt->LowerBound) {
                            // new bucket is not in old buckets
                            result.push_back(newBucket);
                        } else {
                            return false; // we did not find current old bucket
                        }
                    }
                } else { // we have found all the old buckets, just add all non-empty new ones
                    result.push_back(newBucket);
                }
            }
            return oldBucketIt == oldBuckets.end(); // true if all old buckets were found
        }

        static TMaybe<TValue> GetDiff(const TValue& currentAbs, TUnistatDiffValue&& prevValue) {
            TExtractedValue current;
            TExtractedValue previous;
            currentAbs.Update(current);
            prevValue.Abs.Update(previous);

            if (current.FloatMaybe.Defined() && previous.FloatMaybe.Defined()) {
                const double currentDouble = current.FloatMaybe.GetRef();
                const double previousDouble = previous.FloatMaybe.GetRef();
                if (previousDouble <= currentDouble) {
                    return TValue(currentDouble - previousDouble);
                }
                return std::move(prevValue.DiffValue);
            }
            if (current.BucketsPtr != nullptr && previous.BucketsPtr != nullptr) {
                const TUgramBuckets& currBuckets = *current.BucketsPtr;
                const TUgramBuckets& prevBuckets = *previous.BucketsPtr;

                TVector<TUgramBucket> result(Reserve(currBuckets.size()));
                bool correctDiff = CalculateDiffBuckets(currBuckets, prevBuckets, result);

                if (correctDiff) {
                    return TValue((!result.empty()) ? THgram::UgramNoCheck(std::move(result)) : THgram::EmptyUgram());
                } else {
                    return std::move(prevValue.DiffValue);
                }
            }
            return std::move(prevValue.DiffValue);
        }
    };

    void ProcessAllDiffs(TUnistatTagsMap& tagsToSignals, TMaybe<TUnistatState>& prevState, TUnistatStats& stats) {
        TUnistatTaggedDiffValues res;

        // filter absolute values, compute diff values and save previous hashmap
        for (auto it = tagsToSignals.begin(); it != tagsToSignals.end();) {
            auto prev = it;
            ++it;

            auto& tagToSignals = *prev;
            const TString& tags = tagToSignals.first;

            TUnistatDiffCounters* prevTagValuePtr = nullptr;

            if (prevState.Defined()) {
                auto prevIt = prevState.GetRef().TaggedDiffValues.find(tags);
                if (prevIt != prevState.GetRef().TaggedDiffValues.end()) {
                    prevTagValuePtr = &prevIt->second;
                }
            }

            auto& signals = tagToSignals.second;

            THashMap<TSignalName, TValue> filteredValues;
            THashMap<TSignalName, TUnistatDiffValue> diffValues;

            for (auto& [name, value]: signals) {
                if (IsDiffUnistat(name)) {
                    TMaybe<TValue> diffValueMaybe;

                    if (prevTagValuePtr) {
                        auto valueIt = prevTagValuePtr->find(name);
                        if (!valueIt.IsEnd()) {
                            diffValueMaybe = TDiffBuilder::GetDiff(value, std::move(valueIt->second));
                            if (diffValueMaybe.Defined()) {
                                IncStats(name, stats);
                                // TValue copying - heavy op
                                filteredValues.emplace(name, diffValueMaybe.GetRef().GetValue());
                            }
                        }
                    }
                    diffValues.emplace(name,
                       TUnistatDiffValue{.Abs = std::move(value), .DiffValue = std::move(diffValueMaybe)});
                } else {
                    IncStats(name, stats);
                    filteredValues.emplace(name, std::move(value));
                }
            }
            if (!diffValues.empty()) {
                res[tags] = std::move(diffValues);
            }

            if (!filteredValues.empty()) {
                signals = std::move(filteredValues);
            } else {
                tagsToSignals.erase(prev); //Destroys tags ref
            }
        }
        if (!prevState.Defined()) {
            prevState.ConstructInPlace();
        }
        prevState.GetRef().TaggedDiffValues = std::move(res);
    }

    void CollectUgramBucketStats(const TUnistatTagsMap& tagsToSignals, TMaybe<TUnistatState>& state, TUnistatStats& stats) {
        TUnistatTaggedUgramBuckets nextBucketState;

        for (auto it = tagsToSignals.begin(); it != tagsToSignals.end(); ++it) {
            const TString& tags = it->first;
            TUnistatUgramBucketBorders* prevIterationTagUgramBuckets = nullptr;
            TUnistatUgramBucketBorders curIterationTagUgramBuckets;
            if (state.Defined()) {
                auto bucketsIt = state->TaggedPrevUgramBuckets.find(tags);
                if (bucketsIt != state->TaggedPrevUgramBuckets.end()) {
                    prevIterationTagUgramBuckets = &bucketsIt->second;
                }
            }

            for (auto& [name, value]: it->second) {
                TExtractedValue currentAbsoluteValue;
                value.Update(currentAbsoluteValue);
                if (currentAbsoluteValue.BucketsPtr) {
                    TVector<double> curBucketBorders;
                    curBucketBorders.reserve(currentAbsoluteValue.BucketsPtr->size() * 2);
                    for (auto& bucket: *currentAbsoluteValue.BucketsPtr) {
                        if (curBucketBorders.empty() || curBucketBorders.back() != bucket.LowerBound) {
                            curBucketBorders.push_back(bucket.LowerBound);
                            curBucketBorders.push_back(bucket.UpperBound);
                        } else {
                            curBucketBorders.push_back(bucket.UpperBound);
                        }
                    }

                    stats.AddUgramBucketBorderCount(curBucketBorders.size());

                    if (prevIterationTagUgramBuckets) {
                        auto prevBordersIt = prevIterationTagUgramBuckets->find(name);
                        if (prevBordersIt != prevIterationTagUgramBuckets->end() && prevBordersIt->second != curBucketBorders) {
                            stats.NumUgramsChangedBuckets++;
                        }
                    }

                    curIterationTagUgramBuckets.emplace(name, std::move(curBucketBorders));
                }
            }
            if (!curIterationTagUgramBuckets.empty()) {
                nextBucketState.emplace(tags, std::move(curIterationTagUgramBuckets));
            }
        }

        if (!state.Defined()) {
            state.ConstructInPlace();
        }
        state->TaggedPrevUgramBuckets = std::move(nextBucketState);
    }
}

namespace NZoom {
    namespace NPythonTest {
        using namespace NZoom::NPrivate;

        //Testing hooks
        TMaybe<TValue> GetUnistatValuesDiff(const TValue& currentAbs, TUnistatDiffValue&& prevValue) {
            return TDiffBuilder::GetDiff(currentAbs, std::move(prevValue));
        }

        class TTestCallback: public TValueHolderCallback {
        public:
            TTestCallback(TZoomJsonDeserializingCallback& callback)
                : TValueHolderCallback(callback)
            {
            }

            bool OnOpenArray() final {
                Callback.SetConsumer(&HgramCallback);
                return true;
            }

            bool OnCloseArray() final {
                return true;
            }

            bool OnEnd() final {
                return true;
            }
        };

        bool DeserializeUnistatHgramAgentValue(const TStringBuf jsonStr, TValue& dst) {
            TZoomJsonDeserializingCallback cb(nullptr);
            TTestCallback testCallback(cb);
            cb.SetConsumer(&testCallback);
            NJson::ReadJsonFast(jsonStr, &cb);
            dst = std::move(testCallback.GetValue());
            return testCallback.GetStatus() != EDeserializationStatus::INVALID_UNSPECIFIED;
        }
    }
}

TUgramBuckets TUnistatStats::InitializeUgramBucketCountUgram() {
    return NZoom::NHgram::TUgramBuckets({
        {0, 20, 0},
        {20, 30, 0},
        {30, 35, 0},
        {35, 40, 0},
        {40, 45, 0},
        {45, 51, 0}, // e.g. 49 consecutive buckets -> 50 borders -> fall into this bucket
        {51, 52, 0}, // e.g. 50 consecutive buckets -> 51 borders -> fall into this bucket
        {52, 55, 0}, // e.g. 51 consecutive buckets -> 52 borders -> fall into this bucket
        {55, 60, 0},
        {60, 65, 0},
        {65, 70, 0},
        {70, 80, 0},
        {80, 90, 0},
        {90, 100, 0},
        {100, 150, 0},
        {150, 200, 0},
        {200, 500, 0},
        {500, 1000, 0},
        {1000, 10000, 0},
        {10000, 100000000, 0} // everything that is too large goes in here too
    });
}

void TUnistatStats::AddUgramBucketBorderCount(size_t borderCount) {
    auto it = LowerBound(
        UgramSizes.begin(),
        UgramSizes.end(),
        borderCount,
        [](const TUgramBucket& bucket, size_t value) {
            if (bucket.UpperBound == value && bucket.LowerBound == value) {
                return false;
            } else {
                return bucket.UpperBound <= value;
            }
        }
    );
    if (it == UgramSizes.end()) {
        UgramSizes.back().Weight++; // add everything that is too large here
    } else {
        it->Weight++;
    }
}

TValue* TUnistatStats::GetAndResetUgramSizes() {
    auto ugramSizes = InitializeUgramBucketCountUgram();
    ugramSizes.swap(UgramSizes);
    return new TValue(NHgram::THgram::UgramNoCheck(std::move(ugramSizes)));
}

TUnistatDeserializer::TUnistatDeserializer(const TStringBuf prefix, const size_t limit)
    : Prefix(TString::Join(prefix, NPrivate::SIGNAL_MODULE_DELIMITER))
    , Limit(limit)
{
}

TUnistatValuesIterator TUnistatDeserializer::Loads(const TStringBuf jsonStr) {
    TUnistatStats stats;
    stats.ResponseBytes += jsonStr.size();
    TZoomJsonDeserializingCallback cb(nullptr);
    NPrivate::TUnistatValueCallback unistatCallback(cb, Limit, Prefix);
    cb.SetConsumer(&unistatCallback);
    NJson::ReadJsonFast(jsonStr, &cb);

    NPrivate::CollectUgramBucketStats(unistatCallback.TagsToSignals, PrevState, stats);
    NPrivate::ProcessAllDiffs(unistatCallback.TagsToSignals, PrevState, stats);

    stats.FilteredSignals = unistatCallback.GetFilteredCount();
    stats.InvalidSignals = unistatCallback.InvalidCounter;
    const double total = unistatCallback.Count + unistatCallback.InvalidCounter + stats.FilteredSignals;
    if (Limit != 0) {
        stats.LimitUsage = total * 100 / Limit;
    }
    stats.Status = unistatCallback.GlobalStatus;
    stats.ParseErrors = unistatCallback.ParseErrorsCount;
    return TUnistatValuesIterator(std::move(unistatCallback.TagsToSignals), stats);
}

void TUnistatDeserializer::Loads(const TStringBuf jsonStr, TUnistatValuesIterator& it) {
    it = Loads(jsonStr);
}

TUnistatValuesIterator::TUnistatValuesIterator()
    : Position(Data.end())
{
}

TUnistatValuesIterator::TUnistatValuesIterator(TUnistatTagsMap&& data,
    const TUnistatStats& stats
)
    : Data(std::move(data))
    , Stats(stats)
    , Position(Data.begin())
{
}

bool TUnistatValuesIterator::IsValid() const noexcept {
    return !Position.IsEnd();
}

std::pair<TString, NZoom::NRecord::TRecord*> TUnistatValuesIterator::GetAndMove() {
    TString tag = Position->first;
    auto& signals = Position->second;

    TSeries tagValues;
    tagValues.reserve(signals.size());

    for (auto& [signal, value]: signals) {
        tagValues.emplace_back(signal, std::move(value));
    }

    THolder<TRecord> recordHolder = MakeHolder<TRecord>(std::move(tagValues));
    auto prev = Position;
    ++Position;
    Data.erase(prev);
    return std::make_pair(tag, recordHolder.Release());
}

const TUnistatStats& TUnistatValuesIterator::GetStats() const noexcept {
    return Stats;
}
