#include "unistat.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/env.h>

#include <util/folder/pathsplit.h>
#include <util/stream/file.h>


using namespace NZoom::NHgram;
using namespace NZoom::NPython;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;

namespace NZoom::NPythonTest {
    TMaybe<TValue> GetUnistatValuesDiff(const TValue& currentAbs, TUnistatDiffValue&& prevValue);
    bool DeserializeUnistatHgramAgentValue(const TStringBuf jsonStr, TValue& dst);
}

namespace {
    struct TUgramBucketsExtractor final: public NZoom::NValue::IUpdatable, public NZoom::NHgram::IHgramStorageCallback {

        const NZoom::NHgram::TUgramBuckets* BucketsPtr = nullptr;

        void MulNone() override {
        }

        void MulFloat(const double) override {
        }

        void MulVec(const TVector<double>&) override {
        }

        void MulCountedSum(const double, const ui64) override {
        }

        void MulHgram(const NZoom::NHgram::THgram& value) override {
            value.Store(*this);
        }

        void OnStoreSmall(const TVector<double>&, const size_t) override {
        }

        void OnStoreNormal(const TVector<double>&, const size_t, const i16) override {
        }

        void OnStoreUgram(const NZoom::NHgram::TUgramBuckets& buckets) final {
            BucketsPtr = &buckets;
        }
    };
}

using namespace NZoom::NPythonTest;

Y_UNIT_TEST_SUITE(TZoomTUnistatParserTest) {

    using TNamedVals = std::initializer_list<std::pair<TString, TValue>>;

    TUnistatStats RunTest(TUnistatValuesIterator&& it,
        std::initializer_list<std::pair<TStringBuf, TNamedVals>> target)
    {
        THashMap<TString, THolder<TRecord>> val;
        while(it.IsValid()) {
            val.emplace(it.GetAndMove());
        }

        THashMap<TString, THolder<TRecord>> targetRecords;
        for(const auto& kvPair: target) {
            TVector<std::pair<NZoom::NSignal::TSignalName, TValue>> values;
            values.reserve(kvPair.second.size());
            for (const auto& item: kvPair.second) {
                values.emplace_back(item.first, item.second.GetValue());
            }
            targetRecords.emplace(kvPair.first, new TRecord(std::move(values)));
        }

        UNIT_ASSERT_VALUES_EQUAL(val.size(), targetRecords.size());
        for (auto it = val.begin(); it != val.end(); ++it) {
            const auto targetIt = targetRecords.find(it->first);
            UNIT_ASSERT(!targetIt.IsEnd());
            UNIT_ASSERT_EQUAL(*(it->second), *(targetIt->second));
        }
        return it.GetStats();
    }

    Y_UNIT_TEST(DeserializationAbsSingleFloatWithoutTags) {
        TUnistatDeserializer deserializer("module", 3);
        RunTest(deserializer.Loads("[[\"foo_ammm\", 2.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-foo_ammm"), TValue(2.0))})}
        );

        RunTest(deserializer.Loads("[[\"foo_ammm\", 1.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-foo_ammm"), TValue(1.0))})}
        );
    }

    Y_UNIT_TEST(DeserializationAbsSingleFloatWithTags) {
        TUnistatDeserializer deserializer("module", 3);
        RunTest(deserializer.Loads("[[\"ctype=prod;foo_ammm\", 1.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf("ctype=prod"),
                {std::make_pair(TString("module-foo_ammm"), TValue(1.0))})}
        );

        RunTest(deserializer.Loads("[[\"ctype=prod;foo_ammm\", 2.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf("ctype=prod"),
                {std::make_pair(TString("module-foo_ammm"), TValue(2.0))})}
        );
    }

    Y_UNIT_TEST(DeserializationOverrideItype) {
        TUnistatDeserializer deserializer("unistat", 3);
        RunTest(deserializer.Loads("[[\"itype=example;foo_ammm\", 1.0]]"),
                {std::make_pair<TStringBuf, TNamedVals>(TStringBuf("itype=example"),
                                                        {std::make_pair(TString("unistat-foo_ammm"), TValue(1.0))})}
        );

        RunTest(deserializer.Loads("[[\"itype=example;foo_ammm\", 2.0]]"),
                {std::make_pair<TStringBuf, TNamedVals>(TStringBuf("itype=example"),
                                                        {std::make_pair(TString("unistat-foo_ammm"), TValue(2.0))})}
        );
    }

    Y_UNIT_TEST(DeserializationDiffSingleFloat) {
        TUnistatDeserializer deserializer("module", 3);

        RunTest(deserializer.Loads("[[\"foo_dmmm\", 1.0]]"), {});

        RunTest(deserializer.Loads("[[\"foo_dmmm\", 2.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-foo_dmmm"), TValue(1.0))})}
        );

        RunTest(deserializer.Loads("[[\"foo_dmmm\", 1.5]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-foo_dmmm"), TValue(1.0))})}
        );

        RunTest(deserializer.Loads("[[\"foo_dmmm\", 2.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-foo_dmmm"), TValue(0.5))})}
        );
    }

    Y_UNIT_TEST(DeserializationDiffManyFloatWithTags) {
        TUnistatDeserializer deserializer("module", 3);
        RunTest(
            deserializer.Loads("[[\"ctype=prod;foo_dmmm\", 1.0], [\"ctype=prod;foo_ammm\", 5.0],"
                    "[\"ctype=preprod;foo_dmmm\", 10.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf("ctype=prod"),
                {std::make_pair(TString("module-foo_ammm"), TValue(5.0))})}
        );
        RunTest(
            deserializer.Loads("[[\"ctype=prod;foo_dmmm\", 2.0], [\"ctype=prod;foo_ammm\", 5.0],"
                    "[\"ctype=preprod;foo_dmmm\", 20.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf("ctype=prod"),
                {std::make_pair(TString("module-foo_ammm"), TValue(5.0)),
                    std::make_pair(TString("module-foo_dmmm"), TValue(1.0)),
                }),
             std::make_pair<TStringBuf, TNamedVals>(TStringBuf("ctype=preprod"),
                 {std::make_pair(TString("module-foo_dmmm"), TValue(10.0))})
            });
        RunTest(
            deserializer.Loads("[[\"ctype=prod;foo_dmmm\", 5.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf("ctype=prod"),
                {std::make_pair(TString("module-foo_dmmm"), TValue(3.0))})}
        );
    }


    Y_UNIT_TEST(DeserializationCorrectHistograms) {
        TUnistatDeserializer deserializer("module", 3);

        RunTest(deserializer.Loads("[[\"foo_dhhh\", [[1, 1], [2, 1]]], [\"foo_ahhh\", 5.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-foo_ahhh"), TValue(5.0))})}
        );


        RunTest(deserializer.Loads("[[\"foo_dhhh\", [[2, 2], [3, 1]]], [\"foo_ahhh\", 3.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-foo_ahhh"), TValue(3.0))})}
        );

        RunTest(deserializer.Loads("[[\"foo_dhhh\", [[2, 12], [3, 1]]], [\"foo_ahhh\", 2.0]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-foo_ahhh"), TValue(2.0)),
                 std::make_pair(TString("module-foo_dhhh"),
                     TValue(THgram::Ugram(TUgramBuckets({
                         TUgramBucket(2.0, 3.0, 10.0),
                         TUgramBucket(3.0, 3.0, 0.0)
                     }))))
                })}
        );
    }


    Y_UNIT_TEST(RealResponse) {
        TUnistatDeserializer deserializer("module", 3);
        // NOTE(rocco66): skip null, skip diff
        const TStringBuf testData = "[[\"WORKER-ALL-RULES-TIME_dhhh\",null],[\"WORKER-TOTAL-REQUESTS_dmmm\",0],"
            "[\"WORKER-FRESH-AGE-MINS_axxx\",4411]]"sv;
        RunTest(deserializer.Loads(testData),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-WORKER-FRESH-AGE-MINS_axxx"), TValue(4411.0))})}
        );

        // NOTE(rocco66): same values, all diff is zeros now
        RunTest(deserializer.Loads(testData),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-WORKER-FRESH-AGE-MINS_axxx"), TValue(4411.0)),
                 std::make_pair(TString("module-WORKER-TOTAL-REQUESTS_dmmm"), TValue(0.0)),
                })}
        );
    }

    Y_UNIT_TEST(ManyTags) {
        TUnistatDeserializer deserializer("module", 3);
        RunTest(deserializer.Loads("[[\"x=1;y=2;a_ammm\",1]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf("x=1;y=2"),
                {std::make_pair(TString("module-a_ammm"), TValue(1.0))})}
        );
    }

    Y_UNIT_TEST(SkipIncorrectSignal) {
        TUnistatDeserializer deserializer("module", 3);
        RunTest(deserializer.Loads("[[\"a_mm\",1], [\"b_ammm\", 2]]"),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-b_ammm"), TValue(2.0))})}
        );
    }

    Y_UNIT_TEST(LimitUnistat) {
        TUnistatDeserializer deserializer("module", 3);
        {
            TStringBuf testData = TStringBuf("[[\"a_ammm\",1],[\"b_ammm\",2],[\"c_ammm\",3],[\"d_ammm\",4]]");
            TUnistatStats stats = RunTest(deserializer.Loads(testData),
                {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                    {std::make_pair(TString("module-a_ammm"), TValue(1.0)),
                     std::make_pair(TString("module-b_ammm"), TValue(2.0)),
                     std::make_pair(TString("module-c_ammm"), TValue(3.0))
                    })}
            );
            UNIT_ASSERT_VALUES_EQUAL(stats.FilteredSignals, 1);
        }
        {
            TStringBuf testData = TStringBuf("[[\"a_ammm\",1],[\"b_ammm\",2],[\"c_ammm\",3],[\"d_ammm\",4],[\"e_ammm\",5]]");
            TUnistatStats stats = RunTest(deserializer.Loads(testData),
                {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                    {std::make_pair(TString("module-a_ammm"), TValue(1.0)),
                     std::make_pair(TString("module-b_ammm"), TValue(2.0)),
                     std::make_pair(TString("module-c_ammm"), TValue(3.0))
                    })}
            );
            UNIT_ASSERT_VALUES_EQUAL(stats.FilteredSignals, 2);
        }
    }

    Y_UNIT_TEST(BadValues) {
        TUnistatDeserializer deserializer("module", 100);
        TStringBuf testData = "[[\"signal1_ammm\", null],[\"signal2_ammm\", \"foo\"],"
            "[\"signal3_ammm\", []], [\"signal3_ahhh\", [[1, 2], 0, []]], [\"signal5_ahhh\", \"foo\"],"
            "[\"SOURCES-images-like-IMAGESCBIR_SIMILAR_MMETA-Error-HTTP-1.1-503-too-many-requests-in-progressContent-Length--0--_dmmm\", 15],"
            "[\"signal_ammm\", 15]]"sv;
        RunTest(deserializer.Loads(testData),
            {std::make_pair<TStringBuf, TNamedVals>(TStringBuf(""),
                {std::make_pair(TString("module-signal_ammm"), TValue(15.0))
                })}
        );
    }

    Y_UNIT_TEST(LargeLimit) {
        TUnistatDeserializer deserializer("module", 1000);
        TString testData = TFileInput(
            JoinPaths(
                TStringBuf(ArcadiaSourceRoot()),
                TStringBuf("infra/yasm/zoom/python/unistat/ut/GOLOVANSUPPORT-428.json"))
        ).ReadAll();
        deserializer.Loads(testData);
    }

    void DoTestFloatDiff(const double prev, const double currentAbs, const TMaybe<double>& targetDiff) {
        TUnistatDiffValue diffValue{.Abs = TValue(prev), .DiffValue = Nothing()};
        TValue curr(currentAbs);
        TMaybe<TValue> diff = GetUnistatValuesDiff(curr, std::move(diffValue));
        TMaybe<TValue> target = targetDiff.Defined() ? TValue(targetDiff.GetRef()) : TMaybe<TValue>();
        UNIT_ASSERT_VALUES_EQUAL(diff, target);
    }

    Y_UNIT_TEST(FloatUnistatDiff) {
        DoTestFloatDiff(5.0, 6.0, 1.0);
        DoTestFloatDiff(5.0, 6.5, 1.5);
        DoTestFloatDiff(6.0, 5.0, Nothing());
        DoTestFloatDiff(5.0, 5.0, 0.0);
    }

    TValue ToHgram(TStringBuf val) {
        TValue value;
        UNIT_ASSERT(DeserializeUnistatHgramAgentValue(val, value));
        return value;
    }

    void DoTestHgramDiff(const TStringBuf currentAbs, const TStringBuf prev, const TMaybe<TStringBuf>& targetDiff) {
        TValue currentValue = ToHgram(currentAbs);
        TMaybe<TValue> targetValue = targetDiff.Defined() ? ToHgram(targetDiff.GetRef()) : TMaybe<TValue>();

        TUnistatDiffValue diffValue{.Abs = ToHgram(prev), .DiffValue = Nothing()};

        TMaybe<TValue> diff = GetUnistatValuesDiff(currentValue, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(diff, targetValue);
    }

    Y_UNIT_TEST(HgramUnistatDiff) {
        DoTestHgramDiff("[[1, 10]]", "[[1, 5]]", TStringBuf("[[1, 5], [1, 0]]"));
        DoTestHgramDiff("[[1, 10], [2, 0]]", "[[1, 5], [2, 0]]", TStringBuf("[[1, 5], [2, 0]]"));
        DoTestHgramDiff("[[1, 10], [2, 100]]", "[[1, 5], [2, 10]]", TStringBuf("[[1, 5], [2, 90]]"));

        DoTestHgramDiff("[]", "[[3, 1]]", Nothing());
        DoTestHgramDiff("[[1, 10]]", "[]", Nothing());
        DoTestHgramDiff("[[1, 10]]", "[[3, 1]]", Nothing());
        DoTestHgramDiff("[[1, 10], [2, 100]]", "[[1, 5], [2, 10], [3, 1]]", Nothing());
    }

    TUnistatDiffValue UpdateHgramDiffValue(const TStringBuf currentAbs, TUnistatDiffValue&& prevValue) {
        TValue currentValue = ToHgram(currentAbs);
        TMaybe<TValue> diff = GetUnistatValuesDiff(currentValue, std::move(prevValue));

        return TUnistatDiffValue{
            .Abs = std::move(currentValue),
            .DiffValue = std::move(diff)
        };
    }

    size_t GetHgramCount(const TValue& value) {
        size_t result = 0;
        if (value.GetType() == EValueType::USER_HGRAM) {
            TUgramBucketsExtractor extractor;
            value.Update(extractor);
            if (extractor.BucketsPtr) {
                for (const auto& bucket: *extractor.BucketsPtr) {
                    result += static_cast<size_t>(bucket.Weight);
                }
            }
        }
        return result;
    }

    size_t GetHgramCount(const TMaybe<TValue>& value) {
        size_t result = 0;
        if (value.Defined()) {
            return GetHgramCount(value.GetRef());
        }
        return result;
    }

    Y_UNIT_TEST(HgramDiffForBucketVariations) {
        TStringBuf data = "[[0,0],[1,0],[2,0],[3,0],[4,0],[5,0],[6,0]]";
        TUnistatDiffValue diffValue = {.Abs = ToHgram(data), .DiffValue = TMaybe<NZoom::NValue::TValue>()};
        UNIT_ASSERT_VALUES_EQUAL(0, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(data), diffValue.Abs);

        data = TStringBuf("[[0,1],[1,0],[2,0],[3,0],[4,0],[5,1],[6,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(data), diffValue.Abs);

        data = TStringBuf("[[0,2],[1,0],[2,3],[3,0],[4,0],[5,1],[6,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(4, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(6, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[0,1],[1,0],[2,3],[3,0],[4,0],[5,0],[6,0]]")), diffValue.DiffValue.GetRef());

        data = TStringBuf("[[0,2],[1,0],[2,3],[3,0],[4,0],[5,1],[6,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(0, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(6, GetHgramCount(diffValue.Abs));

        data = TStringBuf("[[0,2],[1,1],[2,3],[3,0],[4,0],[5,1],[6,1],[7,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(8, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[0,0],[1,1],[2,0],[3,0],[4,0],[5,0],[6,1],[7,0]]")), diffValue.DiffValue.GetRef());

        data = TStringBuf("[[0,2],[1,1],[2,3],[3,0],[4,0],[5,1],[6,2],[7,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(9, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[0,0],[1,0],[2,0],[3,0],[4,0],[5,0],[6,1],[7,0]]")), diffValue.DiffValue.GetRef());

        data = TStringBuf("[[0,2],[1,1],[2,3],[3,0],[4,0],[5,1],[6,2],[7,0],[8,0],[9,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(0, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(9, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[0,0],[1,0],[2,0],[3,0],[4,0],[5,0],[6,0],[7,0],[8,0],[9,0]]")), diffValue.DiffValue.GetRef());
    }

    Y_UNIT_TEST(HgramDiffNewBuckets) {
        TStringBuf data = "[[2,1],[3,0],[5,1],[6,0]]";
        TUnistatDiffValue diffValue = {.Abs = ToHgram(data), .DiffValue = TMaybe<NZoom::NValue::TValue>()};
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(data), diffValue.Abs);

        data = TStringBuf("[[0,1],[1,0],[2,2],[3,1],[4,0],[5,1],[6,0],[7,1],[8,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(4, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(6, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[0,1],[1,0],[2,1],[3,1],[4,0],[5,0],[6,0],[7,1],[8,0]]")), diffValue.DiffValue);

        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(0, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(6, GetHgramCount(diffValue.Abs));

        // remove bucket
        data = TStringBuf("[[0,1],[1,0],[2,2],[3,1],[4,0],[5,1],[6,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(0, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(5, GetHgramCount(diffValue.Abs));

        data = TStringBuf("[[0,1],[1,0],[2,2],[3,1],[4,0],[5,1],[6,0],[7,1],[8,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(6, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[0,0],[1,0],[2,0],[3,0],[4,0],[5,0],[6,0],[7,1],[8,0]]")), diffValue.DiffValue);

        // decrease bucket weight
        data = TStringBuf("[[0,1],[1,0],[2,1],[3,1],[4,0],[5,1],[6,0],[7,1],[8,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(5, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[0,0],[1,0],[2,0],[3,0],[4,0],[5,0],[6,0],[7,1],[8,0]]")), diffValue.DiffValue);
    }

    Y_UNIT_TEST(HgramDiffNewIncorrectBuckets) {
        TStringBuf data = "[[2,1],[3,0],[5,1],[6,0]]";
        TUnistatDiffValue diffValue = {.Abs = ToHgram(data), .DiffValue = TMaybe<NZoom::NValue::TValue>()};
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(data), diffValue.Abs);

        data = TStringBuf("[[2,1],[3,0],[5,2],[6,0]]");
        TStringBuf correctDiff = "[[2,0],[3,0],[5,1],[6,0]]";

        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(3, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(correctDiff), diffValue.DiffValue);
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(data), diffValue.Abs);

        data = TStringBuf("[[2,1],[4,1],[6,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(correctDiff), diffValue.DiffValue);
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(data), diffValue.Abs);

        data = TStringBuf("[[1,1],[3,1],[6,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(correctDiff), diffValue.DiffValue);
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(data), diffValue.Abs);

        data = TStringBuf("[[5,1],[10,1],[12,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(correctDiff), diffValue.DiffValue);
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(data), diffValue.Abs);
    }

    Y_UNIT_TEST(HgramDiffNewSinglePointBuckets) {
        TStringBuf data = "[[2,1],[3,0]]";
        TUnistatDiffValue diffValue = {.Abs = ToHgram(data), .DiffValue = TMaybe<NZoom::NValue::TValue>()};
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(data), diffValue.Abs);

        data = TStringBuf("[[2,1],[2,2],[3,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(3, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[2,1],[2,1],[3,0]]")), diffValue.DiffValue);

        data = TStringBuf("[[2,1],[2,2],[3,1],[3,1],[4,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(5, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[2,0],[2,0],[3,1],[3,1],[4,0]]")), diffValue.DiffValue);

        data = TStringBuf("[[2,1],[2,2],[3,1],[3,1],[4,0],[6,7]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(7, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(12, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[2,0],[2,0],[3,0],[3,0],[4,0],[6,7]]")), diffValue.DiffValue);
    }

    Y_UNIT_TEST(HgramDiffSinglePointToBucket) {
        TStringBuf data = "[[4,0]]";
        TUnistatDiffValue diffValue = {.Abs = ToHgram(data), .DiffValue = TMaybe<NZoom::NValue::TValue>()};
        UNIT_ASSERT_VALUES_EQUAL(0, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(data), diffValue.Abs);

        data = TStringBuf("[[4,1]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[4,1]]")), diffValue.DiffValue);

        data = TStringBuf("[[4,2]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(1, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[4,1]]")), diffValue.DiffValue);

        data = TStringBuf("[[4,5]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(3, GetHgramCount(diffValue.DiffValue));
        UNIT_ASSERT_VALUES_EQUAL(5, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[4,3]]")), diffValue.DiffValue);

        data = TStringBuf("[[4,2],[6,0]]");
        diffValue = UpdateHgramDiffValue(data, std::move(diffValue));
        UNIT_ASSERT_VALUES_EQUAL(3, GetHgramCount(diffValue.DiffValue)); // diff bigger than abs because we keep the previous diff
        UNIT_ASSERT_VALUES_EQUAL(2, GetHgramCount(diffValue.Abs));
        UNIT_ASSERT_VALUES_EQUAL(ToHgram(TStringBuf("[[4,3]]")), diffValue.DiffValue);
    }

    Y_UNIT_TEST(JsonValidatationTest) {
        TUnistatDeserializer deserializer("module", 100);
        TStringBuf testData;

        testData = TStringBuf("[[\"signal_ahhh\", [1,2,3,4,5,6]]]");
        UNIT_ASSERT_EQUAL(deserializer.Loads(testData).GetStats().Status, EDeserializationStatus::OK);
        testData = TStringBuf("[[\"signal_dhhh\", [[1,2],[2,3]]]]");
        UNIT_ASSERT_EQUAL(deserializer.Loads(testData).GetStats().Status, EDeserializationStatus::OK);
        testData = TStringBuf("[[\"signal_dhhh\", 10.0]]");
        UNIT_ASSERT_EQUAL(deserializer.Loads(testData).GetStats().Status, EDeserializationStatus::OK);

        testData = TStringBuf("[[\"signal_dhhh\", [[[1,2],[2,3]]]]]");
        UNIT_ASSERT_EQUAL(deserializer.Loads(testData).GetStats().Status, EDeserializationStatus::EXTRA_DEPTH);
        testData = TStringBuf("[[\"signal1_ahhh\", [0.0, 1.0, 2.0]], [\"signal2_dhhh\", [[[1,2],[2,3]]]], [\"signal3_ahhh\", 5.0]]");
        UNIT_ASSERT_EQUAL(deserializer.Loads(testData).GetStats().Status, EDeserializationStatus::EXTRA_DEPTH);

        testData = TStringBuf("[[\"signal_dhhh\", [[1,2],[2,3,4]]]]");
        auto stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_EQUAL(stats.Status, EDeserializationStatus::TOO_MUCH_ARGUMENTS_IN_BUCKET);
        UNIT_ASSERT_EQUAL(stats.ParseErrors, 1);

        testData = TStringBuf("[[\"signal1_dhhh\", [[1,2],[2,3,8]]], [\"signal2_dhhh\", [[1,2],[2,3,8]]], [\"signal3_dhhh\", [[1,2],[2,3,8]]]]");
        stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_EQUAL(stats.Status, EDeserializationStatus::TOO_MUCH_ARGUMENTS_IN_BUCKET);
        UNIT_ASSERT_EQUAL(stats.ParseErrors, 3);
    }

    Y_UNIT_TEST(UgramBucketChangeStatTest) {
        TUnistatDeserializer deserializer("module", 100);
        TStringBuf testData;

        testData =
            "["
            "[\"tag=v1;s1_ahhh\", [[0, 1], [10, 1], [20, 2], [20, 1], [30, 0]]],"
            "[\"tag=v2;s1_ahhh\", [[0, 1], [10, 1], [20, 2], [30, 0]]],"
            "[\"tag=v2;s2_ahhh\", [[0, 1], [10, 1], [20, 2], [30, 0]]],"
            "[\"tag=v2;s3_dhhh\", [[0, 1], [10, 1], [20, 2], [30, 0]]]"
            "]"sv;
        auto stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_VALUES_EQUAL(stats.NumUgramsChangedBuckets, 0);
        UNIT_ASSERT_VALUES_EQUAL(stats.UgramSizes.front().Weight, 4);

        // check that ugram sizes stats ugram is correct
        TMaybe<double> prevBucketUpperBound;
        for (auto& ugramSizeBucket: stats.UgramSizes) {
            UNIT_ASSERT(ugramSizeBucket.LowerBound + 1 <= ugramSizeBucket.UpperBound);
            if (prevBucketUpperBound.Defined()) {
                UNIT_ASSERT_VALUES_EQUAL(ugramSizeBucket.LowerBound, *prevBucketUpperBound);
            }
            prevBucketUpperBound = ugramSizeBucket.UpperBound;
        }

        testData =
            "["
            "[\"tag=v1;s1_ahhh\", [[0, 1], [10, 1], [20, 2], [20, 1], [30, 0]]],"
            "[\"tag=v2;s1_ahhh\", [[0, 4], [10, 1], [20, 2], [33, 0]]],"
            "[\"tag=v2;s2_ahhh\", [[0, 2], [10, 1], [20, 2], [30, 0]]],"
            "[\"tag=v2;s3_dhhh\", [[0, 1], [10, 1], [20, 1], [35, 0]]]"
            "]"sv;
        stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_VALUES_EQUAL(stats.NumUgramsChangedBuckets, 2);
        UNIT_ASSERT_VALUES_EQUAL(stats.UgramSizes.front().Weight, 4);

        testData =
            "["
            "[\"tag=v1;s1_ahhh\", [[0, 1], [10, 1], [20, 2], [20, 1], [30, 0]]],"
            "[\"tag=v1;s3_dhhh\", [[0, 90], [10, 1], [222, 1], [333, 0]]],"
            "[\"tag=v2;s1_ahhh\", [[0, 4], [10, 1], [20, 234], [33, 0]]],"
            "[\"tag=v2;s2_ahhh\", [[0, 2], [10, 123], [20, 2], [30, 0]]],"
            "[\"tag=v2;s3_dhhh\", [[0, 90], [10, 1], [20, 1], [35, 0]]]"
            "]"sv;
        stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_VALUES_EQUAL(stats.NumUgramsChangedBuckets, 0);
        UNIT_ASSERT_VALUES_EQUAL(stats.UgramSizes.front().Weight, 5);

        testData =
            "["
            "[\"tag=v1;s1_ahhh\", [[0, 1], [10, 1], [20, 2], [20, 1], [30, 0]]],"
            "[\"tag=v1;s3_dhhh\", [[0, 90], [10, 1], [222, 1], [333, 0]]],"
            "[\"tag=v2;s1_ahhh\", [[0, 4], [10, 1], [20, 234], [33, 0]]],"
            "[\"tag=v2;s2_ahhh\", [[0, 2], [10, 123], [20, 2], [30, 0]]],"
            "[\"tag=v2;s3_dhhh\", [[0, 90], [10, 1], [20, 1], [35, 0]]]"
            "]"sv;
        stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_VALUES_EQUAL(stats.NumUgramsChangedBuckets, 0);
        UNIT_ASSERT_VALUES_EQUAL(stats.UgramSizes.front().Weight, 5);

        testData =
            "["
            "[\"tag=v1;s1_ahhh\", [[0, 1], [10, 1], [20, 2], [30, 0]]]"
            "]"sv;
        stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_VALUES_EQUAL(stats.NumUgramsChangedBuckets, 1);
        UNIT_ASSERT_VALUES_EQUAL(stats.UgramSizes.front().Weight, 1);

        testData =
            "["
            "[\"tag=v1;s1_ahhh\", [[0, 1], [10, 1], [20, 2], [30, 1]]]"
            "]"sv;
        stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_VALUES_EQUAL(stats.NumUgramsChangedBuckets, 1);
        UNIT_ASSERT_VALUES_EQUAL(stats.UgramSizes.front().Weight, 1);

        testData =
            "["
            "[\"tag=v1;s1_ahhh\", [[0, 1], [10, 1], [20, 2], [30, 2]]]"
            "]"sv;
        stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_VALUES_EQUAL(stats.NumUgramsChangedBuckets, 0);
        UNIT_ASSERT_VALUES_EQUAL(stats.UgramSizes.front().Weight, 1);

        testData =
            "["
            "[\"tag=v1;s1_ahhh\", [[0, 1], [10, 1], [20, 2], [30, 1], [30, 17], [40, 0]]]"
            "]"sv;
        stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_VALUES_EQUAL(stats.NumUgramsChangedBuckets, 1);
        UNIT_ASSERT_VALUES_EQUAL(stats.UgramSizes.front().Weight, 1);

        testData =
            "["
            "[\"tag=v1;s1_ahhh\", [[0, 1], [10, 1], [20, 2], [30, 1], [30, 178], [40, 0]]]"
            "]"sv;
        stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_VALUES_EQUAL(stats.NumUgramsChangedBuckets, 0);
        UNIT_ASSERT_VALUES_EQUAL(stats.UgramSizes.front().Weight, 1);

        testData =
            "[[\"tag=v1;s1_ahhh\", ["
            "[0, 1],"
            "[10, 1],"
            "[20, 2],"
            "[30, 1],"
            "[30, 178],"
            "[31, 1],"
            "[32, 1],"
            "[33, 1],"
            "[34, 1],"
            "[35, 1],"
            "[36, 1],"
            "[37, 1],"
            "[38, 1],"
            "[39, 1],"
            "[40, 1],"
            "[41, 1],"
            "[42, 1],"
            "[43, 1],"
            "[44, 1],"
            "[45, 0]"
            "]]]"sv;
        stats = deserializer.Loads(testData).GetStats();
        UNIT_ASSERT_VALUES_EQUAL(stats.NumUgramsChangedBuckets, 1);
        UNIT_ASSERT_VALUES_EQUAL(stats.UgramSizes.front().UpperBound, 20); // expect the first stats bucket to be [0, 20]
        auto bucketIt = stats.UgramSizes.begin();
        UNIT_ASSERT_VALUES_EQUAL(bucketIt->Weight, 0);
        ++bucketIt;
        UNIT_ASSERT_VALUES_EQUAL(bucketIt->Weight, 1);
    }
}
