# -*- coding: utf-8 -*-
import requests
import os
import re

ABC_URL = "https://abc-back.yandex-team.ru/api/v4"
ABC_TOKEN = os.environ["ABC_TOKEN"]
STARTRACK_TOKEN = os.environ["STARTRACK_TOKEN"]

abc_session = requests.session()
abc_session.headers["Authorization"] = "OAuth {}".format(ABC_TOKEN)

st_session = requests.session()
st_session.headers["Authorization"] = "OAuth {}".format(STARTRACK_TOKEN)


class Resource(object):
    def __init__(self, dc, cpu, memory, hdd, ssd, ipv4):
        self.dc = dc
        self.cpu = cpu
        self.memory = memory
        self.hdd = hdd
        self.ssd = ssd
        self.ipv4 = ipv4

    def __repr__(self):
        return "Resource dc={} cpu={} memory={} hdd={} ssd={} ipv4={}".\
            format(self.dc, self.cpu, self.memory, self.hdd, self.ssd, self.ipv4)


def get_attribute_by_name(resource, name, default_value):
    for attribute in resource["attributes"]:
        if attribute["name"] == name:
            return attribute["value"]

    return default_value


def build_resources(resources):
    resources_per_dc = {}

    for resource in resources:
        if get_attribute_by_name(resource, "segment", "") == "default":
            dc = get_attribute_by_name(resource, "location", None)
            assert(dc is not None)

            if dc not in resources_per_dc.keys():
                resources_per_dc[dc] = Resource(dc, 0, 0, 0, 0, 0)

            res = resources_per_dc[dc]
            res.cpu += int(get_attribute_by_name(resource, "cpu", 0))
            res.memory += int(get_attribute_by_name(resource, "memory", 0))
            res.hdd += int(get_attribute_by_name(resource, "hdd", 0))
            res.ssd += int(get_attribute_by_name(resource, "ssd", 0))
            res.ipv4 += int(get_attribute_by_name(resource, "ip4", 0))

    return resources_per_dc


def add_resources(total_sources, resources):
    for dc, resource in resources.iteritems():
        if dc not in total_sources.keys():
            total_sources[dc] = Resource(dc, 0, 0, 0, 0, 0)

        res = total_sources[dc]
        res.cpu += resource.cpu
        res.memory += resource.memory
        res.hdd += resource.hdd
        res.ssd += resource.ssd
        res.ipv4 += resource.ipv4


def get_resources_from_abc(slug):
    print "Querying resources {}".format(slug)
    res = abc_session.get(
        "{}/resources/consumers/?fields=resource,state&supplier=1979&service={}".format(ABC_URL, slug),
        timeout=20
    )
    res.raise_for_status()

    granted_resources = [
        result["resource"] for result in res.json()["results"]
        if result["state"] == "granted" and
        get_attribute_by_name(result["resource"], "segment", "") == "default"]

    requested_resources = [(result["resource"], result["id"]) for result in res.json()["results"]
                           if result["state"] == "approved" and
                           get_attribute_by_name(result["resource"], "segment", "") == "default"]

    #filter out non-default segment

    request_ids = [request_id for resource, request_id in requested_resources]
    print "Granted resources={} requested_resources={} request_ids={}".format(
        granted_resources, requested_resources, request_ids)

    granted_resources_per_dc = build_resources(granted_resources)
    requested_resources_per_dc = build_resources([resource for resource, request_id in requested_resources])

    print "Granted resources per dc={} requested_resources={}".format(
        granted_resources_per_dc, requested_resources_per_dc)

    if requested_resources_per_dc is None:
        return None, None

    resources_per_dc = {}
    add_resources(resources_per_dc, granted_resources_per_dc)
    add_resources(resources_per_dc, requested_resources_per_dc)

    print "Resources_per_dc per dc={}".format(resources_per_dc)

#в одном ДЦ в сумме всех запросов диска=0. Так обычно ошибаются когда диски не указывают совсем
    for dc, res in resources_per_dc.iteritems():
        if res.ssd == 0 and res.hdd == 0:
            return None, None

    return (None, None) if len(resources_per_dc) == 0 else (resources_per_dc, request_ids)


def load_tickets():
    print "Loading tickets"

    filters = {
        "filter":
            {
                "queue": "YPRES",
                "status": "open",
                "assignee": "empty()"
            }
    }

    results_count = 100
    page = 1

    tickets = []

    while results_count == 100:
        results = st_session.post("https://st-api.yandex-team.ru/v2/issues/"
                                  "_search?perPage={}&page={}&orderBy=keyd&orderAsc=false]".format(results_count, page),
                                  json=filters,
                                  timeout=20)
        results = results.json()
        results_count = len(results)
        tickets.extend(results)
        page += 1

    tickets = [ticket for ticket in tickets if ticket["summary"].startswith("Запрос квот для сервиса:".decode("utf-8"))]
    print "Loaded {} tickets".format(len(tickets))
    return tickets


def auto_approve_request(request):
    print "Check if request is suitable for auto approve={}".format(request)
    cpu = 0
    memory = 0
    hdd = 0
    ssd = 0
    ipv4 = 0

    for dc, req in request["resources"].iteritems():
        cpu += req.cpu
        memory += req.memory
        hdd += req.hdd
        ssd += req.ssd
        ipv4 += req.ipv4

    suitable_for_autoapprove = True if cpu <= 100 and memory <= 400 and ipv4 == 0 else False
    print "Suitable for autoapprove={}".format(suitable_for_autoapprove)

    return suitable_for_autoapprove


def add_ticket_comment(ticket, comment):
    print "Adding ticket {} comment".format(ticket)
    result = st_session.post("https://st-api.yandex-team.ru/v2/issues/{}/comments".format(ticket),
                             json={"text": comment}
                             )
    result.raise_for_status()


def close_ticket(ticket):
    print "Closing ticket".format(ticket)

    resp = st_session.get("https://st-api.yandex-team.ru/v2/issues/{}".format(ticket))
    resp.raise_for_status()

    if resp.json()["status"]["key"] != "closed":
        add_ticket_comment(ticket, "Квота выдана")

        resp = st_session.post("https://st-api.yandex-team.ru/v2/issues/{}/transitions/close/_execute".
                               format(ticket),
                               json={"resolution": "fixed"})
        resp.raise_for_status()


def approve_request(request):
    print "auto approve request {}, tickets {}".format(request, request["tickets"])

    for request_id in request["request_ids"]:
        print "Marking abc request {} as done".format(request_id)
        res = abc_session.patch("{}/resources/consumers/{}/".format(ABC_URL, request_id), json={"state": "granted"})
        res.raise_for_status()

    for ticket in request["tickets"]:
        close_ticket(ticket)


def ticket_from_trusted_author(author):
    return author in ["slonnn", "sereglond"]


if __name__ == '__main__':
    regexp = re.compile(".*https://abc.yandex-team.ru/services/([a-zA-Z0-9_.-]+)"
                        "/resources/\?state=requested&view=consuming&supplier=1979&type=96.*")

    tickets = load_tickets()
    # service_resource_requests = {}

    # slugs = {}
    #
    # for ticket in tickets:
    #     description = ticket["description"]
    #     reg_match = regexp.search(description)
    #     if reg_match is not None:
    #         slug = reg_match.group(1)
    #
    #         if slug not in slugs.keys():
    #             slugs[slug] = list()
    #
    #         slugs[slug].append(ticket["key"])
    #
    # for slug in slugs.keys():
    #     resources, request_ids = get_resources_from_abc(slug)
    #
    #     if resources is not None:
    #         if len(request_ids) > 0:
    #             service_resource_requests[slug] = {
    #                     "resources": resources,
    #                     "tickets": slugs[slug],
    #                     "request_ids": request_ids
    #                 }
    #
    # for slug, request in service_resource_requests.iteritems():
    #     if auto_approve_request(request):
    #         approve_request(request)

    for ticket in tickets:
        description = ticket["description"]
        reg_match = regexp.search(description)
        if reg_match is not None:
            author = ticket["createdBy"]["id"]
            if ticket_from_trusted_author(author):
                print "Ticket {} from trusted author {}, approving".format(ticket["key"], author)
                close_ticket(ticket["key"])
