#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import print_function

import six

import requests
import collections
import copy
import math

from infra.yp.lib.retrier import retry_me

requests.packages.urllib3.disable_warnings()


DC = ["vla", "man", "sas", "myt"]
GENCFG_URL = 'https://api.gencfg.yandex-team.ru'
NANNY_URL = 'http://nanny.yandex-team.ru'
IO_SSD_PER_CPU = 5  # https://st.yandex-team.ru/RX-1460


def load_nanny_info(service, session):
    response = retry_me(lambda:
                        session.get('{}/v2/services/{}'.format(NANNY_URL, service),
                                    timeout=10),
                        retry_count=10)
    response.raise_for_status()
    service_info = response.json()
    return NannyService(service_info)


def get_resources_for_podset(dc, nanny_service, yp_client):
    pod_sets = yp_client[dc].select_objects(
        'pod_set',
        filter="[/labels/nanny_service_id]=\"{}\"".format(nanny_service.id()),
        selectors=["/meta/id"])

    if pod_sets is not None and len(pod_sets) > 0:
        podset = pod_sets[0]
        podset_id = podset[0]
        pods = yp_client[dc].select_objects(
            'pod',
            filter="[/meta/pod_set_id]=\"{}\"".format(podset_id),
            selectors=[
                "/spec/resource_requests",
                "/spec/disk_volume_requests",
                "/meta/id"])

        if len(pods) > 0:
            account = Account.empty()

            for pod in pods:
                pod_id = pod[2]
                resource_requests = pod[0]
                memory = resource_requests["memory_limit"]
                cpu = resource_requests["vcpu_guarantee"]

                hdd = 0
                ssd = 0
                disk_volume_requests = pod[1]
                for disk_request in disk_volume_requests:
                    quota_policy = disk_request["quota_policy"]
                    if "capacity" in quota_policy:
                        size = quota_policy["capacity"]
                        if disk_request["storage_class"] == "hdd":
                            hdd += size
                        elif disk_request["storage_class"] == "ssd":
                            ssd += size
                        else:
                            print("Unknown disk type={}".format(disk_request["storage_class"]))
                            assert(False and "Unknown disk type")

                ipv4_addresses = yp_client[dc].select_objects(
                    'internet_address',
                    filter="[/status/pod_id]=\"{}\"".format(pod_id),
                    selectors=[
                        "/meta/id"])
                ipv4count = len(ipv4_addresses)

                account.add(Account(cpu, memory, hdd, ssd, ipv4count, io_ssd=0, io_hdd=0, net_bandwidth=0))

            return account

    return Account.empty()


def find_gencfg_groups_by_nanny_service(nanny_service):
    instance_types = nanny_service.runtime_attrs()["content"]["instances"]["chosen_type"]

    if "EXTENDED_GENCFG_GROUPS" != instance_types:
        raise ValueError("Unknown instance list {}".format(instance_types))

    groups_info = [
        (group["name"], group["release"]) for group in
        nanny_service.runtime_attrs()["content"]["instances"]["extended_gencfg_groups"]["groups"]]

    return groups_info


class NannyService(object):
    def __init__(self, spec):
        self.spec = spec

    def id(self):
        return self.spec["_id"]

    def runtime_attrs(self):
        return self.spec["runtime_attrs"]


def is_nanny_service_in_yp(nanny_service):
    return nanny_service.runtime_attrs()["content"]["engines"]["engine_type"] == "YP_LITE"


def calculate_account_for_nanny_service(nanny_service, yp_clients):
    deploy_engine = nanny_service.runtime_attrs()["content"]["engines"]["engine_type"]

    if deploy_engine == "YP_LITE":
        accounts = {}
        dcs = DC
        for dc in dcs:
            accounts[dc] = get_resources_for_podset(dc, nanny_service, yp_clients)
        return accounts

    elif deploy_engine.find("ISS") == 0:
        accounts = collections.defaultdict(Account.empty)
        groups = find_gencfg_groups_by_nanny_service(nanny_service)
        for group, tag in groups:
            resources = get_resources_for_groups_by_instances([group], tag)[0]
            for dc in DC:
                if dc in resources:
                    accounts[dc].add(resources[dc])
        return dict(accounts)

    else:
        raise ValueError("Unknown deploy engine {}".format(deploy_engine))


def get_not_yp_nanny_services(nanny_services, nanny_session):
    yp_nanny_services = []
    not_yp_nanny_services = []
    for nanny_service in nanny_services:
        service_info = load_nanny_info(nanny_service, nanny_session)
        if is_nanny_service_in_yp(service_info):
            yp_nanny_services.append(service_info)
        else:
            not_yp_nanny_services.append(service_info)
    return yp_nanny_services, not_yp_nanny_services


class Account(object):
    def __init__(self, cpu, memory, hdd, ssd, ipv4, io_ssd, io_hdd, net_bandwidth,
                 gpu_models=collections.defaultdict(int)):
        self.cpu = cpu
        self.memory = memory
        self.hdd = hdd
        self.ssd = ssd
        self.ipv4 = ipv4
        self.io_ssd = io_ssd
        self.io_hdd = io_hdd
        self.net_bandwidth = net_bandwidth
        self.gpu_models = gpu_models

    @staticmethod
    def empty():
        return Account(0, 0, 0, 0, 0, 0, 0, 0, collections.defaultdict(int))

    def add(self, account):
        self.cpu += account.cpu
        self.memory += account.memory
        self.hdd += account.hdd
        self.ssd += account.ssd
        self.ipv4 += account.ipv4
        self.io_ssd += account.io_ssd
        self.io_hdd += account.io_hdd
        self.net_bandwidth += account.net_bandwidth
        for gpu_model, gpu_count in six.iteritems(account.gpu_models):
            self.gpu_models[gpu_model] += gpu_count

    def __repr__(self):
        out = "Account cpu={} memory={} hdd={} ssd={} ipv4={}, io_ssd={}, io_hdd={}, net_bandwidth={}, gpu_models={}"
        return out.format(self.cpu, self.memory, self.hdd, self.ssd, self.ipv4, self.io_ssd, self.io_hdd,
                          self.net_bandwidth, self.gpu_models)

    def is_empty(self):
        return self.cpu == 0 or self.memory == 0


def estimate_nanny(nanny_services, nanny_session, yp_clients):
    accounts_total_per_dc = {}

    for nanny_service in nanny_services:
        service_info = load_nanny_info(nanny_service, nanny_session)
        accounts_per_dc = calculate_account_for_nanny_service(service_info, yp_clients)
        for dc, account in six.iteritems(accounts_per_dc):
            if dc not in accounts_total_per_dc.keys():
                accounts_total_per_dc[dc] = account
            else:
                accounts_total_per_dc[dc].add(account)

    return accounts_total_per_dc


def combine_gencfg_and_nanny_groups(gencfg_groups, nanny_services, nanny_session, yp_clients):
    yp_nanny_services, not_yp_nanny_services = get_not_yp_nanny_services(nanny_services, nanny_session)

    nanny_groups = []
    yp_nanny_groups = []

    def fill_groups_by_services(services, groups):
        for service in services:
            groups_by_services = find_gencfg_groups_by_nanny_service(service)
            for group in groups_by_services:
                groups.append(group)

    fill_groups_by_services(not_yp_nanny_services, nanny_groups)
    fill_groups_by_services(yp_nanny_services, yp_nanny_groups)

    uniq_yp_gencfg_groups = set([item[0] for item in yp_nanny_groups])

    gencfg_groups = [(group, "trunk") for group in gencfg_groups]
    gencfg_groups.extend(nanny_groups)

    uniq_groups = set()
    gencfg_groups_with_unique_tags = []
    for item in gencfg_groups:
        if not item[0] in uniq_groups and not item[0] in uniq_yp_gencfg_groups:
            uniq_groups.add(item[0])
            gencfg_groups_with_unique_tags.append(item)

    if yp_nanny_services:
        print("YP nanny services: {}".format(yp_nanny_services))

    return set(gencfg_groups_with_unique_tags), uniq_yp_gencfg_groups


def add_service_account_to_total_account(accounts_per_dc, accounts_total_per_dc):
    for dc, account in six.iteritems(accounts_per_dc):
        if dc not in accounts_total_per_dc.keys():
            accounts_total_per_dc[dc] = Account.empty()
        accounts_total_per_dc[dc].add(account)
    return accounts_total_per_dc


def get_resources_for_instance(instance, use_weakened_resources=False):
    hdd = 0
    ssd = 0
    volumes = instance["volumes"]
    for volume in volumes:
        mount_point, size = volume["dom0_mount_point_root"], volume["quota"]
        if mount_point.startswith("/place") or mount_point == "/cores":
            hdd += size
        elif mount_point == "/ssd":
            ssd += size
        else:
            raise ValueError("Unknown mount point={}".format(mount_point))

    storage_hdd = 0
    storage_ssd = 0
    storages = instance["storages"]

    for type in ["rootfs", "ssd"]:
        if type in storages:
            partition, size = storages[type]["partition"], storages[type]["size"]
            if partition == "hdd":
                storage_hdd += size
            elif partition == "ssd":
                storage_ssd += size

    storage_hdd *= 1024 * 1024 * 1024
    storage_ssd *= 1024 * 1024 * 1024
    hdd = max(hdd, storage_hdd)
    ssd = max(ssd, storage_ssd)

    cpu = instance["power"] / 40 * 1000
    io_hdd = 15 * 1000 if cpu > 0 else 0  # 15 MB/s, multiply by 1000 because cpu is in millicores
    net_bandwidth = 10 * 1000 if cpu > 0 else 0  # 10 MB/s, multiply by 1000 because cpu is in millicores
    memory = instance["porto_limits"]["memory_guarantee"]
    MAGIC_CONST = 50 * 1024 * 1024 * 1024  # 50Gb (see https://st.yandex-team.ru/RX-944)
    if use_weakened_resources:
        hdd = max(hdd, MAGIC_CONST)
        ssd = max(ssd, MAGIC_CONST)
    if ssd > 0 and cpu > 0:
        io_ssd = 30 * 1000

    return Account(cpu, memory, hdd, ssd, ipv4=0, io_ssd=io_ssd, io_hdd=io_hdd, net_bandwidth=net_bandwidth)


def get_resources_by_instances(dc, group, tag):
    dc = dc.lower()
    r = retry_me(lambda:
                 requests.get("{}/{}/searcherlookup/groups/{}/instances".format(GENCFG_URL, tag, group),
                              timeout=30,
                              verify=False),
                 retry_count=10)
    r.raise_for_status()
    instances_info = r.json()
    instances = instances_info["instances"]
    current_account = Account.empty()
    instance_count = 0

    for instance in instances:
        if instance["dc"].lower() == dc.lower():
            account = get_resources_for_instance(instance, use_weakened_resources=True)
            current_account = Account(max(account.cpu, current_account.cpu),
                                      max(account.memory, current_account.memory),
                                      max(account.hdd, current_account.hdd),
                                      max(account.ssd, current_account.ssd),
                                      max(account.ipv4, current_account.ipv4),
                                      max(account.io_ssd, current_account.io_ssd),
                                      max(account.io_hdd, current_account.io_hdd),
                                      max(account.net_bandwidth, current_account.net_bandwidth),
                                      )
            instance_count += 1

    if current_account.is_empty():
        return current_account, 0

    r = retry_me(lambda:
                 requests.get("{}/{}/groups/{}/card".format(GENCFG_URL, tag, group),
                              timeout=30,
                              verify=False),
                 retry_count=10)
    r.raise_for_status()
    card = r.json()
    current_account.hdd = max(current_account.hdd, card["reqs"]["instances"]["disk"])
    return current_account, instance_count


def get_resources_for_groups_by_instances(groups, tag, use_weakened_resources=False):
    accounts_total_per_dc = collections.defaultdict(Account.empty)
    groups_by_dc = collections.defaultdict(list)

    for group in groups:
        r = retry_me(lambda:
                     requests.get("{}/{}/searcherlookup/groups/{}/instances".format(GENCFG_URL, tag, group),
                                  timeout=30,
                                  verify=False),
                     retry_count=10)
        r.raise_for_status()
        instances_info = r.json()
        instances = instances_info["instances"]
        unique_dc = set()
        current_account = Account.empty()

        dc = ""
        for instance in instances:
            dc = instance["dc"]
            account = get_resources_for_instance(instance, use_weakened_resources)
            current_account.add(account)
            unique_dc.add(dc)

        r = retry_me(lambda:
                     requests.get("{}/{}/groups/{}/card".format(GENCFG_URL, tag, group),
                                  timeout=30,
                                  verify=False),
                     retry_count=10)
        r.raise_for_status()
        card = r.json()
        current_account.hdd = max(current_account.hdd, card["reqs"]["instances"]["disk"] * len(instances))

        for dc in unique_dc:
            accounts_total_per_dc[dc].add(current_account)
            groups_by_dc[dc].append(group)

    return dict(accounts_total_per_dc), groups_by_dc


def get_resources_for_groups_with_tag_by_instances(groups_with_tag, use_weakened_resources=False):
    grouping = collections.defaultdict(list)
    for v, k in groups_with_tag:
        grouping[k].append(v)
    accounts_total_per_dc = {}
    groups_by_dc = collections.defaultdict(list)
    for tag, groups in six.iteritems(grouping):
        resources, groups = get_resources_for_groups_by_instances(groups, tag, use_weakened_resources)
        add_service_account_to_total_account(resources, accounts_total_per_dc)
        for key, value in six.iteritems(groups):
            groups_by_dc[key].extend(value)
    return accounts_total_per_dc, groups_by_dc


def estimate_gencfg(gencfg_groups):
    return get_resources_for_groups_with_tag_by_instances(gencfg_groups, True)[0]


def bytes_to_gigabytes(bytes_n):
    return bytes_n / 1024.0 ** 3


def bytes_to_terabytes(bytes_n):
    return bytes_to_gigabytes(bytes_n) / 1024.0


def ceil_with_precision(data, precision):
    return math.ceil(data * 10 ** precision) / 10 ** precision


def customize_quota(value):
    quota = copy.deepcopy(value)
    quota['cpu'] = max(quota['cpu'], 100.0)
    quota['memory'] = ceil_with_precision(max(1, bytes_to_gigabytes(quota["memory"])), 2)
    quota['ssd'] = ceil_with_precision(bytes_to_terabytes(quota["ssd"]), 2)
    quota['hdd'] = ceil_with_precision(bytes_to_terabytes(quota["hdd"]), 2)
    return quota


def amputate_cpu_by_dc(dc, cpu):
    infra_cores = (56 if dc.upper() == 'VLA' else 32)
    infra_cores_coef = float(infra_cores - 2) / infra_cores
    return cpu * infra_cores_coef


def amputate_resources_for_infrastructure(accounts):
    result = accounts
    for dc, account in six.iteritems(accounts):
        result[dc].cpu = amputate_cpu_by_dc(dc, result[dc].cpu)
    return result


def estimate_account(nanny_services, gencfg_groups, nanny_session, yp_client):

    combined_gencfg_groups = combine_gencfg_and_nanny_groups(gencfg_groups, nanny_services, nanny_session, yp_client)[0]
    accounts_total_per_dc = estimate_gencfg(combined_gencfg_groups)

    accounts_total_per_dc = amputate_resources_for_infrastructure(accounts_total_per_dc)

    result = {}
    for key, value in six.iteritems(accounts_total_per_dc):
        customized_quota = customize_quota(value.__dict__)
        result[key] = Account(
            customized_quota['cpu'],
            customized_quota['memory'],
            customized_quota['hdd'],
            customized_quota['ssd'],
            customized_quota['ipv4'],
            io_ssd=0,
            io_hdd=0,
            net_bandwidth=0,
        )

    return result
