from solomon import BasePushApiReporter, OAuthProvider

from contextlib import contextmanager
import os


class Sensor(object):
    def __init__(self):
        self._sum = 0
        self._count = 0

    def collect(self, value):
        self._sum += value
        self._count += 1

    def is_initialized(self):
        return self._count > 0

    def get_value(self):
        return 1.0 * self._sum / self._count


class SolomonClient(object):
    def __init__(self, cluster):
        self._underlying = BasePushApiReporter(
            project="yp",
            cluster=cluster,
            url="http://solomon.yandex.net",
            service="database",
            auth_provider=OAuthProvider(os.environ["SOLOMON_TOKEN"]),
        )
        self._sensors = dict(
            create_snapshot_duration=Sensor(),
            history_events_table_size=Sensor()
        )

    def collect_create_snapshot_duration(self, duration):
        self._sensors["create_snapshot_duration"].collect(duration)

    def collect_history_events_table_size(self, size):
        self._sensors["history_events_table_size"].collect(size)

    def report(self):
        for name, sensor in self._sensors.items():
            if not sensor.is_initialized():
                continue
            self._underlying.set_value(
                ["backup." + name],
                [sensor.get_value()],
                [dict()],
            )


@contextmanager
def create_solomon_client(cluster):
    solomon_client = SolomonClient(cluster)
    try:
        yield solomon_client
    finally:
        solomon_client.report()
